<?php
/**
 * Group Query Class
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCP\Database\Queries;

// Exit if accessed directly
defined( 'ABSPATH' ) || exit;

use RCP\Database\Query;

/**
 * Class used for querying groups
 *
 * @see     \RCP\Database\Queries\Groups::__construct() for accepted arguments.
 * @package RCP\Database\Queries
 * @since   2.0
 */
class Groups extends Query {

	/** Table Properties ******************************************************/

	/**
	 * Name of the database table to query.
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $table_name = 'groups';

	/**
	 * String used to alias the database table in MySQL statement.
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $table_alias = 'gag';

	/**
	 * Name of class used to setup the database schema
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $table_schema = '\\RCP\\Database\\Schemas\\Groups';

	/** Item ******************************************************************/

	/**
	 * Name for a single item
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $item_name = 'group';

	/**
	 * Plural version for a group of items.
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $item_name_plural = 'groups';

	/**
	 * Callback function for turning IDs into objects
	 *
	 * @since  2.0
	 * @access public
	 * @var mixed
	 */
	protected $item_shape = '\\RCPGA_Group';

	/**
	 * Group to cache queries and queried items in.
	 *
	 * @since  2.0
	 * @access public
	 * @var string
	 */
	protected $cache_group = 'groups';

	/**
	 * Sets up the group query, based on the query vars passed.
	 *
	 * @param string|array $query                   {
	 *                                              Optional. Array or query string of membership query parameters.
	 *                                              Default empty.
	 *
	 * @type int           $group_id                A group ID to only return that group. Default empty.
	 * @type array         $group_id__in            Array of group IDs to include. Default empty.
	 * @type array         $group_id__not_in        Array of group IDs to exclude. Default empty.
	 * @type int           $owner_id                An owner user ID to filter by. Default empty.
	 * @type array         $owner_id_in             Array of owner user IDs to include. Default empty.
	 * @type array         $owner_id__not_in        Array of owner user IDs to exclude. Default empty.
	 * @type string        $name                    A specific group name to search for.
	 * @type int           $seats                   Query for groups with this many seats. Default empty.
	 * @type array         $date_query              Query all datetime columns together. See WP_Date_Query.
	 * @type array         $date_created_query      Date query clauses to limit groups by. See WP_Date_Query.
	 *                                              Default null.
	 * @type bool          $count                   Whether to return a group count (true) or array of group
	 *                                              objects. Default false.
	 * @type string        $fields                  Item fields to return. Accepts any column known names
	 *                                              or empty (returns an array of complete group objects). Default
	 *                                              empty.
	 * @type int           $number                  Limit number of groups to retrieve. Default 100.
	 * @type int           $offset                  Number of groups to offset the query. Used to build LIMIT
	 *                                              clause. Default 0.
	 * @type bool          $no_found_rows           Whether to disable the `SQL_CALC_FOUND_ROWS` query. Default true.
	 * @type string|array  $orderby                 Accepts 'group_id', 'owner_id', 'name', 'seats', 'member_count'
	 *                                              and 'date_created'. Also accepts false, an empty array, or
	 *                                              'none' to disable `ORDER BY` clause. Default 'group_id'.
	 * @type string        $order                   How to order results. Accepts 'ASC', 'DESC'. Default 'DESC'.
	 * @type string        $search                  Search term(s) to retrieve matching groups for. Default empty.
	 * @type bool          $update_cache            Whether to prime the cache for found groups. Default false.
	 * }
	 *
	 * @access public
	 * @since  2.0
	 */
	public function __construct( $query = array() ) {
		parent::__construct( $query );

		add_filter( 'query', array( $this, 'fix_group_meta_col_name' ) );
	}

	/**
	 * Replace `rcp_group_group_id` column name with `rcp_group_id` in `delete_all_item_meta()` queries.
	 *
	 * @link  https://github.com/restrictcontentpro/rcp-group-accounts/issues/333
	 *
	 * @param string $query
	 *
	 * @since 2.1
	 * @return string
	 */
	public function fix_group_meta_col_name( $query ) {

		$table = rcpga_group_accounts()->table_groupmeta->get_table_name();

		// Replace `rcp_group_group_id` column with `rcp_group_id`.
		$find    = $this->apply_prefix( "{$this->item_name}_group_id" );
		$replace = $this->apply_prefix( "{$this->item_name}_id" );

		// Bail if the query doesn't look like the one in delete_all_item_meta()
		if ( false === strpos( $query, "SELECT meta_id FROM {$table} WHERE {$find} = " ) ) {
			return $query;
		}

		return str_replace( $find, $replace, $query );

	}

}