<?php
/**
 * Cron Functions
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}


/**
 * Set up the following cron job events:
 *
 * Daily: Remove accounts that haven't accepted group invitation
 * @see   rcpga_purge_unclaimed_invitations()
 *
 * @since 1.4
 * @return void
 */
function rcpga_setup_cron_jobs() {

	if ( ! wp_next_scheduled( 'rcpga_purge_unclaimed_invitations' ) ) {
		wp_schedule_event( current_time( 'timestamp' ), 'daily', 'rcpga_purge_unclaimed_invitations' );
	}

}

add_action( 'wp', 'rcpga_setup_cron_jobs' );

/**
 * Check for unclaimed invitations that are older than the defined period.
 * These invitations are deleted and the users are removed from the group.
 *
 * @since 1.4
 * @return void
 */
function rcpga_purge_unclaimed_invitations() {

	rcp_log( 'Group Accounts: starting rcpga_purge_unclaimed_invitations() cron job.', true );

	global $rcp_options;

	$period = ! empty( $rcp_options['groups_delete_pending_invites'] ) ? $rcp_options['groups_delete_pending_invites'] : 0;

	if ( empty( $period ) || '0' == $period ) {
		rcp_log( 'Group Accounts: deleting pending invites is disabled - exiting rcpga_purge_unclaimed_invitations() cron job.', true );

		return;
	}

	$date    = date( 'Y-m-d', strtotime( $period, current_time( 'timestamp' ) ) );
	$members = rcpga_get_group_members( array(
		'number'           => 99999,
		'role'             => 'invited',
		'date_added_query' => array(
			array(
				'after'  => '0000-00-00 00:00:00',
				'before' => date( 'Y-m-d 23:59:59', strtotime( $date ) )
			)
		)
	) );

	if ( ! empty( $members ) ) {
		foreach ( $members as $member ) {
			/**
			 * @var RCPGA_Group_Member $member
			 */
			rcp_log( sprintf( 'Group Accounts: removing invited user #%d from group #%d.', $member->get_user_id(), $member->get_group_id() ) );
			$member->remove();
		}
	}

	rcp_log( sprintf( 'Group Accounts: found and removed %d expired invited members.', count( $members ) ) );

}

add_action( 'rcpga_purge_unclaimed_invitations', 'rcpga_purge_unclaimed_invitations' );