<?php
/**
 * Main RCP Group Accounts class file
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

// Exit if accessed directly.
defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'RCP_Group_Accounts' ) ) {

	final class RCP_Group_Accounts {

		/**
		 * @var RCP_Group_Accounts The one true RCP_Group_Accounts
		 *
		 * @since 2.0
		 */
		private static $instance;

		/**
		 * RCP Group Accounts loader file.
		 *
		 * @since 2.0
		 * @var string
		 */
		private $file = '';

		/**
		 * Groups database table
		 *
		 * @since 2.0
		 * @var \RCP\Database\Tables\Groups
		 */
		public $table_groups;

		/**
		 * Group meta database table
		 *
		 * @since 2.1
		 * @var \RCP\Database\Tables\Group_Meta
		 */
		public $table_groupmeta;

		/**
		 * Group members database table
		 *
		 * @since 2.0
		 * @var \RCP\Database\Tables\Group_Members
		 */
		public $table_group_members;

		/**
		 * Deprecated groups DB object.
		 *
		 * @deprecated 2.0
		 *
		 * @since      1.0
		 * @var RCPGA_Groups
		 */
		public $groups;

		/**
		 * Deprecated group members DB object.
		 *
		 * @deprecated 2.0
		 *
		 * @since      1.0
		 * @var RCPGA_Group_Members
		 */
		public $members;

		/**
		 * Main RCP_Group_Accounts Instance.
		 *
		 * Insures that only one instance of RCP_Group_Accounts exists in memory at any one
		 * time. Also prevents needing to define globals all over the place.
		 *
		 * @static
		 * @staticvar array $instance
		 *
		 * @uses      RCP_Group_Accounts::setup_constants() Setup constants.
		 * @uses      RCP_Group_Accounts::setup_files() Setup required files.
		 * @see       rcpga_group_accounts()
		 *
		 * @param string $file Main plugin file path.
		 *
		 * @since     2.0
		 * @return RCP_Group_Accounts The one true RCP_Group_Accounts
		 */
		public static function instance( $file = '' ) {

			global $rcp_options;

			// Return if already instantiated
			if ( self::is_instantiated() ) {
				return self::$instance;
			}

			// Setup the singleton
			self::setup_instance( $file );

			// Bootstrap
			self::$instance->setup_constants();
			self::$instance->setup_files();
			self::$instance->setup_application();
			self::$instance->setup_updater();

			if ( empty( $rcp_options['group_invite_subject'] ) ) {
				$rcp_options = wp_parse_args( self::$instance->create_default_email_templates(), $rcp_options );

				update_option( 'rcp_settings', $rcp_options );
			}

			// Return the instance
			return self::$instance;

		}

		/**
		 * Throw error on object clone.
		 *
		 * The whole idea of the singleton design pattern is that there is a single
		 * object therefore, we don't want the object to be cloned.
		 *
		 * @since  2.0
		 * @return void
		 */
		public function __clone() {
			// Cloning instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'rcp-group-accounts' ), '2.0' );
		}

		/**
		 * Disable un-serializing of the class.
		 *
		 * @since  2.0
		 * @return void
		 */
		public function __wakeup() {
			// Unserializing instances of the class is forbidden.
			_doing_it_wrong( __FUNCTION__, __( 'Cheatin&#8217; huh?', 'rcp-group-accounts' ), '2.0' );
		}

		/**
		 * Return whether the main loading class has been instantiated or not.
		 *
		 * @access private
		 * @since  2.0
		 * @return boolean True if instantiated. False if not.
		 */
		private static function is_instantiated() {

			// Return true if instance is correct class
			if ( ! empty( self::$instance ) && ( self::$instance instanceof RCP_Group_Accounts ) ) {
				return true;
			}

			// Return false if not instantiated correctly
			return false;

		}

		/**
		 * Setup the singleton instance
		 *
		 * @param string $file Path to main plugin file.
		 *
		 * @access private
		 * @since  2.0
		 */
		private static function setup_instance( $file = '' ) {
			self::$instance       = new RCP_Group_Accounts();
			self::$instance->file = $file;
		}

		/**
		 * Setup plugin constants.
		 *
		 * @access private
		 * @since  2.0
		 * @return void
		 */
		private function setup_constants() {

			// Plugin version
			if ( ! defined( 'RCPGA_GROUPS_VERSION' ) ) {
				define( 'RCPGA_GROUPS_VERSION', '2.2.1' );
			}

			// Plugin Root File
			if ( ! defined( 'RCPGA_GROUPS_PLUGIN_FILE' ) ) {
				define( 'RCPGA_GROUPS_PLUGIN_FILE', $this->file );
			}

			// Plugin Folder Path
			if ( ! defined( 'RCPGA_GROUPS_PLUGIN_DIR' ) ) {
				define( 'RCPGA_GROUPS_PLUGIN_DIR', plugin_dir_path( RCPGA_GROUPS_PLUGIN_FILE ) );
			}

			// Plugin Folder URL
			if ( ! defined( 'RCPGA_GROUPS_PLUGIN_URL' ) ) {
				define( 'RCPGA_GROUPS_PLUGIN_URL', plugin_dir_url( RCPGA_GROUPS_PLUGIN_FILE ) );
			}

		}

		/**
		 * Include required files.
		 *
		 * @access private
		 * @since  2.0
		 * @return void
		 */
		private function setup_files() {
			$this->include_files();

			// Admin
			if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
				$this->include_admin();
			} else {
				$this->include_frontend();
			}
		}

		/**
		 * Setup the rest of the application
		 *
		 * @since 2.0
		 */
		private function setup_application() {

			self::$instance->table_groups        = new \RCP\Database\Tables\Groups();
			self::$instance->table_groupmeta     = new \RCP\Database\Tables\Group_Meta();
			self::$instance->table_group_members = new \RCP\Database\Tables\Group_Members();
			new \RCPGA\Notices();

			self::$instance->groups  = RCPGA_Groups::get_instance();
			self::$instance->members = RCPGA_Group_Members::get_instance();

		}

		/**
		 * Setup automatic updates
		 *
		 * @since 2.0
		 */
		private function setup_updater() {
			if ( is_admin() && class_exists( 'RCP_Add_On_Updater' ) ) {
				new RCP_Add_On_Updater( 388, RCPGA_GROUPS_PLUGIN_FILE, RCPGA_GROUPS_VERSION );
			}
		}

		/** Includes **************************************************************/

		/**
		 * Include global files
		 *
		 * @access public
		 * @since  3.0
		 */
		private function include_files() {

			// Tables
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/groups/class-groups-table.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/groups/class-group-meta-table.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/members/class-group-members-table.php';

			// Queries
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/groups/class-group-query.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/members/class-group-member-query.php';

			// Schemas
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/groups/class-groups-schema.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/database/members/class-group-members-schema.php';

			// Other includes
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/class-notices.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/cron-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/classes/class-group.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/classes/class-group-member.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/group-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/group-filters.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/group-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/group-meta.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/member-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/member-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/groups/role-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/invite-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/member-registration-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/membership-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/membership-level-functions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/post-access.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/registration-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/shortcodes/class-group-dashboard.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/shortcodes/shortcodes.php';

			// Export - requires RCP version 3.4+
			if ( function_exists( 'rcp_get_csv_exporters' ) ) {
				require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/export/export-functions.php';
			}

			// Deprecated
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/deprecated/class-db-groups.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/deprecated/class-db-group-members.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/deprecated/filters.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/deprecated/functions.php';

		}

		/**
		 * Setup administration
		 *
		 * @since 2.0
		 */
		private function include_admin() {

			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/admin-notices.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/admin-pages.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/class-upgrades.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/screen-options.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/customers/customer-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/groups/groups-page.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/groups/group-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/memberships/membership-actions.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/settings/membership-levels.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/admin/settings/settings.php';

		}

		/**
		 * Setup front-end specific code
		 *
		 * @since 2.0
		 */
		private function include_frontend() {

			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/scripts.php';
			require_once RCPGA_GROUPS_PLUGIN_DIR . 'includes/template-filters.php';

		}

		/**
		 * Setup Email Templates for RCP
		 *
		 * @since 2.2
		 */
		private function create_default_email_templates() {

			$templates = array();

			// Group Invites
			$templates['group_invite_subject'] = sprintf( __( 'Group Invite to join %s', 'rcp-group-accounts' ), '%groupname%' );
			$group_invite_email                = sprintf( __( 'Hi %s', 'rcp-group-accounts' ), '%displayname%' ) . ",\n\n";
			$group_invite_email               .= sprintf( __( '%s has invited you to join %s', 'rcp-group-accounts' ), '%invitername%', '%groupname%' ) . ".\n\n";
			$group_invite_email               .= sprintf( __( 'Group Description: %s', 'rcp-group-accounts' ), '%groupdesc%' ) . ".\n\n";
			$group_invite_email               .= sprintf( __( 'Please click this link to join %s', 'rcp-group-accounts'), '%groupname%' ) . ".\n\n";
			$group_invite_email               .= '%invitelink%';
			$templates['group_invite_email']   = $group_invite_email;

			return $templates;

		}

	}

} // End if class_exists check.

/**
 * Returns the instance of RCP_Group_Accounts.
 *
 * The main function responsible for returning the one true RCP_Group_Accounts
 * instance to functions everywhere.
 *
 * Use this function like you would a global variable, except without needing
 * to declare the global.
 *
 * Example: <?php $rcp = rcpga_group_accounts(); ?>
 *
 * @since 2.0
 * @return RCP_Group_Accounts The one true RCP_Group_Accounts instance.
 */
function rcpga_group_accounts() {
	return RCP_Group_Accounts::instance();
}
