<?php
/**
 * class-notices.php
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA;


class Notices {

	/**
	 * Notice message
	 *
	 * @var string
	 */
	var $message = '';

	/**
	 * Type of notice (`success` or `error`)
	 *
	 * @var string
	 */
	var $type = 'success';

	/**
	 * Notices constructor.
	 *
	 * @since 2.0
	 */
	function __construct() {
		add_action( 'init', array( $this, 'add_notices' ) );
	}

	/**
	 * Add notices to admin and front-end
	 *
	 * @since 2.0
	 * @return void
	 */
	public function add_notices() {

		if ( ! isset( $_GET['rcpga-message'] ) ) {
			return;
		}

		$message = '';
		$type    = 'success';
		$notice  = urldecode( $_GET['rcpga-message'] );

		switch ( $notice ) {

			case 'group-member-added' :
				$message = __( 'Group member added successfully', 'rcp-group-accounts' );
				break;

			case 'group-member-removed' :
				$message = __( 'Group member removed successfully', 'rcp-group-accounts' );
				break;

			case 'group-member-updated' :
				$message = __( 'Group member updated successfully', 'rcp-group-accounts' );
				break;

			case 'group-members-imported' :
				$group_id = ! empty( $_REQUEST['rcpga-group'] ) ? absint( $_REQUEST['rcpga-group'] ) : 0;
				if ( empty( $group_id ) ) {
					break;
				}
				$number_imported = get_option( 'rcpga_import_success_count_' . $group_id );

				if ( empty( $number_imported ) ) {
					$message = __( 'Group members imported successfully', 'rcp-group-accounts' );
				} else {
					$message = sprintf( _n( '%s group member imported successfully', '%s group members imported successfully', $number_imported, 'rcp-group-accounts' ), absint( $number_imported ) );
					delete_option( 'rcpga_import_success_count_' . $group_id );
				}
				break;

			case 'group-members-imported-errors' :
				$group_id = ! empty( $_REQUEST['rcpga-group'] ) ? absint( $_REQUEST['rcpga-group'] ) : 0;
				if ( empty( $group_id ) ) {
					break;
				}
				$errors  = get_option( 'rcpga_import_errors_' . $group_id, array() );
				$message = sprintf( __( 'Errors: %s', 'rcp-group-accounts' ), '<br>' . implode( '<br>', $errors ) );
				$type    = 'error';

				// Also show number that were successful.
				$number_imported = get_option( 'rcpga_import_success_count_' . $group_id );
				if ( ! empty( $number_imported ) ) {
					$message = sprintf( _n( '%s group member imported successfully', '%s group members imported successfully', $number_imported, 'rcp-group-accounts' ), absint( $number_imported ) ) . '<br><br>' . $message;
					delete_option( 'rcpga_import_success_count_' . $group_id );
				}

				// Delete the option to prevent unnecessarily storing personal data - it's no longer needed now.
				delete_option( 'rcpga_import_errors_' . $group_id );
				break;

			case 'role-updated' :
				$message = __( 'Member\'s role successfully updated', 'rcp-group-accounts' );
				break;

			case 'password-updated' :
				$message = __( 'Member\'s password successfully updated', 'rcp-group-accounts' );
				break;

			case 'group-added' :
				$message = __( 'Group added successfully', 'rcp-group-accounts' );
				break;

			case 'group-updated' :
				$message = __( 'Group updated successfully', 'rcp-group-accounts' );
				break;

			case 'group-deleted' :
				$message = __( 'Group deleted successfully', 'rcp-group-accounts' );
				break;

			case 'invite-sent' :
				$message = __( 'Invite sent successfully', 'rcp-group-accounts' );
				break;

			case 'invite-accepted' :
				$message = __( 'Group membership confirmed. Please update your password now.', 'rcp-group-accounts' );
				break;

			case 'invite-resent' :
				$message = __( 'Your invite link has expired. A new one has been sent to your email.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'invite-expired' :
				$message = __( 'Your invite link has expired.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'invite-invalid' :
				$message = __( 'Invite link invalid.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'invite-completed' :
				if ( is_user_logged_in() ) {
					$message = __( 'Your group membership has already been confirmed.', 'rcp-group-accounts' );
				} else {
					$message = __( 'Your group membership has already been confirmed. You may log in below.', 'rcp-group-accounts' );
				}
				break;

			case 'member-registered' :
				$message = __( 'You have been successfully added to the group.', 'rcp-group-accounts' );
				break;

			case 'group-owner-changed' :
				$message = __( 'Group owner successfully changed.', 'rcp-group-accounts' );
				break;

			case 'no-user' :
				$message = __( 'That email does not appear to exist in our system', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'empty-email' :
				$message = __( 'Please enter a valid email address', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'empty-group-name' :
				$message = __( 'Please enter a group name', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'code-unavailable' :
				$message = __( 'This code is already in use', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'no-permission' :
				$message = __( 'You do not have permission to perform that action', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'no-group' :
				$message = __( 'Oops, no group ID was specified.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'no-member' :
				$message = __( 'Oops, no member was specified.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'existing_user_login' :
				$message = __( 'The username you entered has already been taken. Please choose another.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'has-group' :
				$message = __( 'This user is already in a group. Users may only be part of one group at a time.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'seats-maxed' :
				$message = __( 'There are not enough seats left in this group to handle this request.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'active-user' :
				if ( is_admin() ) {
					$message = __( 'The member you are trying to add already has a membership. Please expire or delete their membership so you can add them to the group.', 'rcp-group-accounts' );
				} else {
					$message = __( 'The member you are trying to add already has a membership. Please contact the member directly and have them cancel their existing membership so you can add them to the group.', 'rcp-group-accounts' );
				}
				$type    = 'error';
				break;

			case 'no-csv' :
				$message = __( 'Please upload a CSV file', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'removed-from-group' :
				$message = __( 'You have successfully been removed from the group.', 'rcp-group-accounts' );
				break;

			case 'member-not-added' :
				$message = __( 'This member could not be added.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

			case 'default-error' :
				$message = __( 'Something went wrong, please try again.', 'rcp-group-accounts' );
				$type    = 'error';
				break;

		}

		/**
		 * Filters the notice message.
		 *
		 * @param string $message Notice message.
		 * @param string $notice  Notice slug.
		 * @param string $type    Type of notice (`success` or `error`).
		 */
		$message = apply_filters( 'rcpga_notice_message', $message, $notice, $type );

		/**
		 * Filters the notice type.
		 *
		 * @param string $type    Type of notice (`success` or `error`).
		 * @param string $notice  Notice slug.
		 * @param string $message Notice message.
		 */
		$type = apply_filters( 'rcpga_notice_type', $type, $notice, $message );

		if ( empty( $message ) ) {
			return;
		}

		$this->message = $message;
		$this->type    = $type;

		add_action( 'admin_notices', array( $this, 'print_message_admin' ) );
		add_action( 'rcpga_dashboard_notifications', array( $this, 'print_message_front' ) );
		add_action( 'rcp_profile_editor_messages', array( $this, 'print_message_front' ) );
		add_action( 'rcp_before_login_form_fields', array( $this, 'print_message_front' ) );
		add_action( 'rcp_subscription_details_top', array( $this, 'print_message_front' ) );

	}

	/**
	 * Print notice in the admin area.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function print_message_admin() {
		$class = ( 'success' == $this->type ) ? 'updated' : 'error';
		printf( '<div class="%s"><p>%s</p></div>', esc_attr( sanitize_html_class( $class ) ), wp_kses_post( $this->message ) );
	}

	/**
	 * Print notice on the front-end.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function print_message_front() {
		$class = ( 'success' == $this->type ) ? 'rcp_success' : 'rcp_error';
		printf( '<p class="%s"><span>%s</span></p>', esc_attr( sanitize_html_class( $class ) ), wp_kses_post( $this->message ) );
	}

}