<?php
/**
 * Membership Level Settings
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA\Admin\Settings;

/**
 * Add a new membership level table column for "Seats".
 *
 * @param array $columns
 *
 * @since 2.0
 * @return array
 */
function seat_count_column( $columns ) {

	// We specifically want to add our column BEFORE "order".
	if ( array_key_exists( 'order', $columns ) ) {
		$offset  = array_search( 'order', array_keys( $columns ) );
		$columns = array_merge(
			array_slice( $columns, 0, $offset ),
			array( 'group_seats' => __( 'Seats', 'rcp-group-accounts' ) ),
			array_slice( $columns, $offset, null )
		);
	} else {
		$columns['group_seats'] = __( 'Seats', 'rcp-group-accounts' );
	}

	return $columns;

}

add_filter( 'rcp_membership_levels_list_table_columns', __NAMESPACE__ . '\seat_count_column' );

/**
 * Display the number of Group Accounts seats for the membership level.
 *
 * @param mixed  $value            Column value.
 * @param object $membership_level Membership level being displayed.
 *
 * @since 2.0
 * @return int
 */
function seat_count_value( $value, $membership_level ) {

	if ( 'per_seat' === rcpga_get_level_seat_pricing_type( $membership_level->id ) ) {
		return sprintf( '%d-%d', rcpga_get_level_min_seats( $membership_level->id ), rcpga_get_level_max_seats( $membership_level->id ) );
	} else {
		return rcpga_get_level_group_seats_allowed( $membership_level->id );
	}

}

add_filter( 'rcp_membership_levels_list_table_column_group_seats', __NAMESPACE__ . '\seat_count_value', 10, 2 );

/**
 * Display group accounts fields on the Add/Edit Membership Level forms.
 *
 * @param object|null $level Membership level object from the database, or null if this is a new level.
 *
 * @since 2.0
 * @return void
 */
function display_membership_level_fields( $level = null ) {

	/**
	 * Filters whether or not group accounts is enabled by default.
	 *
	 * @param bool        $enabled
	 * @param object|null $level
	 */
	$enabled               = apply_filters( 'rcpga_default_level_group_accounts_status', false, $level );
	$enabled               = ( empty( $level->id ) ) ? $enabled : rcpga_is_level_group_accounts_enabled( $level->id );
	$pricing_type          = ( empty( $level->id ) ) ? '' : rcpga_get_level_seat_pricing_type( $level->id );
	$pricing_type_disabled = false;
	$seat_count            = ( empty( $level->id ) ) ? 0 : rcpga_get_level_group_seats_allowed( $level->id );
	$min_seats             = ( empty( $level->id ) ) ? 1 : rcpga_get_level_min_seats( $level->id );
	$max_seats             = ( empty( $level->id ) ) ? 0 : rcpga_get_level_max_seats( $level->id );

	// Per-seat pricing is only allowed on RCP 3.3.9+;
	if ( version_compare( RCP_PLUGIN_VERSION, '3.4', '<' ) ) {
		$pricing_type          = 'fixed';
		$pricing_type_disabled = true;
	}
	?>

	<tr class="form-field">
		<th scope="row" valign="top">
			<label for="rcpga-group-seats-allow"><?php _e( 'Allow Group Account', 'rcp-group-accounts' ); ?></label>
		</th>
		<td>
			<input id="rcpga-group-seats-allow" type="checkbox" name="rcpga-group-seats-allow" <?php checked( $enabled ); ?> />
			<span class="description"><?php _e( 'Check to enable group accounts for this membership level.', 'rcp-group-accounts' ); ?></span>
		</td>
	</tr>

	<tr class="form-field"<?php echo $enabled ? '' : ' style="display:none;"'; ?>>
		<th scope="row" valign="top">
			<label for="rcpga-group-seat-pricing-type"><?php _e( 'Group Seat Pricing', 'rcp-group-accounts' ); ?></label>
		</th>
		<td>
			<select id="rcpga-group-seat-pricing-type" name="rcpga-group-seat-pricing-type"<?php echo $pricing_type_disabled ? ' disabled="disabled"' : ''; ?>>
				<option value="fixed" <?php selected( 'fixed', $pricing_type ); ?>><?php esc_html_e( 'Fixed Price', 'rcp-group-accounts' ); ?></option>
				<option value="per_seat" <?php selected( 'per_seat', $pricing_type ); ?>><?php esc_html_e( 'Per-Seat Price', 'rcp-group-accounts' ); ?></option>
			</select>
			<p class="description">
				<?php _e( '<strong>Fixed:</strong> Customers pay a fixed price for a specific number of seats.', 'rcp-group-accounts' ); ?>
				<br/>
				<?php _e( '<strong>Per-Seat:</strong> Customers can choose how many seats they want to purchase and are charged the specified "Price" for each seat.', 'rcp-group-accounts' ); ?>
				<?php if ( $pricing_type_disabled ) : ?>
				<br/>
				<?php printf( __( 'You must upgrade to RCP version %s or later to use per-seat pricing. You are on version %s.', 'rcp-group-accounts' ), '3.4', RCP_PLUGIN_VERSION ); ?>
				<?php endif; ?>
			</p>
		</td>
	</tr>

	<tr class="form-field" <?php echo ( $enabled && 'fixed' === $pricing_type ) ? '' : 'style="display:none;"' ?>>
		<th scope="row" valign="top">
			<label for="rcpga-group-seats"><?php _e( 'Group Seats', 'rcp-group-accounts' ); ?></label>
		</th>
		<td>
			<input id="rcpga-group-seats" type="number" name="rcpga-group-seats" value="<?php echo absint( $seat_count ); ?>" min="0" style="width: 100px;"/>
			<p class="description"><?php _e( 'The number of group seats available to this level including the group owner.', 'rcp-group-accounts' ); ?></p>
		</td>
	</tr>

	<tr class="form-field" <?php echo ( $enabled && 'per_seat' === $pricing_type ) ? '' : 'style="display:none;"' ?>>
		<th scope="row" valign="top">
			<label for="rcpga-group-min-seats"><?php _e( 'Group Minimum Seats', 'rcp-group-accounts' ); ?></label>
		</th>
		<td>
			<input id="rcpga-group-min-seats" type="number" name="rcpga-group-min-seats" value="<?php echo absint( $min_seats ); ?>" min="1" style="width: 100px;"/>
			<p class="description"><?php _e( 'Minimum number of seats allowed to be purchased.', 'rcp-group-accounts' ); ?></p>
		</td>
	</tr>

	<tr class="form-field" <?php echo ( $enabled && 'per_seat' === $pricing_type ) ? '' : 'style="display:none;"' ?>>
		<th scope="row" valign="top">
			<label for="rcpga-group-max-seats"><?php _e( 'Group Maximum Seats', 'rcp-group-accounts' ); ?></label>
		</th>
		<td>
			<input id="rcpga-group-max-seats" type="number" name="rcpga-group-max-seats" value="<?php echo absint( $max_seats ); ?>" min="0" style="width: 100px;"/>
			<p class="description"><?php _e( 'Maximum number of seats allowed to be purchased. Set to <code>0</code> for no limit.', 'rcp-group-accounts' ); ?></p>
		</td>
	</tr>

	<?php

}

add_action( 'rcp_add_subscription_form', __NAMESPACE__ . '\display_membership_level_fields' );
add_action( 'rcp_edit_subscription_form', __NAMESPACE__ . '\display_membership_level_fields' );

/**
 * Save the group accounts settings for the membership level.
 *
 * @param int   $membership_level_id ID of the membership level being added / edited.
 * @param array $args                Array of data.
 *
 * @since 2.0
 * @return void
 */
function save_membership_level_fields( $membership_level_id, $args ) {

	// Bail if using activate/deactivate row action.
	if ( ! empty( $_GET['rcp-action'] ) && in_array( $_GET['rcp-action'], array( 'activate_subscription', 'deactivate_subscription' ) ) ) {
		return;
	}

	// Bail if using activate/deactivate bulk action.
	if ( ! empty($_REQUEST['_wpnonce'] ) && wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-membershiplevels' ) ) {
		return;
	}

	if ( isset( $_POST['rcpga-group-seats-allow'] ) ) {
		rcpga_enable_level_group_accounts( $membership_level_id );

		$type = ! empty( $_POST['rcpga-group-seat-pricing-type'] ) ? $_POST['rcpga-group-seat-pricing-type'] : 'fixed';

		if ( ! in_array( $type, array( 'fixed', 'per_seat' ) ) ) {
			$type = 'fixed';
		}

		// Per-seat pricing requires RCP 3.3.9 or later.
		if ( 'per_seat' === $type && version_compare( RCP_PLUGIN_VERSION, '3.4', '<' ) ) {
			$type = 'fixed';
		}
	} else {
		rcpga_disable_level_group_accounts( $membership_level_id );

		$type = '';
	}

	// Save seat pricing type.
	if ( ! empty( $type ) ) {
		rcp_update_membership_level_meta( $membership_level_id, 'group_seat_pricing_type', sanitize_text_field( $type ) );
	} else {
		rcp_delete_membership_level_meta( $membership_level_id, 'group_seat_pricing_type' );
	}

	// Save fixed number of seats.
	if ( ! empty( $_POST['rcpga-group-seats'] ) && 'fixed' === $type ) {
		rcpga_set_level_group_seats_allowed( $membership_level_id, absint( $_POST['rcpga-group-seats'] ) );
	} else {
		rcpga_remove_level_seat_count( $membership_level_id );
	}

	// Save minimum seats.
	if ( 'per_seat' === $type ) {
		$min = ! empty( $_POST['rcpga-group-min-seats'] ) && absint( $_POST['rcpga-group-min-seats'] ) >= 1 ? absint( $_POST['rcpga-group-min-seats'] ) : 1;

		rcp_update_membership_level_meta( $membership_level_id, 'group_seats_min', absint( $min ) );
	} else {
		rcp_delete_membership_level_meta( $membership_level_id, 'group_seats_min' );
	}

	// Save maximum seats.
	if ( 'per_seat' === $type ) {
		$max = ! empty( $_POST['rcpga-group-max-seats'] ) && absint( $_POST['rcpga-group-max-seats'] ) >= 0 ? absint( $_POST['rcpga-group-max-seats'] ) : 0;

		rcp_update_membership_level_meta( $membership_level_id, 'group_seats_max', absint( $max ) );
	} else {
		rcp_delete_membership_level_meta( $membership_level_id, 'group_seats_max' );
	}

}

add_action( 'rcp_add_subscription', __NAMESPACE__ . '\save_membership_level_fields', 10, 2 );
add_action( 'rcp_edit_subscription_level', __NAMESPACE__ . '\save_membership_level_fields', 10, 2 );