<?php
/**
 * Admin Group Members Table
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA\Admin;

use \RCP\Admin\List_Table;

/**
 * Class Group_Members_Table
 *
 * @since   2.0
 * @package RCPGA\Admin
 */
class Group_Members_Table extends List_Table {

	/**
	 * Constructor.
	 *
	 * @since 2.0
	 * @see   WP_List_Table::__construct()
	 */
	public function __construct() {

		parent::__construct( array(
			'singular' => __( 'Group Member', 'rcp-group-accounts' ),
			'plural'   => __( 'Group Members', 'rcp-group-accounts' ),
			'ajax'     => false,
		) );

		$this->process_bulk_action();
		$this->get_counts();

	}

	/**
	 * Get the base URL for the groups list table.
	 *
	 * @since 2.0
	 * @return string Base URL.
	 */
	public function get_base_url() {
		$args = array(
			'rcpga-view' => 'view-members'
		);

		$group_id = $this->get_group_id();

		if ( ! empty( $group_id ) ) {
			$args['rcpga-group'] = urlencode( $group_id );
		}

		return get_groups_page( $args );
	}

	/**
	 * Retrieve the table columns.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_columns() {

		$columns = array(
			'cb'         => '<input type="checkbox" />',
			'name'       => __( 'Member Name', 'rcp' ),
			'group_name' => __( 'Group Name', 'rcp' ),
			'role'       => __( 'Role', 'rcp' ),
			'date_added' => __( 'Date Added', 'rcp' )
		);

		/*
		 * Backwards compatibility: add an "extra" column if someone is hooking into the old action to add
		 * their own column. Everything gets bundled into one column because this is the only way we can realistically
		 * do it.
		 */
		if ( has_action( 'rcpga_group_members_page_table_header' ) ) {
			$columns['custom'] = __( 'Extra', 'rcp-group-accounts' );
		}

		/**
		 * Filters the Groups table columns.
		 *
		 * @param array $columns
		 *
		 * @since 2.0
		 */
		$columns = apply_filters( 'rcpga_group_members_list_table_columns', $columns );

		return $columns;

	}

	/**
	 * Retrieve the sortable columns.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_sortable_columns() {
		return array(
			'role'       => array( 'role', false ),
			'date_added' => array( 'date_added', false )
		);
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @since 2.0
	 * @return string
	 */
	protected function get_primary_column_name() {
		return 'name';
	}

	/**
	 * Render the checkbox column.
	 *
	 * @param \RCPGA_Group_Member $member
	 *
	 * @since 2.0
	 * @return string
	 */
	public function column_cb( $member ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			'group_member_id',
			$member->get_id()
		);
	}

	/**
	 * Renders the "Name" column
	 *
	 * @param \RCPGA_Group_Member $member
	 *
	 * @since 2.0
	 * @return string
	 */
	public function column_name( $member ) {

		$user    = get_userdata( $member->get_user_id() );
		$actions = array();

		if ( 'owner' != $member->get_role() ) {

			if ( 'admin' === $member->get_role() ) {

				// Admin - set as member
				$url = wp_nonce_url( add_query_arg( array(
					'rcpga-action'    => 'make-member',
					'group-member-id' => urlencode( $member->get_id() )
				), $this->get_base_url() ), 'rcpga_make_member' );

				$actions['make_member'] = '<a href="' . esc_url( $url ) . '">' . __( 'Set as Member', 'rcp-group-accounts' ) . '</a>';

			} elseif ( 'member' === $member->get_role() ) {

				// Member - set as admin
				$url = wp_nonce_url( add_query_arg( array(
					'rcpga-action'    => 'make-admin',
					'group-member-id' => urlencode( $member->get_id() )
				), $this->get_base_url() ), 'rcpga_make_admin' );

				$actions['make_admin'] = '<a href="' . esc_url( $url ) . '">' . __( 'Set as Admin', 'rcp-group-accounts' ) . '</a>';

			} else {

				// Invited - re-send invite
				$url = wp_nonce_url( add_query_arg( array(
					'rcpga-action'    => 'resend-invite',
					'group-member-id' => urlencode( $member->get_id() )
				), $this->get_base_url() ), 'rcpga_resend_invite' );

				$actions['resend_invite'] = '<a href="' . esc_url( $url ) . '">' . __( 'Resend Invite', 'rcp-group-accounts' ) . '</a>';

				// Invited - set as member
				$url = wp_nonce_url( add_query_arg( array(
					'rcpga-action'    => 'make-member',
					'group-member-id' => urlencode( $member->get_id() )
				), $this->get_base_url() ), 'rcpga_make_member' );

				$actions['make_admin'] = '<a href="' . esc_url( $url ) . '">' . __( 'Set as Member', 'rcp-group-accounts' ) . '</a>';

			}

			// Make Owner
			$url = wp_nonce_url( add_query_arg( array(
				'rcpga-action'    => 'make-owner',
				'group-member-id' => urlencode( $member->get_id() )
			), $this->get_base_url() ), 'rcpga_make_owner' );

			$actions['make_owner'] = '<a href="' . esc_url( $url ) . '" class="rcpga-make-owner">' . __( 'Set as Owner', 'rcp-group-accounts' ) . '</a>';

			// Remove from group
			$url = wp_nonce_url( add_query_arg( array(
				'rcpga-action'    => 'remove-member',
				'group-member-id' => urlencode( $member->get_id() )
			), $this->get_base_url() ), 'rcpga_remove_from_group' );

			$actions['remove'] = '<span class="trash "><a href="' . esc_url( $url ) . '" class="rcp-member-delete">' . __( 'Remove from Group', 'rcp-group-accounts' ) . '</a></span>';

		}

		$edit_user_url = add_query_arg( 'user_id', urlencode( $member->get_user_id() ), admin_url( 'user-edit.php' ) );

		$actions['user_id'] = '<span class="rcp-id-col">' . sprintf( __( 'User ID: %s', 'rcp-group-accounts' ), '<a href="' . esc_url( $edit_user_url ) . '">' . $member->get_user_id() ) . '</span></span>';

		/**
		 * Filters the row actions.
		 *
		 * @param array               $actions Default actions.
		 * @param \RCPGA_Group_Member $member  Group member object.
		 *
		 * @since 2.0
		 */
		$actions = apply_filters( 'rcpga_group_members_list_table_row_actions', $actions, $member );

		$customer_page = $edit_user_url;
		$customer      = rcp_get_customer_by_user_id( $member->get_user_id() );

		if ( ! empty( $customer ) ) {
			$customer_page = rcp_get_customers_admin_page( array(
				'customer_id' => $customer->get_id(),
				'view'        => 'edit'
			) );
		}

		$name = ! empty( $user->display_name ) ? $user->display_name : $user->user_login;

		return '<strong><a class="row-title" href="' . esc_url( $customer_page ) . '">' . esc_html( $name ) . '</a></strong>' . $this->row_actions( $actions );

	}

	/**
	 * This function renders any other columns in the list table.
	 *
	 * @param \RCPGA_Group_Member $member      Group object.
	 * @param string              $column_name The name of the column
	 *
	 * @since 2.0
	 * @return string Column Name
	 */
	public function column_default( $member, $column_name ) {

		$value = '';

		/*
		 * Backwards compatibility: show content of custom columns from old action hook.
		 */
		if ( 'custom' == $column_name && has_action( 'rcpga_group_members_page_table_column' ) ) {
			ob_start();
			do_action( 'rcpga_group_members_page_table_column', $member );
			$column_content = ob_get_clean();

			$value = wp_strip_all_tags( $column_content );
		} else {

			switch ( $column_name ) {
				case 'group_name' :
					$group = rcpga_get_group( $member->get_group_id() );
					if ( ! empty( $group ) ) {
						$value = '<a href="' . esc_url( get_groups_page( array(
								'rcpga-group' => urlencode( $group->get_group_id() ),
								'rcpga-view'  => 'edit'
							) ) ) . '">' . $group->get_name() . '</a>';
					}
					break;

				case 'role' :
					$value = rcpga_get_member_role_label( $member->get_role() );
					break;

				case 'date_added' :
					$value = $member->get_date_added( true );
					break;
			}

		}

		/**
		 * Filters the column value.
		 *
		 * @param string              $value  Column value.
		 * @param \RCPGA_Group_Member $member Group member object.
		 *
		 * @since 2.0
		 */
		$value = apply_filters( 'rcpga_group_members_list_table_column_' . $column_name, $value, $member );

		return $value;

	}

	/**
	 * Message to be displayed when there are no group members.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function no_items() {
		esc_html_e( 'No group members found.', 'rcp-group-accounts' );
	}

	/**
	 * Retrieve the bulk actions.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_bulk_actions() {
		return array(
			'remove' => __( 'Remove from Group', 'rcp-group-accounts' )
		);
	}

	/**
	 * Process bulk actions.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function process_bulk_action() {

		// Bail if a nonce was not supplied.
		if ( ! isset( $_REQUEST['_wpnonce'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-groupmembers' ) ) {
			return;
		}

		$ids = wp_parse_id_list( (array) $this->get_request_var( 'group_member_id', false ) );

		// Bail if no IDs
		if ( empty( $ids ) ) {
			return;
		}

		foreach ( $ids as $id ) {
			$member = rcpga_get_group_member_by_id( absint( $id ) );

			if ( empty( $member ) ) {
				return;
			}

			switch ( $this->current_action() ) {
				case 'remove':
					$member->remove();
					break;
			}
		}

		$this->show_admin_notice( $this->current_action() );

	}

	/**
	 * Show admin notice for bulk actions.
	 *
	 * @param string $action The action to show the notice for.
	 *
	 * @access private
	 * @since  2.0
	 * @return void
	 */
	private function show_admin_notice( $action ) {

		$message = '';

		switch ( $action ) {
			case 'remove' :
				$message = __( 'Member(s) removed.', 'rcp-group-accounts' );
				break;
		}

		if ( empty( $message ) ) {
			return;
		}

		echo '<div class="updated"><p>' . $message . '</p></div>';

	}

	/**
	 * Get the ID of the group currently being queried
	 *
	 * @since 2.0
	 * @return int
	 */
	private function get_group_id() {
		return ! empty( $_GET['rcpga-group'] ) ? absint( $_GET['rcpga-group'] ) : 0;
	}

	/**
	 * Set the group member counts
	 *
	 * @since 2.0
	 * @return void
	 */
	public function get_counts() {
		$args     = array();
		$group_id = $this->get_group_id();

		if ( ! empty( $group_id ) ) {
			$args['group_id'] = $group_id;
		}

		$this->counts = rcpga_get_group_member_counts( $args );
	}

	/**
	 * Retrieve group member data
	 *
	 * @param bool $count Whether to get group member objects (false) or just count the total number (true).
	 *
	 * @since 2.0
	 * @return array|int
	 */
	public function members_data( $count = false ) {

		$search = $this->get_search();

		$args = array(
			'role'    => $this->get_request_var( 'status', '' ),
			'number'  => $this->per_page,
			'offset'  => $this->get_offset(),
			'orderby' => sanitize_text_field( $this->get_request_var( 'orderby', 'role' ) ),
			'order'   => sanitize_text_field( $this->get_request_var( 'order', 'DESC' ) )
		);

		if ( ! empty( $search ) ) {
			// First we have to search for user accounts.
			$user_ids = get_users( array(
				'number' => - 1,
				'search' => '*' . $search . '*',
				'fields' => 'ids'
			) );

			// No user results - bail.
			if ( empty( $user_ids ) ) {
				return $count ? 0 : array();
			}

			// Include these user IDs in our search.
			$args['user_id__in'] = array_map( 'absint', $user_ids );
		}

		if ( ! empty( $_GET['rcpga-group'] ) ) {
			$args['group_id'] = absint( $_GET['rcpga-group'] );
		}

		if ( $count ) {
			return rcpga_count_group_members( $args );
		}

		return rcpga_get_group_members( $args );

	}

	/**
	 * Setup the final data for the table.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function prepare_items() {
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );
		$this->items           = $this->members_data();

		$total = $this->members_data( true );

		// Setup pagination
		$this->set_pagination_args( array(
			'total_items' => $total,
			'per_page'    => $this->per_page,
			'total_pages' => ceil( $total / $this->per_page )
		) );
	}

	/**
	 * Show the search field.
	 *
	 * @param string $text     Label for the search box
	 * @param string $input_id ID of the search box
	 *
	 * @since 2.0
	 * @return void
	 */
	public function search_box( $text, $input_id ) {

		// Bail if no items and no search
		if ( ! $this->get_search() && ! $this->has_items() ) {
			return;
		}

		$orderby  = $this->get_request_var( 'orderby' );
		$order    = $this->get_request_var( 'order' );
		$group_id = $this->get_request_var( 'rcpga-group' );
		$view     = $this->get_request_var( 'rcpga-view' );
		$input_id = $input_id . '-search-input';

		if ( ! empty( $orderby ) ) {
			echo '<input type="hidden" name="orderby" value="' . esc_attr( $orderby ) . '" />';
		}

		if ( ! empty( $order ) ) {
			echo '<input type="hidden" name="order" value="' . esc_attr( $order ) . '" />';
		}

		if ( ! empty( $group_id ) ) {
			echo '<input type="hidden" name="rcpga-group" value="' . esc_attr( $group_id ) . '" />';
		}

		if ( ! empty( $view ) ) {
			echo '<input type="hidden" name="rcpga-view" value="' . esc_attr( $view ) . '" />';
		}
		?>
		<p class="search-box">
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_html( $text ); ?>
				:</label>
			<input type="search" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>" placeholder="<?php esc_attr_e( 'Username or email', 'rcp-group-accounts' ); ?>"/>
			<?php submit_button( esc_html( $text ), 'button', false, false, array( 'ID' => 'search-submit' ) ); ?>
		</p>
		<?php

	}

}