<?php
/**
 * Admin Group Table
 *
 * @package   rcp-group-accounts
 * @copyright Copyright (c) 2019, Restrict Content Pro team
 * @license   GPL2+
 * @since     2.0
 */

namespace RCPGA\Admin;

use \RCP\Admin\List_Table;

/**
 * Class Groups_Table
 *
 * @since   2.0
 * @package RCPGA\Admin
 */
class Groups_Table extends List_Table {

	/**
	 * Constructor.
	 *
	 * @since 2.0
	 * @see   WP_List_Table::__construct()
	 */
	public function __construct() {

		parent::__construct( array(
			'singular' => __( 'Group', 'rcp-group-accounts' ),
			'plural'   => __( 'Groups', 'rcp-group-accounts' ),
			'ajax'     => false,
		) );

		$this->process_bulk_action();
		$this->get_counts();

	}

	/**
	 * Get the base URL for the groups list table.
	 *
	 * @since 2.0
	 * @return string Base URL.
	 */
	public function get_base_url() {
		return get_groups_page();
	}

	/**
	 * Retrieve the table columns.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_columns() {

		$columns = array(
			'cb'           => '<input type="checkbox" />',
			'name'         => __( 'Name', 'rcp' ),
			'owner'        => __( 'Owner', 'rcp' ),
			'description'  => __( 'Description', 'rcp' ),
			'members'      => __( 'Members', 'rcp' ),
			'seats'        => __( 'Seats', 'rcp' ),
			'date_created' => __( 'Date Created', 'rcp' ),
		);

		/*
		 * Backwards compatibility: add an "extra" column if someone is hooking into the old action to add
		 * their own column. Everything gets bundled into one column because this is the only way we can realistically
		 * do it.
		 */
		if ( has_action( 'rcpga_groups_page_table_header' ) ) {
			$columns['custom'] = __( 'Extra', 'rcp-group-accounts' );
		}

		/**
		 * Filters the Groups table columns.
		 *
		 * @param array $columns
		 *
		 * @since 2.0
		 */
		$columns = apply_filters( 'rcpga_groups_list_table_columns', $columns );

		return $columns;

	}

	/**
	 * Retrieve the sortable columns.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_sortable_columns() {
		return array(
			'name'         => array( 'name', false ),
			'members'      => array( 'members', false ),
			'seats'        => array( 'seats', false ),
			'date_created' => array( 'date_created', false )
		);
	}

	/**
	 * Gets the name of the primary column.
	 *
	 * @since 2.0
	 * @return string
	 */
	protected function get_primary_column_name() {
		return 'name';
	}

	/**
	 * Render the checkbox column.
	 *
	 * @param \RCPGA_Group $group
	 *
	 * @since 2.0
	 * @return string
	 */
	public function column_cb( $group ) {
		return sprintf(
			'<input type="checkbox" name="%1$s[]" value="%2$s" />',
			'group_id',
			$group->get_group_id()
		);
	}

	/**
	 * Renders the "Name" column
	 *
	 * @param \RCPGA_Group $group
	 *
	 * @since 2.0
	 * @return string
	 */
	public function column_name( $group ) {

		$edit_group_url    = get_groups_page( array(
			'rcpga-group' => $group->get_group_id(),
			'rcpga-view'  => 'edit'
		) );
		$group_members_url = get_groups_page( array(
			'rcpga-group' => $group->get_group_id(),
			'rcpga-view'  => 'view-members'
		) );
		$group_delete_url  = wp_nonce_url( add_query_arg( array(
			'rcpga-action' => 'delete-group',
			'rcpga-group'  => $group->get_group_id()
		), $this->get_base_url() ), 'rcpga_delete_group_nonce' );

		$actions = array(
			'edit_group'   => '<a href="' . esc_url( $edit_group_url ) . '">' . __( 'Edit Group', 'rcp-group-accounts' ) . '</a>',
			'view_members' => '<a href="' . esc_url( $group_members_url ) . '">' . __( 'View Members', 'rcp-group-accounts' ) . '</a>',
			'delete_group' => '<span class="trash"><a href="' . esc_url( $group_delete_url ) . '" class="rcpga-group-delete">' . __( 'Delete', 'rcp-group-accounts' ) . '</a></span>',
			'group_id'     => '<span class="rcp-id-col">' . sprintf( __( 'ID: %d', 'rcp-group-accounts' ), $group->get_group_id() ) . '</span>'
		);

		/**
		 * Filters the row actions.
		 *
		 * @param array        $actions Default actions.
		 * @param \RCPGA_Group $group   Group object.
		 *
		 * @since 2.0
		 */
		$actions = apply_filters( 'rcpga_groups_list_table_row_actions', $actions, $group );

		return '<strong><a class="row-title" href="' . esc_url( $edit_group_url ) . '">' . esc_html( $group->get_name() ) . '</a></strong>' . $this->row_actions( $actions );

	}

	/**
	 * This function renders any other columns in the list table.
	 *
	 * @param \RCPGA_Group $group       Group object.
	 * @param string       $column_name The name of the column
	 *
	 * @since 2.0
	 * @return string Column Name
	 */
	public function column_default( $group, $column_name ) {

		$value = '';

		/*
		 * Backwards compatibility: show content of custom columns from old action hook.
		 */
		if ( 'custom' == $column_name && has_action( 'rcpga_groups_page_table_column' ) ) {
			ob_start();
			do_action( 'rcpga_groups_page_table_column', $group );
			$column_content = ob_get_clean();

			$value = wp_strip_all_tags( $column_content );
		} else {

			switch ( $column_name ) {
				case 'owner' :
					$owner         = $group->get_owner();
					$customer      = rcp_get_customer_by_user_id( $owner->ID );

					if ( ! empty( $customer ) ) {
						$customer_page = rcp_get_customers_admin_page( array(
							'customer_id' => $customer->get_id(),
							'view'        => 'edit'
						) );
					} else {
						$customer_page = add_query_arg( 'user_id', urlencode( $owner->ID ), admin_url( 'user-edit.php' ) );
					}

					$value = '<a href="' . esc_url( $customer_page ) . '">' . $owner->display_name . '</a>';
					break;

				case 'description' :
					$value = $group->get_description();
					break;

				case 'members' :
					$value = $group->get_member_count();
					break;

				case 'seats' :
					$value = $group->get_seats();
					break;

				case 'date_created' :
					$value = $group->get_date_created( true );
					break;
			}

		}

		/**
		 * Filters the column value.
		 *
		 * @param string       $value Column value.
		 * @param \RCPGA_Group $group Group object.
		 *
		 * @since 2.0
		 */
		$value = apply_filters( 'rcpga_groups_list_table_column_' . $column_name, $value, $group );

		return $value;

	}

	/**
	 * Message to be displayed when there are no groups.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function no_items() {
		esc_html_e( 'No groups found.', 'rcp-group-accounts' );
	}

	/**
	 * Retrieve the bulk actions.
	 *
	 * @since 2.0
	 * @return array
	 */
	public function get_bulk_actions() {
		return array(
			'delete' => __( 'Permanently Delete', 'rcp-group-accounts' )
		);
	}

	/**
	 * Process bulk actions.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function process_bulk_action() {

		// Bail if a nonce was not supplied.
		if ( ! isset( $_REQUEST['_wpnonce'] ) ) {
			return;
		}

		if ( ! wp_verify_nonce( $_REQUEST['_wpnonce'], 'bulk-groups' ) ) {
			return;
		}

		$ids = wp_parse_id_list( (array) $this->get_request_var( 'group_id', false ) );

		// Bail if no IDs
		if ( empty( $ids ) ) {
			return;
		}

		foreach ( $ids as $id ) {
			$group = rcpga_get_group( absint( $id ) );

			if ( empty( $group ) ) {
				continue;
			}

			switch ( $this->current_action() ) {
				case 'delete':
					$group->delete();
					break;
			}
		}

		$this->show_admin_notice( $this->current_action() );

	}

	/**
	 * Show admin notice for bulk actions.
	 *
	 * @param string $action The action to show the notice for.
	 *
	 * @access private
	 * @since  2.0
	 * @return void
	 */
	private function show_admin_notice( $action ) {

		$message = '';

		switch ( $action ) {
			case 'delete' :
				$message = __( 'Group(s) deleted.', 'rcp-group-accounts' );
				break;
		}

		if ( empty( $message ) ) {
			return;
		}

		echo '<div class="updated"><p>' . $message . '</p></div>';

	}

	/**
	 * Set the group counts
	 *
	 * @since 2.0
	 * @return void
	 */
	public function get_counts() {
		$this->counts = array(
			'total' => rcpga_count_groups()
		);
	}

	/**
	 * Retrieve group data
	 *
	 * @param bool $count Whether to get group objects (false) or just count the total number (true).
	 *
	 * @since 2.0
	 * @return array|int
	 */
	public function groups_data( $count = false ) {

		$search = $this->get_search();

		$args = array(
			'number'  => $this->per_page,
			'offset'  => $this->get_offset(),
			'orderby' => sanitize_text_field( $this->get_request_var( 'orderby', 'date_created' ) ),
			'order'   => sanitize_text_field( $this->get_request_var( 'order', 'DESC' ) )
		);

		if ( ! empty( $search ) ) {
			$args['search'] = $search;
		}

		if ( $count ) {
			return rcpga_count_groups( $args );
		}

		return rcpga_get_groups( $args );

	}

	/**
	 * Setup the final data for the table.
	 *
	 * @since 2.0
	 * @return void
	 */
	public function prepare_items() {
		$columns  = $this->get_columns();
		$hidden   = array();
		$sortable = $this->get_sortable_columns();

		$this->_column_headers = array( $columns, $hidden, $sortable );
		$this->items           = $this->groups_data();

		$total = $this->groups_data( true );

		// Setup pagination
		$this->set_pagination_args( array(
			'total_items' => $total,
			'per_page'    => $this->per_page,
			'total_pages' => ceil( $total / $this->per_page )
		) );
	}

	/**
	 * Show the search field.
	 *
	 * @param string $text     Label for the search box
	 * @param string $input_id ID of the search box
	 *
	 * @since 2.0
	 */
	public function search_box( $text, $input_id ) {

		// Bail if no items and no search
		if ( ! $this->get_search() && ! $this->has_items() ) {
			return;
		}

		$orderby     = $this->get_request_var( 'orderby' );
		$order       = $this->get_request_var( 'order' );
		$search_type = $this->get_request_var( 'search_type', 'group' );
		$input_id    = $input_id . '-search-input';

		if ( ! empty( $orderby ) ) {
			echo '<input type="hidden" name="orderby" value="' . esc_attr( $orderby ) . '" />';
		}

		if ( ! empty( $order ) ) {
			echo '<input type="hidden" name="order" value="' . esc_attr( $order ) . '" />';
		}

		?>

		<p class="search-box">
			<label class="screen-reader-text" for="rcp-groups-search-type"><?php esc_html_e( 'Choose a field to search', 'rcp-group-accounts' ); ?></label>
			<select id="rcp-groups-search-type" name="search_type" style="float:left;">
				<option value="group" <?php selected( $search_type, 'group' ); ?>><?php _e( 'Group Name/Code', 'rcp' ); ?></option>
				<option value="member" <?php selected( $search_type, 'member' ); ?>><?php _e( 'Group Member', 'rcp' ); ?></option>
			</select>
			<label class="screen-reader-text" for="<?php echo esc_attr( $input_id ); ?>"><?php echo esc_html( $text ); ?>
				:</label>
			<input type="search" id="<?php echo esc_attr( $input_id ); ?>" name="s" value="<?php _admin_search_query(); ?>"/>
			<?php submit_button( esc_html( $text ), 'button', false, false, array( 'ID' => 'search-submit' ) ); ?>
		</p>

		<?php
	}

}