<?php

namespace PrestoPlayer\Pro\Support;

class Utility {

	/**
	 * Is it an hls stream
	 *
	 * @param string $src
	 * @return boolean
	 */
	public static function isStream( $src ) {
		if ( empty( $src ) ) {
			return false;
		}

		$parts = pathinfo( $src );

		if ( ! empty( $parts['extension'] ) ) {
			return 'm3u8' === $parts['extension'] || 'jpg' === $parts['extension'];
		}

		return false;
	}

	/**
	 * Sanitize GUID parameter
	 *
	 * Removes whitespace and strips any characters that aren't
	 * alphanumeric or hyphens (UUID format).
	 *
	 * @param mixed $value The GUID value to sanitize.
	 * @return string Sanitized GUID.
	 */
	public static function sanitizeGuid( $value ) {
		if ( ! is_string( $value ) ) {
			return '';
		}
		// Remove any whitespace
		$value = trim( $value );
		// Only allow alphanumeric characters and hyphens (UUID format)
		return preg_replace( '/[^a-zA-Z0-9\-]/', '', $value );
	}

	/**
	 * Validate GUID parameter format
	 *
	 * GUID is treated as an opaque identifier provided by Bunny.
	 * Validates format and prevents path traversal attacks.
	 * Existence is validated at the application layer.
	 *
	 * @param mixed $value The GUID value to validate.
	 * @return bool
	 */
	public static function validateGuidFormat( $value ) {
		if ( ! is_string( $value ) ) {
			return false;
		}

		$value = trim( $value );

		// Check for empty value
		if ( empty( $value ) ) {
			return false;
		}

		// Check for path traversal attempts
		if ( strpos( $value, '..' ) !== false || strpos( $value, '/' ) !== false ) {
			return false;
		}

		// Validate GUID format - only alphanumeric and hyphens allowed
		// Bunny GUIDs are typically UUIDs: 8-4-4-4-12 hex digits
		// Using loose validation to remain flexible if the format changes
		return (bool) preg_match( '/^[a-zA-Z0-9\-]+$/', $value );
	}
}
