<?php

namespace PrestoPlayer\Pro\Services;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Models\Bunny\PullZone;
use PrestoPlayer\Pro\Models\Bunny\StorageZone;
use PrestoPlayer\Pro\Services\Bunny\BunnyTranscriptionService;

class Settings {

	public function register() {
		add_action( 'admin_init', array( $this, 'registerSettings' ) );
		add_action( 'rest_api_init', array( $this, 'registerSettings' ) );

		add_filter( 'pre_option_presto_player_fluentcrm', array( $this, 'filterFluent' ) );

		// Update libraries when transcription settings change
		add_action( 'update_option_presto_player_bunny_stream', array( $this, 'onTranscriptionSettingsChange' ), 10, 3 );

		// Also handle first-time add (update_option on non-existent option uses add_option internally).
		add_action( 'add_option_presto_player_bunny_stream', array( $this, 'onTranscriptionSettingsAdd' ), 10, 2 );
	}

	// option will update if fluent crm is active
	public function filterFluent( $value ) {
		if ( is_array( $value ) ) {
			return array_merge( $value, array( 'connected' => is_plugin_active( 'fluent-crm/fluent-crm.php' ) ) );
		}
		return $value;
	}

	public function registerSettings() {
		\register_setting(
			'presto_player',
			'presto_player_bunny_stream_public',
			array(
				'type'         => 'object',
				'description'  => __( 'Bunny.net public video stream settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_bunny_stream_public',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'video_library_id'      => array(
								'type' => 'integer',
							),
							'video_library_api_key' => array(
								'type' => 'string',
							),
							'pull_zone_id'          => array(
								'type' => 'integer',
							),
							'pull_zone_url'         => array(
								'type' => 'string',
							),
							'token_auth_key'        => array(
								'type' => 'string',
							),
						),
					),
				),
				'default'      => array(),
			)
		);

		\register_setting(
			'presto_player',
			'presto_player_bunny_stream',
			array(
				'type'         => 'object',
				'description'  => __( 'Bunny.net stream general settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_bunny_stream',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'hls_start_level'        => array(
								'type' => 'integer',
							),
							'disable_legacy_storage' => array(
								'type' => 'boolean',
							),
							'transcribe_enabled'     => array(
								'type' => 'boolean',
							),
							'transcribe_languages'   => array(
								'type'  => 'array',
								'items' => array(
									'type' => 'string',
								),
							),
							'webhook_url'            => array(
								'type' => 'string',
							),
						),
					),
				),
				'default'      => array(
					'hls_start_level'        => 480,
					'disable_legacy_storage' => false,
					'transcribe_enabled'     => false,
					'transcribe_languages'   => array(),
				),
			)
		);

		\register_setting(
			'presto_player',
			'presto_player_bunny_stream_private',
			array(
				'type'         => 'object',
				'description'  => __( 'Bunny.net private video stream settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_bunny_stream_private',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'video_library_id'      => array(
								'type' => 'integer',
							),
							'video_library_api_key' => array(
								'type' => 'string',
							),
							'pull_zone_id'          => array(
								'type' => 'integer',
							),
							'pull_zone_url'         => array(
								'type' => 'string',
							),
							'token_auth_key'        => array(
								'type' => 'string',
							),
						),
					),
				),
				'default'      => array(),
			)
		);

		$pull_zone = new PullZone();
		\register_setting(
			'presto_player',
			'presto_player_bunny_pull_zones',
			array(
				'type'         => 'object',
				'description'  => __( 'Bunny.net settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_bunny_pull_zones',
					'type'   => 'object',
					'schema' => array(
						'properties' => $pull_zone->getFillableSchema(),
					),
				),
				'default'      => array(),
			)
		);

		$storage_zone = new StorageZone();
		\register_setting(
			'presto_player',
			'presto_player_bunny_storage_zones',
			array(
				'type'         => 'object',
				'description'  => __( 'Bunny.net settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_bunny_storage_zones',
					'type'   => 'object',
					'schema' => array(
						'properties' => $storage_zone->getFillableSchema(),
					),
				),
				'default'      => array(),
			)
		);

		// unique install id
		\register_setting(
			'presto_player',
			'presto_player_bunny_uid',
			array(
				'type'         => 'string',
				'description'  => __( 'A generated unique install id for Bunny.net.', 'presto-player-pro' ),
				'show_in_rest' => true,
			)
		);

		// mailchimp
		\register_setting(
			'presto_player',
			'presto_player_mailchimp',
			array(
				'type'         => 'object',
				'description'  => __( 'Mailchimp settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_mailchimp',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'api_key'   => array(
								'type' => 'string',
							),
							'connected' => array(
								'type' => 'boolean',
							),
						),
					),
				),
				'default'      => array(
					'api_key'   => '',
					'connected' => false,
				),
			)
		);

		// mailerlite
		\register_setting(
			'presto_player',
			'presto_player_mailerlite',
			array(
				'type'         => 'object',
				'description'  => __( 'Mailerlite settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_mailerlite',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'api_key'   => array(
								'type' => 'string',
							),
							'connected' => array(
								'type' => 'boolean',
							),
						),
					),
				),
				'default'      => array(
					'api_key'   => '',
					'connected' => false,
				),
			)
		);

		// activecampaign
		\register_setting(
			'presto_player',
			'presto_player_activecampaign',
			array(
				'type'         => 'object',
				'description'  => __( 'ActiveCampaign settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_activecampaign',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'api_key'   => array(
								'type' => 'string',
							),
							'url'       => array(
								'type' => 'string',
							),
							'connected' => array(
								'type' => 'boolean',
							),
						),
					),
				),
				'default'      => array(
					'api_key'   => '',
					'url'       => '',
					'connected' => false,
				),
			)
		);

		// fluent crm
		\register_setting(
			'presto_player',
			'presto_player_fluentcrm',
			array(
				'type'         => 'object',
				'description'  => __( 'Is fluent crm connected', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_fluentcrm',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'connected' => array(
								'type' => 'boolean',
							),
						),
					),
				),
				'default'      => array(
					'connected' => false,
				),
			)
		);

		/**
		 * License
		 */
		\register_setting(
			'presto_player',
			'presto_player_license',
			array(
				'type'         => 'object',
				'description'  => __( 'License settings.', 'presto-player-pro' ),
				'show_in_rest' => array(
					'name'   => 'presto_player_license',
					'type'   => 'object',
					'schema' => array(
						'properties' => array(
							'key' => array(
								'type' => 'string',
							),
						),
					),
				),
				'default'      => array(
					'enable'     => false,
					'purge_data' => true,
				),
			)
		);
	}

	/**
	 * Handle transcription settings changes.
	 *
	 * Updates video libraries when transcription settings are modified.
	 *
	 * @param mixed  $old_value The old option value.
	 * @param mixed  $new_value The new option value.
	 * @param string $option    The option name.
	 */
	public function onTranscriptionSettingsChange( $old_value, $new_value, $option ) {
		// Only process changes to the bunny stream option
		if ( 'presto_player_bunny_stream' !== $option ) {
			return;
		}

		// Only proceed if transcription settings have actually changed
		$old_transcribe_enabled   = isset( $old_value['transcribe_enabled'] ) ? $old_value['transcribe_enabled'] : false;
		$old_transcribe_languages = isset( $old_value['transcribe_languages'] ) ? $old_value['transcribe_languages'] : array();

		$new_transcribe_enabled   = isset( $new_value['transcribe_enabled'] ) ? $new_value['transcribe_enabled'] : false;
		$new_transcribe_languages = isset( $new_value['transcribe_languages'] ) ? $new_value['transcribe_languages'] : array();

		// Check if transcription settings changed
		if ( $old_transcribe_enabled !== $new_transcribe_enabled ||
			$old_transcribe_languages !== $new_transcribe_languages ) {

			$this->updateLibrariesTranscriptionSettings();
		}
	}

	/**
	 * Handle transcription settings when option is first added.
	 *
	 * When the bunny stream option doesn't exist yet, WordPress uses add_option()
	 * internally, so update_option_* never fires. This hook ensures we sync
	 * transcription settings to libraries on first-time save.
	 *
	 * @param string $option The option name.
	 * @param mixed  $value  The option value.
	 */
	public function onTranscriptionSettingsAdd( $option, $value ) {
		if ( 'presto_player_bunny_stream' !== $option || ! is_array( $value ) ) {
			return;
		}
		// Value already saved; sync libraries with current DB state.
		$this->updateLibrariesTranscriptionSettings();
	}

	/**
	 * Update all video libraries with current transcription settings.
	 *
	 * @return void
	 */
	protected function updateLibrariesTranscriptionSettings(): void {
		$transcribe_enabled = (bool) Setting::get( 'bunny_stream', 'transcribe_enabled', false );
		$types              = array( 'public', 'private' );
		foreach ( $types as $type ) {
			$library_id = Setting::get( 'bunny_stream_' . $type, 'video_library_id' );
			if ( $library_id && is_numeric( $library_id ) ) {
				BunnyTranscriptionService::updateLibraryTranscriptionSettings( $type, $library_id, $transcribe_enabled );
			}
		}
	}
}
