<?php
/**
 * Bunny Webhook Service.
 *
 * Handles webhook secret generation and management for Bunny.net integrations.
 *
 * @package PrestoPlayer\Pro\Services\Bunny
 */

namespace PrestoPlayer\Pro\Services\Bunny;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Controllers\BunnyVideoLibraryController;
use PrestoPlayer\Pro\Models\Bunny\Video;

/**
 * BunnyWebhookService class.
 */
class BunnyWebhookService {

	/**
	 * Bunny.net webhook status code for caption generation complete.
	 *
	 * @var int
	 */
	const WEBHOOK_STATUS_CAPTIONS_COMPLETE = 9;

	/**
	 * Option key for storing the webhook secret.
	 *
	 * @var string
	 */
	const WEBHOOK_SECRET_KEY = 'presto_bunny_webhook_secret';

	/**
	 * The webhook URL.
	 *
	 * @var string
	 */
	const WEBHOOK_ENDPOINT = '/presto-player/v1/bunny/stream/webhook';

	/**
	 * Video library type ('public' or 'private').
	 *
	 * @var string|null
	 */
	protected $type = null;

	/**
	 * Video library ID.
	 *
	 * @var int|null
	 */
	protected $library_id = null;

	/**
	 * Cached webhook secret.
	 *
	 * @var string|null
	 */
	protected $secret = null;

	/**
	 * Constructor.
	 *
	 * @param string|null $type        Optional. Video library type ('public' or 'private').
	 * @param int|null    $library_id  Optional. Video library ID. If not provided and type is set, will be fetched automatically.
	 */
	public function __construct( $type = null, $library_id = null ) {
		if ( ! empty( $type ) ) {
			$this->type = $type;
			// Fetch library_id if not provided.
			if ( null === $library_id ) {
				$this->library_id = Setting::get( 'bunny_stream_' . $type, 'video_library_id' );
			} else {
				$this->library_id = $library_id;
			}
		}
	}

	/**
	 * Get or generate a webhook secret for Bunny.net integration.
	 *
	 * Checks if a webhook secret already exists in WordPress options.
	 * If it exists, returns the existing secret. If not, generates a new
	 * cryptographically secure secret and stores it.
	 * Uses cached secret if available.
	 *
	 * @return string The webhook secret.
	 */
	public function getOrGenerateSecret() {
		// Return cached secret if available.
		if ( null !== $this->secret ) {
			return $this->secret;
		}

		// Check if secret already exists.
		$existing_secret = get_option( self::WEBHOOK_SECRET_KEY );

		if ( ! empty( $existing_secret ) ) {
			$this->secret = $existing_secret;
			return $this->secret;
		}

		// Generate a new cryptographically secure secret.
		$secret = $this->generateSecureSecret();

		// Store the secret.
		update_option( self::WEBHOOK_SECRET_KEY, $secret );

		// Cache the secret.
		$this->secret = $secret;

		return $this->secret;
	}

	/**
	 * Generate a cryptographically secure random secret.
	 *
	 * Uses PHP's random_bytes() and bin2hex() to produce a 64-character hex
	 * string. Avoids WordPress pluggable functions so this can run during
	 * early boot (e.g. migrations) when wp_hash() may not be loaded.
	 *
	 * @return string The generated secret.
	 */
	public function generateSecureSecret() {
		return bin2hex( random_bytes( 32 ) );
	}

	/**
	 * Get the webhook URL with the secret parameter.
	 *
	 * @param string $base_url The base webhook URL.
	 * @return string The webhook URL with secret parameter.
	 */
	public function getWebhookUrlWithSecret( $base_url ) {
		$secret = $this->getOrGenerateSecret();
		return add_query_arg( 'secret', $secret, $base_url );
	}

	/**
	 * Get the webhook URL for Bunny.net notifications.
	 *
	 * Uses home_url() instead of rest_url() to avoid issues when called early in boot.
	 * Includes the webhook secret as a query parameter.
	 *
	 * @return string The webhook URL with secret parameter.
	 */
	public function getWebhookUrl() {
		// Use home_url() which is available earlier than rest_url().
		// rest_url() requires $wp_rewrite which may not be initialized yet.
		$rest_prefix = rest_get_url_prefix(); // Returns 'wp-json' by default.
		$base_url    = trailingslashit( home_url() ) . $rest_prefix . self::WEBHOOK_ENDPOINT;

		// Include webhook secret in URL.
		return $this->getWebhookUrlWithSecret( $base_url );
	}

	/**
	 * Verify a webhook secret.
	 *
	 * @param string $provided_secret The secret provided in the webhook request.
	 * @return bool True if the secret is valid, false otherwise.
	 */
	public function verifySecret( $provided_secret ) {
		// Use cached secret if available, otherwise fetch from options.
		$stored_secret = null !== $this->secret ? $this->secret : get_option( self::WEBHOOK_SECRET_KEY );
		return ! empty( $stored_secret ) && hash_equals( $stored_secret, $provided_secret );
	}

	/**
	 * Regenerate the webhook secret.
	 *
	 * Generates a new secret and replaces the existing one.
	 * Use with caution as this will break existing webhook integrations.
	 *
	 * @return string The new webhook secret.
	 */
	protected function regenerateSecret() {
		$new_secret = $this->generateSecureSecret();
		update_option( self::WEBHOOK_SECRET_KEY, $new_secret );
		// Update cached secret.
		$this->secret = $new_secret;
		return $this->secret;
	}

	/**
	 * Delete the webhook secret.
	 *
	 * Removes the webhook secret from WordPress options.
	 *
	 * @return bool True if the option was deleted, false otherwise.
	 */
	protected function deleteSecret() {
		return delete_option( self::WEBHOOK_SECRET_KEY );
	}

	/**
	 * Handle webhook notification from Bunny Stream.
	 *
	 * Processes webhook events and delegates to appropriate handlers.
	 * Currently handles caption generation completion events.
	 * Uses class properties for type and library_id set via constructor.
	 *
	 * @param int    $status     The webhook status code.
	 * @param string $video_guid The video GUID.
	 * @return array Response data with 'processed' boolean, 'action' string, and 'library_id'.
	 */
	public function handleWebhook( $status = 0, $video_guid = '' ) {
		$result = array(
			'processed'  => false,
			'action'     => 'ignored',
			'library_id' => $this->library_id ?? 0,
		);

		// Only process caption generation events (Status 9).
		if ( self::WEBHOOK_STATUS_CAPTIONS_COMPLETE === (int) $status ) {
			// Clear caption cache for the video.
			if ( ! empty( $video_guid ) ) {
				$video = new Video( (object) array( 'guid' => $video_guid ), $this->type ?? 'public' );
				$video->captions()->clearCaptionCache();
				$result['processed'] = true;
				$result['action']    = 'cache_cleared';
			}
		}

		return $result;
	}

	/**
	 * Validate and register webhook URL with Bunny.net video library.
	 *
	 * Validates the webhook URL format and registers it with the specified
	 * Bunny.net video library. If no webhook URL is provided, uses the default
	 * generated webhook URL.
	 * Uses class properties for type and library_id set via constructor.
	 *
	 * @param string                           $webhook_url      Optional. Custom webhook URL. If empty, uses default generated URL.
	 * @param BunnyVideoLibraryController|null $library_controller Optional. Library controller instance for dependency injection.
	 * @return true|\WP_Error True on success, WP_Error on failure.
	 */
	public function registerWebhookUrl( $webhook_url = '', $library_controller = null ) {
		// Validate type is set via constructor.
		if ( empty( $this->type ) || ! in_array( $this->type, array( 'public', 'private' ), true ) ) {
			return new \WP_Error(
				'invalid_library_type',
				__( 'Library type must be either "public" or "private".', 'presto-player-pro' ),
				array( 'status' => 400 )
			);
		}

		// Validate library_id is set via constructor.
		if ( empty( $this->library_id ) ) {
			return new \WP_Error(
				'no_library',
				__( 'No Bunny.net video library found. Please connect to Bunny.net first.', 'presto-player-pro' ),
				array( 'status' => 400 )
			);
		}

		// Use custom webhook URL if provided and not empty, otherwise use default.
		$webhook_url_to_use = ! empty( $webhook_url ) && trim( $webhook_url ) !== ''
			? trim( $webhook_url )
			: $this->getWebhookUrl();

		// Validate URL format.
		if ( empty( $webhook_url_to_use ) || ! filter_var( $webhook_url_to_use, FILTER_VALIDATE_URL ) ) {
			return new \WP_Error(
				'invalid_url',
				__( 'Invalid webhook URL format.', 'presto-player-pro' ),
				array( 'status' => 400 )
			);
		}

		// Get library controller instance.
		if ( null === $library_controller ) {
			$library_controller = new BunnyVideoLibraryController();
		}

		// Update the library with webhook URL.
		$result = $library_controller->update(
			$this->type,
			$this->library_id,
			array(
				'WebhookUrl' => $webhook_url_to_use,
			)
		);

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}
}
