<?php
/**
 * Bunny Transcription Service.
 *
 * Handles transcription for Bunny.net videos: initiation, caption retrieval
 * with caching, and cache invalidation. Webhook handling is in BunnyWebhookService.
 *
 * @package PrestoPlayer\Pro\Services\Bunny
 */

namespace PrestoPlayer\Pro\Services\Bunny;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Services\Bunny\BunnyService;
use PrestoPlayer\Pro\Support\AbstractBunnyStreamController;
use PrestoPlayer\Pro\Models\Bunny\Video;
use PrestoPlayer\Pro\Controllers\BunnyVideoLibraryController;

class BunnyTranscriptionService extends AbstractBunnyStreamController {

	/**
	 * Cache key prefix for caption transients.
	 *
	 * @var string
	 */
	const CACHE_PREFIX = 'presto_bunny_captions_';

	/**
	 * Cache duration in seconds (24 hours).
	 *
	 * @var int
	 */
	const CACHE_DURATION = DAY_IN_SECONDS;

	/**
	 * Endpoint for transcription operations (uses videos endpoint).
	 *
	 * @var string
	 */
	protected $endpoint = 'videos';

	/**
	 * Model class (not used for transcription, but required by parent).
	 *
	 * @var string
	 */
	protected $model = Video::class;

	/**
	 * Video instance this service operates on.
	 *
	 * @var Video|null
	 */
	protected $video;

	/*
	|--------------------------------------------------------------------------
	| Constructor & Magic Methods
	|--------------------------------------------------------------------------
	*/

	/**
	 * Constructor.
	 *
	 * @param Video|null $video Video instance to operate on. Optional for utility methods.
	 *
	 * @throws \InvalidArgumentException If video GUID or type is empty.
	 */
	public function __construct( $video = null ) {
		if ( null === $video ) {
			// For utility methods, initialize with empty type.
			parent::__construct( '' );
			return;
		}

		if ( empty( $video->guid ) ) {
			throw new \InvalidArgumentException( 'Video GUID is required' );
		}
		if ( empty( $video->type ) ) {
			throw new \InvalidArgumentException( 'Video type is required' );
		}
		$this->video = $video;
		parent::__construct( $video->type );
	}

	/**
	 * Forward call to method.
	 *
	 * @param string $method Method to call.
	 * @param mixed  $params Method params.
	 * @return mixed
	 */
	public function __call( $method, $params ) {
		return call_user_func_array( array( $this, $method ), $params );
	}

	/**
	 * Static Facade Accessor.
	 *
	 * @param string $method_name Method to call.
	 * @param mixed  $params Method params.
	 * @return mixed
	 */
	public static function __callStatic( $method_name, $params ) {
		$instance = new static( null );
		return call_user_func_array( array( $instance, $method_name ), $params );
	}

	/*
	|--------------------------------------------------------------------------
	| Getters
	|--------------------------------------------------------------------------
	*/

	/**
	 * Get captions for a Bunny video.
	 *
	 * @return array Array of caption objects/arrays.
	 */
	protected function getCaptions() {
		$captions = $this->getCaptionsFromCache();

		if ( null !== $captions ) {
			return $captions;
		}

		$captions = $this->fetchCaptionsFromApi();
		$this->cacheCaptions( $captions );

		return $captions;
	}

	/**
	 * Get captions from cache.
	 *
	 * @return array|null Cached captions array, or null if not cached.
	 */
	private function getCaptionsFromCache() {
		$cached = get_transient( $this->getCacheKey() );

		if ( false === $cached ) {
			return null;
		}

		return $cached;
	}

	/**
	 * Get the cache key for this video's captions.
	 *
	 * @return string The cache key used for storing/retrieving caption transients.
	 */
	protected function getCacheKey() {
		return self::CACHE_PREFIX . $this->video->guid;
	}

	/**
	 * Check if transcription is enabled in settings.
	 *
	 * @return bool True if transcription is enabled.
	 */
	protected function isTranscriptionEnabled() {
		return (bool) Setting::get( 'bunny_stream', 'transcribe_enabled', false );
	}

	/**
	 * Get configured transcription languages.
	 *
	 * @return array Array of language codes.
	 */
	protected function getTranscriptionLanguages() {
		$languages = Setting::get( 'bunny_stream', 'transcribe_languages', array() );
		return is_array( $languages ) ? $languages : array();
	}

	/**
	 * Get the pullzone URL from video or settings.
	 *
	 * Resolves pullzone URL in order of preference:
	 * 1. Video instance pullzone_url (if available)
	 * 2. Settings based on video type
	 *
	 * @return string|null The pullzone URL or null if not found.
	 */
	protected function getPullzoneUrl() {
		if ( $this->video && ! empty( $this->video->pullzone_url ) ) {
			return $this->video->pullzone_url;
		}

		if ( ! empty( $this->type ) ) {
			return Setting::get( 'bunny_stream_' . $this->type, 'pull_zone_url' );
		}

		return null;
	}

	/**
	 * Format captions for player consumption.
	 *
	 * @param array       $captions     Array of caption metadata objects.
	 * @param string|null $video_guid   The Bunny video GUID. Falls back to instance video if null.
	 * @param string|null $pullzone_url Pull zone URL. Falls back to instance data or settings if null.
	 * @return array Array of formatted caption objects.
	 */
	protected function formatCaptionsForPlayer( $captions, $video_guid = null, $pullzone_url = null ) {
		$formatted = array();

		// Use instance data if available, fall back to explicit parameters.
		$video_guid   = $video_guid ?? ( $this->video ? $this->video->guid : null );
		$pullzone_url = $pullzone_url ?? $this->getPullzoneUrl();

		// Validate and sanitize inputs.
		$video_guid   = sanitize_text_field( $video_guid ?? '' );
		$pullzone_url = sanitize_text_field( $pullzone_url ?? '' );

		if ( empty( $video_guid ) || strpos( $video_guid, '..' ) !== false || strpos( $video_guid, '/' ) !== false ) {
			return array();
		}

		$video_guid = trim( $video_guid );
		if ( empty( $video_guid ) ) {
			return array();
		}

		if ( empty( $pullzone_url ) || strpos( $pullzone_url, '..' ) !== false || strpos( $pullzone_url, '/' ) !== false ) {
			return array();
		}

		foreach ( $captions as $caption ) {
			if ( ! isset( $caption->srclang ) ) {
				continue;
			}

			$srclang = preg_replace( '/[^a-z0-9\-]/i', '', trim( (string) $caption->srclang ) );

			if ( empty( $srclang ) || strpos( $srclang, '..' ) !== false ) {
				continue;
			}

			$caption_url = sprintf(
				'https://%s/%s/captions/%s.vtt',
				$pullzone_url,
				rawurlencode( $video_guid ),
				rawurlencode( $srclang )
			);

			if ( ! wp_http_validate_url( $caption_url ) ) {
				continue;
			}

			$formatted[] = apply_filters(
				'presto_player_bunny_caption_formatted',
				array(
					'label'   => isset( $caption->label ) ? sanitize_text_field( $caption->label ) : $srclang,
					'src'     => esc_url( $caption_url ),
					'srcLang' => $srclang,
					'version' => isset( $caption->version ) ? absint( $caption->version ) : null,
					'source'  => 'bunny',
				),
				$caption
			);
		}

		return $formatted;
	}

	/*
	|--------------------------------------------------------------------------
	| API / Fetch Methods
	|--------------------------------------------------------------------------
	*/

	/**
	 * Fetch captions from Bunny.net API.
	 *
	 * Retrieves video data and extracts caption metadata, then formats
	 * it for player consumption with proper URLs.
	 *
	 * @return array Array of caption objects.
	 */
	protected function fetchCaptionsFromApi() {
		$video_data = $this->get( $this->video->guid );

		if ( is_wp_error( $video_data ) || empty( $video_data->captions ) ) {
			return array();
		}

		return $this->formatCaptionsForPlayer( $video_data->captions );
	}

	/**
	 * Transcribe a video.
	 *
	 * Initiates transcription for the video with specified target languages.
	 * Uses 'force=true' to regenerate captions even if they already exist.
	 *
	 * @param array $target_languages Array of language codes.
	 * @return object|\WP_Error The response or error
	 */
	protected function transcribe( $target_languages = array() ) {
		if ( empty( $target_languages ) || ! is_array( $target_languages ) ) {
			return new \WP_Error( 'missing_languages', 'Target languages are required' );
		}

		if ( empty( $this->library_id ) ) {
			return new \WP_Error( 'missing_library_id', 'Library ID is not configured' );
		}

		if ( empty( $this->client ) ) {
			return new \WP_Error( 'missing_api_key', 'API key is not configured' );
		}

		$endpoint = "library/{$this->library_id}/videos/{$this->video->guid}/transcribe";

		return $this->client->post(
			$endpoint,
			array(
				'query' => array( 'force' => 'true' ),
				'body'  => array(
					'targetLanguages' => $target_languages,
				),
			)
		);
	}

	/*
	|--------------------------------------------------------------------------
	| Cache / Clear / Delete Methods
	|--------------------------------------------------------------------------
	*/

	/**
	 * Cache captions.
	 *
	 * Caches captions for 24 hours. Users can manually clear the cache
	 * via the clear cache button when captions become available.
	 *
	 * @param array $captions Captions to cache.
	 */
	private function cacheCaptions( array $captions ) {
		set_transient( $this->getCacheKey(), $captions, self::CACHE_DURATION );
	}

	/**
	 * Clear caption cache for a video.
	 *
	 * @return bool True if the transient was deleted, false otherwise.
	 */
	protected function clearCaptionCache() {
		$transient_key = self::CACHE_PREFIX . $this->video->guid;
		return delete_transient( $transient_key );
	}

	/**
	 * Clear all caption cache transients.
	 *
	 * @return int Number of transients cleared.
	 */
	protected function clearAllCaptionsCache() {
		global $wpdb;

		$deleted = 0;

		$transient_names = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT REPLACE(option_name, '_transient_', '') FROM {$wpdb->options} WHERE option_name LIKE %s",
				$wpdb->esc_like( '_transient_' . self::CACHE_PREFIX ) . '%'
			)
		);

		foreach ( $transient_names as $transient_name ) {
			if ( delete_transient( $transient_name ) ) {
				++$deleted;
			}
		}

		return $deleted;
	}

	/**
	 * Delete a caption from a video.
	 *
	 * @param string $source_language Source language code (e.g., 'en', 'fr').
	 * @return true|\WP_Error True on success, WP_Error on failure.
	 */
	protected function deleteCaption( $source_language = '' ) {
		if ( empty( $source_language ) ) {
			return new \WP_Error( 'missing_source_language', 'Source language is required' );
		}

		if ( empty( $this->library_id ) ) {
			return new \WP_Error( 'missing_library_id', 'Library ID is not configured' );
		}

		if ( empty( $this->api_key ) || ! $this->client ) {
			return new \WP_Error( 'missing_api_key', 'API key is not configured' );
		}

		$endpoint = sprintf(
			'library/%s/videos/%s/captions/%s',
			rawurlencode( $this->library_id ),
			rawurlencode( $this->video->guid ),
			rawurlencode( $source_language )
		);
		$result   = $this->client->delete( $endpoint );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}

	/**
	 * Update library transcription settings to match the current transcribe setting.
	 *
	 * @param string $type               Video library type ('public' or 'private').
	 * @param int    $library_id         Video library ID.
	 * @param bool   $transcribe_enabled Whether transcription is enabled.
	 * @return \WP_Error|true True on success, WP_Error on failure.
	 */
	protected function updateLibraryTranscriptionSettings( $type, $library_id, $transcribe_enabled ) {
		if ( ! in_array( $type, array( 'public', 'private' ), true ) ) {
			return new \WP_Error( 'invalid_library_type', 'Library must be either public or private' );
		}

		if ( empty( $library_id ) || ! is_numeric( $library_id ) ) {
			return new \WP_Error( 'invalid_library_id', 'Library ID is invalid or not configured' );
		}

		$args = array(
			'EnableTranscribing'           => (bool) $transcribe_enabled,
			'TranscribingCaptionLanguages' => $transcribe_enabled ? $this->getTranscriptionLanguages() : array(),
		);

		if ( $transcribe_enabled ) {
			$args['KeepOriginalFiles'] = true;
		}

		$library_controller = new BunnyVideoLibraryController();
		$result             = $library_controller->update( $type, $library_id, $args );

		if ( is_wp_error( $result ) ) {
			return $result;
		}

		return true;
	}
}
