<?php
/**
 * REST API controller for Bunny.net webhook endpoints.
 *
 * @package PrestoPlayer\Pro\Services\API
 */

namespace PrestoPlayer\Pro\Services\API;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Services\Bunny\BunnyWebhookService;
use PrestoPlayer\Pro\Support\Utility;

/**
 * REST API controller for Bunny.net webhook endpoints.
 *
 * Handles webhook registration, URL retrieval, and webhook processing
 * for Bunny Stream transcription notifications.
 */
class RestBunnyWebhookController {

	/**
	 * REST API namespace
	 *
	 * @var string
	 */
	protected $namespace = 'presto-player';

	/**
	 * REST API version
	 *
	 * @var string
	 */
	protected $version = 'v1';

	/**
	 * REST API base route
	 *
	 * @var string
	 */
	protected $base = 'bunny/stream/webhook';

	/**
	 * Register controller
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Static Facade Accessor.
	 *
	 * @param string $method_name Method to call.
	 * @param mixed  $params Method params.
	 * @return mixed
	 */
	public static function __callStatic( $method_name, $params ) {
		$instance = new static();
		return call_user_func_array( array( $instance, $method_name ), $params );
	}

	/**
	 * Get the base endpoint path for webhook routes
	 *
	 * @return string
	 */
	protected function get_base_path() {
		return "{$this->namespace}/{$this->version}/{$this->base}";
	}

	/**
	 * Get the webhook endpoint path
	 *
	 * @return string
	 */
	public static function get_webhook_path() {
		$instance = new static();
		return $instance->get_base_path();
	}

	/**
	 * Get the webhook URL endpoint path
	 *
	 * @return string
	 */
	public static function get_webhook_url_path() {
		$instance = new static();
		return $instance->get_base_path() . '/url';
	}

	/**
	 * Get the register webhook endpoint path
	 *
	 * @return string
	 */
	public static function get_register_webhook_path() {
		$instance = new static();
		return $instance->get_base_path() . '/register';
	}

	/**
	 * Register webhook routes
	 *
	 * @return void
	 */
	public function register_routes() {
		/**
		 * Webhook endpoint for Bunny Stream transcription notifications
		 */
		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base,
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'handle_webhook' ),
					'permission_callback' => array( $this, 'webhook_permissions_check' ),
					'args'                => array(
						'VideoLibraryId' => array(
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
							'validate_callback' => function ( $value ) {
								return is_numeric( $value ) && intval( $value ) > 0;
							},
						),
						'VideoGuid'      => array(
							'type'              => 'string',
							'sanitize_callback' => array( Utility::class, 'sanitizeGuid' ),
							'validate_callback' => array( Utility::class, 'validateGuidFormat' ),
						),
						'Status'         => array(
							'type'              => 'integer',
							'sanitize_callback' => 'absint',
							'validate_callback' => function ( $value ) {
								// Webhook status might be 0, so just check numeric.
								return is_numeric( $value );
							},
						),
					),
				),
			)
		);

		/**
		 * Get default webhook URL
		 */
		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/url',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_webhook_url' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
				),
			)
		);

		/**
		 * Register webhook URL with Bunny.net
		 */
		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/register',
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'register_webhook' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
					'args'                => array(
						'type'        => array(
							'type'     => 'string',
							'required' => true,
							'enum'     => array( 'public', 'private' ),
						),
						'webhook_url' => array(
							'type'              => 'string',
							'sanitize_callback' => 'esc_url_raw',
						),
					),
				),
			)
		);
	}

	/**
	 * Check if user has permission to transcribe videos
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return bool|\WP_Error
	 */
	public function transcribe_permissions_check( \WP_REST_Request $request ) {
		if ( ! is_user_logged_in() ) {
			return new \WP_Error(
				'rest_not_logged_in',
				__( 'You must be logged in to perform this action.', 'presto-player' ),
				array( 'status' => 401 )
			);
		}

		$has_permission = current_user_can( 'upload_files' ) && current_user_can( 'edit_posts' );
		$has_permission = apply_filters( 'presto_player_transcribe_capability', $has_permission, $request );

		if ( ! $has_permission ) {
			return new \WP_Error(
				'rest_forbidden',
				__( 'You do not have permission to generate automatic captions for videos.', 'presto-player' ),
				array( 'status' => 403 )
			);
		}

		return true;
	}

	/**
	 * Verify webhook secret for Bunny Stream webhook endpoint
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return bool|\WP_Error
	 */
	public function webhook_permissions_check( \WP_REST_Request $request ) {
		// Check for webhook secret in query parameters.
		$provided_secret = $request->get_param( 'secret' );

		// If no secret provided, return error.
		if ( empty( $provided_secret ) ) {
			return new \WP_Error( 'missing_secret', 'Missing secret parameter', array( 'status' => 400 ) );
		}

		// Verify the webhook secret.
		$service = new BunnyWebhookService();
		if ( ! $service->verifySecret( $provided_secret ) ) {
			return new \WP_Error( 'invalid_secret', 'Invalid secret', array( 'status' => 403 ) );
		}

		return true;
	}

	/**
	 * Factory method for getting webhook URL.
	 * Wraps BunnyWebhookService::getWebhookUrl() for testability.
	 *
	 * @return string
	 */
	protected function getWebhookUrlFromService() {
		$service = new BunnyWebhookService();
		return $service->getWebhookUrl();
	}

	/**
	 * Factory method for registering webhook URL.
	 * Wraps BunnyWebhookService::registerWebhookUrl() for testability.
	 *
	 * @param string $type        Video library type ('public' or 'private').
	 * @param string $webhook_url Optional. Custom webhook URL.
	 * @return true|\WP_Error
	 */
	protected function registerWebhookUrlViaService( $type, $webhook_url = '' ) {
		// Create service instance with type to use class properties.
		$service = new BunnyWebhookService( $type );
		return $service->registerWebhookUrl( $webhook_url );
	}

	/**
	 * Factory method for handling webhook.
	 * Wraps BunnyWebhookService::handleWebhook() for testability.
	 *
	 * @param int    $status           The webhook status code.
	 * @param string $video_guid       The video GUID.
	 * @param string $type             Video type ('public' or 'private').
	 * @param int    $video_library_id The video library ID.
	 * @return array
	 */
	protected function handleWebhookViaService( $status, $video_guid, $type, $video_library_id ) {
		// Create service instance with type and library_id for this specific webhook.
		$service = new BunnyWebhookService( $type, $video_library_id );
		return $service->handleWebhook( $status, $video_guid );
	}

	/**
	 * Handle webhook notifications from Bunny Stream for transcriptions
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_REST_Response
	 */
	public function handle_webhook( \WP_REST_Request $request ) {
		try {
			// Process the webhook payload.
			$video_guid       = $request->get_param( 'VideoGuid' );
			$status           = $request->get_param( 'Status' );
			$video_library_id = $request->get_param( 'VideoLibraryId' );

			// Determine type based on matching library_id, default to 'public'.
			$type = ( (string) Setting::get( 'bunny_stream_private', 'video_library_id' ) === (string) $video_library_id ) ? 'private' : 'public';

			// Handle webhook through webhook service.
			$result = $this->handleWebhookViaService( $status, $video_guid, $type, $video_library_id );

			// Return success to acknowledge receipt.
			if ( $result['processed'] ) {
				return new \WP_REST_Response( array( 'status' => 'processed' ), 200 );
			} else {
				return new \WP_REST_Response( array( 'status' => 'ignored' ), 200 );
			}
		} catch ( \Exception $e ) {
			return new \WP_Error( 'internal_server_error', 'Internal server error', array( 'status' => 500 ) );
		}
	}

	/**
	 * Get default webhook URL.
	 *
	 * @param \WP_REST_Request $request Full data about the request. Unused but required by REST API.
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function get_webhook_url( \WP_REST_Request $request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		try {
			$default_url = $this->getWebhookUrlFromService();
			return new \WP_REST_Response(
				array(
					'url' => $default_url,
				),
				200
			);
		} catch ( \Exception $e ) {
			return new \WP_Error( 'webhook_error', $e->getMessage(), array( 'status' => 500 ) );
		}
	}

	/**
	 * Register webhook URL with Bunny.net video library.
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_REST_Response|\WP_Error
	 */
	public function register_webhook( \WP_REST_Request $request ) {
		try {
			$type        = $request->get_param( 'type' );
			$webhook_url = $request->get_param( 'webhook_url' );

			// Use the service method to validate and register the webhook.
			$result = $this->registerWebhookUrlViaService( $type, $webhook_url );

			if ( is_wp_error( $result ) ) {
				return $result;
			}

			return new \WP_REST_Response(
				array(
					'success' => true,
					'message' => __( 'Webhook registered successfully.', 'presto-player-pro' ),
				),
				200
			);

		} catch ( \Exception $e ) {
			return new \WP_Error( 'webhook_error', $e->getMessage(), array( 'status' => 500 ) );
		}
	}
}
