<?php

namespace PrestoPlayer\Pro\Services\API;

use PrestoPlayer\Models\Setting;
use PrestoPlayer\Pro\Models\Bunny\Video;
use PrestoPlayer\Pro\Controllers\BunnyVideoLibraryController;
use PrestoPlayer\Pro\Services\Bunny\BunnyTranscriptionService;
use PrestoPlayer\Pro\Support\Utility;

class RestBunnyTranscriptionController {

	/**
	 * REST API namespace
	 *
	 * @var string
	 */
	protected $namespace = 'presto-player';

	/**
	 * REST API version
	 *
	 * @var string
	 */
	protected $version = 'v1';

	/**
	 * REST API base route
	 *
	 * @var string
	 */
	protected $base = 'bunny/stream/transcription';

	/**
	 * Valid visibility types
	 *
	 * @var array
	 */
	const VISIBILITY_TYPES = array( 'public', 'private' );

	/**
	 * Register controller
	 *
	 * @return void
	 */
	public function register() {
		add_action( 'rest_api_init', array( $this, 'register_routes' ) );
	}

	/**
	 * Static Facade Accessor.
	 *
	 * @param string $method_name Method to call.
	 * @param mixed  $params Method params.
	 * @return mixed
	 */
	public static function __callStatic( $method_name, $params ) {
		$instance = new static();
		return call_user_func_array( array( $instance, $method_name ), $params );
	}

	/**
	 * Get the base endpoint path for transcription routes
	 *
	 * @return string
	 */
	protected function get_base_path() {
		return "{$this->namespace}/{$this->version}/{$this->base}";
	}

	/**
	 * Get the captions endpoint path
	 *
	 * @return string
	 */
	protected function get_captions_path() {
		return $this->get_base_path() . '/captions';
	}

	/**
	 * Get the transcribe endpoint path
	 *
	 * @return string
	 */
	protected function get_transcribe_path() {
		return $this->get_base_path() . '/transcribe';
	}

	/**
	 * Get the delete caption endpoint base path
	 *
	 * @return string
	 */
	protected function get_delete_caption_path() {
		return $this->get_base_path() . '/captions';
	}

	/**
	 * Get the clear cache endpoint path
	 *
	 * @return string
	 */
	protected function get_clear_cache_path() {
		return $this->get_base_path() . '/captions/clear-cache';
	}

	/**
	 * Register transcription routes
	 *
	 * @return void
	 */
	public function register_routes() {
		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/transcribe',
			array(
				array(
					'methods'             => \WP_REST_Server::CREATABLE,
					'callback'            => array( $this, 'transcribe_video' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
					'args'                => array(
						'guid'            => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => array( Utility::class, 'sanitizeGuid' ),
							'validate_callback' => array( Utility::class, 'validateGuidFormat' ),
						),
						'type'            => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => 'sanitize_text_field',
							'validate_callback' => array( $this, 'validate_visibility_type' ),
						),
						'targetLanguages' => array(
							'type'              => 'array',
							'required'          => true,
							'sanitize_callback' => array( $this, 'sanitize_target_languages' ),
							'validate_callback' => array( $this, 'validate_target_languages' ),
						),
					),
				),
			)
		);

		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/captions',
			array(
				array(
					'methods'             => \WP_REST_Server::READABLE,
					'callback'            => array( $this, 'get_captions' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
					'args'                => array(
						'guid' => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => array( Utility::class, 'sanitizeGuid' ),
							'validate_callback' => array( Utility::class, 'validateGuidFormat' ),
						),
						'type' => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => 'sanitize_text_field',
							'validate_callback' => array( $this, 'validate_visibility_type' ),
						),
					),
				),
			)
		);

		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/captions/(?P<guid>[\S]+)/(?P<language>[\S]+)',
			array(
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'delete_caption' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
					'args'                => array(
						'guid'     => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => array( Utility::class, 'sanitizeGuid' ),
							'validate_callback' => array( Utility::class, 'validateGuidFormat' ),
						),
						'language' => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => array( $this, 'sanitize_language' ),
							'validate_callback' => array( $this, 'validate_language' ),
						),
						'type'     => array(
							'type'              => 'string',
							'required'          => true,
							'sanitize_callback' => 'sanitize_text_field',
							'validate_callback' => array( $this, 'validate_visibility_type' ),
						),
					),
				),
			)
		);

		// Route to clear all caption cache.
		register_rest_route(
			"{$this->namespace}/{$this->version}",
			'/' . $this->base . '/captions/clear-cache',
			array(
				array(
					'methods'             => \WP_REST_Server::DELETABLE,
					'callback'            => array( $this, 'clear_all_caption_cache' ),
					'permission_callback' => array( $this, 'transcribe_permissions_check' ),
				),
			)
		);
	}

	/**
	 * Check if user has permission to transcribe videos
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return bool|\WP_Error
	 */
	public function transcribe_permissions_check( \WP_REST_Request $request ) {
		if ( ! is_user_logged_in() ) {
			return new \WP_Error(
				'rest_not_logged_in',
				__( 'You must be logged in to perform this action.', 'presto-player' ),
				array( 'status' => 401 )
			);
		}

		$has_permission = current_user_can( 'upload_files' ) && current_user_can( 'edit_posts' );
		$has_permission = apply_filters( 'presto_player_transcribe_capability', $has_permission, $request );

		if ( ! $has_permission ) {
			return new \WP_Error(
				'rest_forbidden',
				__( 'You do not have permission to generate automatic captions for videos.', 'presto-player' ),
				array( 'status' => 403 )
			);
		}

		return true;
	}

	/**
	 * Validate visibility type parameter
	 *
	 * @param string $value The visibility type value to validate.
	 * @return bool
	 */
	public function validate_visibility_type( $value ) {
		return in_array( $value, self::VISIBILITY_TYPES, true );
	}

	/**
	 * Sanitize a single language code
	 *
	 * Language codes are typically 2-3 lowercase alphabetic characters,
	 * but Bunny.net auto-generated captions use format like 'en-auto'.
	 * This method ensures only valid characters are allowed.
	 *
	 * @param string $value The language code to sanitize.
	 * @return string
	 */
	public function sanitize_language( $value ) {
		if ( ! is_string( $value ) ) {
			return '';
		}
		// Allow alphabetic characters and hyphens (for auto-generated like 'en-auto'), then lowercase
		return strtolower( preg_replace( '/[^a-zA-Z\-]/', '', $value ) );
	}

	/**
	 * Sanitize target languages array
	 *
	 * @param mixed $value The target languages value to sanitize.
	 * @return array
	 */
	public function sanitize_target_languages( $value ) {
		if ( ! is_array( $value ) ) {
			return array();
		}
		return array_map( array( $this, 'sanitize_language' ), $value );
	}

	/**
	 * Validate target languages array
	 *
	 * Validates that all provided language codes have a valid format (2-3 lowercase letters).
	 * Actual language support is validated by Bunny.net API.
	 *
	 * @param mixed $value The target languages value to validate.
	 * @return bool
	 */
	public function validate_target_languages( $value ) {
		if ( ! is_array( $value ) || empty( $value ) ) {
			return false;
		}

		foreach ( $value as $lang ) {
			if ( ! is_string( $lang ) || empty( $lang ) ) {
				return false;
			}

			// Validate format: 2-3 lowercase letters (ISO 639 language codes)
			if ( ! preg_match( '/^[a-z]{2,3}$/', $lang ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Validate a single language code for delete operations
	 *
	 * Allows both standard language codes (en, es) and auto-generated
	 * caption codes from Bunny.net (en-auto, es-auto).
	 * Only validates format; actual language support is validated by Bunny.net API.
	 *
	 * @param string $value The language code to validate.
	 * @return bool
	 */
	public function validate_language( $value ) {
		if ( ! is_string( $value ) || empty( $value ) ) {
			return false;
		}

		// Validate standard language code format (2-3 lowercase letters)
		if ( preg_match( '/^[a-z]{2,3}$/', $value ) ) {
			return true;
		}

		// Validate auto-generated caption format (e.g., 'en-auto')
		if ( preg_match( '/^[a-z]{2,3}-auto$/', $value ) ) {
			return true;
		}

		return false;
	}

	/**
	 * Transcribe video
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function transcribe_video( \WP_REST_Request $request ) {
		try {
			$guid             = $request->get_param( 'guid' );
			$type             = $request->get_param( 'type' );
			$target_languages = $request->get_param( 'targetLanguages' );

			// Ensure KeepOriginalFiles is enabled on Bunny Stream for on-demand automatic caption generation.
			$keep_original_files = $this->ensure_keep_original_files( $type );
			if ( is_wp_error( $keep_original_files ) ) {
				return new \WP_Error(
					'keep_original_files_error',
					__( 'Failed to ensure KeepOriginalFiles is enabled for on-demand automatic caption generation.', 'presto-player' ),
					array( 'status' => 500 )
				);
			}

			$video       = $this->makeVideo( (object) array( 'guid' => $guid ), $type );
			$transcribed = $video->captions()->transcribe( $target_languages );

			if ( is_wp_error( $transcribed ) ) {
				return $transcribed;
			}

			return rest_ensure_response( $transcribed );
		} catch ( \Exception $e ) {
			return new \WP_Error(
				'transcribe_error',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}

	/**
	 * Get captions for a Bunny video
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function get_captions( \WP_REST_Request $request ) {
		try {
			$guid       = $request->get_param( 'guid' );
			$visibility = $request->get_param( 'type' );

			$video  = $this->makeVideo( (object) array( 'guid' => $guid ), $visibility );
			$tracks = $video->captions()->getCaptions();

			return rest_ensure_response(
				array(
					'tracks' => $tracks,
				)
			);
		} catch ( \Exception $e ) {
			return new \WP_Error(
				'captions_error',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}

	/**
	 * Delete a caption from a Bunny video
	 *
	 * @param \WP_REST_Request $request Full data about the request.
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function delete_caption( \WP_REST_Request $request ) {
		try {
			$guid       = $request->get_param( 'guid' );
			$language   = $request->get_param( 'language' );
			$visibility = $request->get_param( 'type' );

			$video   = $this->makeVideo( (object) array( 'guid' => $guid ), $visibility );
			$deleted = $video->captions()->deleteCaption( $language );

			if ( is_wp_error( $deleted ) ) {
				return $deleted;
			}

			$video->captions()->clearCaptionCache();

			return rest_ensure_response(
				array(
					'success' => true,
				)
			);
		} catch ( \Exception $e ) {
			return new \WP_Error(
				'delete_caption_error',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}

	/**
	 * Clear all caption cache transients
	 *
	 * @param \WP_REST_Request $request Full data about the request. Unused but required by REST API.
	 * @return \WP_Error|\WP_REST_Response
	 */
	public function clear_all_caption_cache( \WP_REST_Request $request ) { // phpcs:ignore Generic.CodeAnalysis.UnusedFunctionParameter.Found
		try {
			$deleted = BunnyTranscriptionService::clearAllCaptionsCache();

			return rest_ensure_response(
				array(
					'success' => true,
					'cleared' => $deleted,
				)
			);
		} catch ( \Exception $e ) {
			return new \WP_Error(
				'clear_cache_error',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}

	/**
	 * Factory method for creating a library controller instance.
	 *
	 * @return BunnyVideoLibraryController
	 */
	protected function makeLibraryController() {
		return new BunnyVideoLibraryController();
	}

	/**
	 * Factory method for creating a Video instance.
	 *
	 * @param object $video_data Video data object.
	 * @param string $type Video type (public or private).
	 * @return Video
	 */
	protected function makeVideo( $video_data, $type ) {
		return new Video( $video_data, $type );
	}

	/**
	 * Ensure KeepOriginalFiles is enabled for on-demand transcription
	 *
	 * @param string $type The type of video (public or private).
	 * @return \WP_Error|Object Response from the library controller update or WP_Error on failure.
	 */
	public function ensure_keep_original_files( string $type ) {
		try {
			$library_id = Setting::get( 'bunny_stream_' . $type, 'video_library_id' );
			if ( ! $library_id || ! is_numeric( $library_id ) ) {
				return new \WP_Error(
					'library_id_not_found',
					__( 'Library ID not found.', 'presto-player' ),
					array( 'status' => 404 )
				);
			}

			// Update the library to ensure KeepOriginalFiles is enabled.
			$library_controller = $this->makeLibraryController();
			return $library_controller->update( $type, $library_id, array( 'KeepOriginalFiles' => true ) );
		} catch ( \Exception $e ) {
			return new \WP_Error(
				'keep_original_files_error',
				$e->getMessage(),
				array( 'status' => 500 )
			);
		}
	}
}
