<?php

namespace PrestoPlayer\Pro\Models;

use PrestoPlayer\Models\Model;
use PrestoPlayer\Models\Video;
use PrestoPlayer\Support\Utility;

class Visit extends Model {



	protected $table = 'presto_player_visits';

	/**
	 * Table name with prefix for custom queries
	 *
	 * @var string
	 */
	protected $prefixed_table;

	/**
	 * Constructor
	 */
	public function __construct( $id = 0 ) {
		global $wpdb;
		$this->prefixed_table = $wpdb->prefix . $this->table;
		parent::__construct( $id );
	}

	/**
	 * Model Schema
	 *
	 * @var array
	 */
	public function schema() {
		return array(
			'id'         => array(
				'type' => 'integer',
			),
			'user_id'    => array(
				'type'    => 'integer',
				'default' => get_current_user_id(),
			),
			'duration'   => array(
				'type' => 'integer',
			),
			'video_id'   => array(
				'type' => 'integer',
			),
			'ip_address' => array(
				'type'              => 'string',
				'sanitize_callback' => function ( $ip ) {
					return filter_var( $ip, FILTER_VALIDATE_IP );
				},
				'default'           => Utility::getIPAddress(),
			),
			'created_at' => array(
				'type' => 'string',
			),
			'updated_at' => array(
				'type' => 'string',
			),
			'deleted_at' => array(
				'type' => 'string',
			),
		);
	}

	/**
	 * Queryable attributes
	 *
	 * @var array
	 */
	protected $queryable = array( 'id', 'user_id', 'duration', 'video_id', 'ip_address' );

	/**
	 * Top videos
	 *
	 * @param array $args
	 * @return array
	 */
	public function topVideos( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'duration' => false,
				'page'     => 1,
				'per_page' => 10,
				'start'    => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'      => current_time( 'mysql' ),
			)
		);

		$limit  = (int) $args['per_page'];
		$offset = (int) ( $args['per_page'] * ( $args['page'] - 1 ) );

		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		// Build query based on whether user_id filter is set.
		if ( isset( $args['user_id'] ) ) {
			$results = $this->getTopVideosByUser( $args, $limit, $offset );
			$total   = $this->getTopVideosTotalByUser( $args );
		} else {
			$results = $this->getTopVideos( $args, $limit, $offset );
			$total   = $this->getTopVideosTotal( $args );
		}

		$output = array();
		foreach ( $results as $data ) {
			$output[] = array(
				'video' => ( new Video() )->get( $data->id )->toObject(),
				'stats' => array(
					array(
						'id'    => 'views_count',
						'title' => __( 'Views', 'presto-player-pro' ),
						/* translators: %d: visit count */
						'data'  => sprintf( __( '%d views', 'presto-player-pro' ), (int) $data->visit_count ),
					),
					array(
						'id'        => 'average_duration',
						'title'     => __( 'Avg. View Time', 'presto-player-pro' ),
						'className' => 'presto-badge',
						'data'      => (int) $data->average_duration ? Utility::human_readable_duration( gmdate( 'H:i:s', $data->average_duration ) ) : __( '0 seconds' ),
					),
				),
			);
		}

		return array(
			'total' => (int) $total,
			'data'  => $output,
		);
	}

	/**
	 * Get top videos watched by a specific user
	 *
	 * @param array $args Contains user_id, date range filters
	 * @param int   $limit Number of results per page
	 * @param int   $offset Pagination offset
	 * @return array Video stats with view count and average duration
	 */
	public function getTopVideosByUser( $args, $limit, $offset ) {
		global $wpdb;

		return $wpdb->get_results(
			$wpdb->prepare(
				'SELECT video_id AS id,
					COUNT(video_id) AS visit_count,
					AVG(duration) AS average_duration
				FROM %i
				WHERE (created_at >= %s AND created_at <= %s)
				AND user_id = %d
				AND duration > 0
				GROUP BY video_id
				ORDER BY visit_count DESC
				LIMIT %d OFFSET %d',
				$this->prefixed_table,
				$args['start'],
				$args['end'],
				$args['user_id'],
				$limit,
				$offset
			)
		);
	}

	/**
	 * Get top videos total count filtered by user ID
	 *
	 * @param array $args
	 * @return int
	 */
	public function getTopVideosTotalByUser( $args ) {
		global $wpdb;

		return $wpdb->get_var(
			$wpdb->prepare(
				'SELECT COUNT(*) GroupAmountTimes FROM
					(
						SELECT video_id AS id,
						COUNT(video_id) AS visit_count
						FROM %i
						WHERE (created_at >= %s AND created_at <= %s)
						AND user_id = %d
						AND duration > 0
						GROUP BY video_id
						ORDER BY visit_count DESC
					) AS SubQuery',
				$this->prefixed_table,
				$args['start'],
				$args['end'],
				$args['user_id']
			)
		);
	}

	/**
	 * Get top videos results without user ID filter
	 *
	 * @param array $args
	 * @param int   $limit
	 * @param int   $offset
	 * @return array
	 */
	public function getTopVideos( $args, $limit, $offset ) {
		global $wpdb;

		return $wpdb->get_results(
			$wpdb->prepare(
				'SELECT video_id AS id,
					COUNT(video_id) AS visit_count,
					AVG(duration) AS average_duration
				FROM %i
				WHERE (created_at >= %s AND created_at <= %s)
				GROUP BY video_id
				ORDER BY visit_count DESC
				LIMIT %d OFFSET %d',
				$this->prefixed_table,
				$args['start'],
				$args['end'],
				$limit,
				$offset
			)
		);
	}

	/**
	 * Get top videos total count without user ID filter
	 *
	 * @param array $args
	 * @return int
	 */
	public function getTopVideosTotal( $args ) {
		global $wpdb;

		return $wpdb->get_var(
			$wpdb->prepare(
				'SELECT COUNT(*) GroupAmountTimes FROM
					(
						SELECT video_id AS id,
						COUNT(video_id) AS visit_count
						FROM %i
						WHERE (created_at >= %s AND created_at <= %s)
						GROUP BY video_id
						ORDER BY visit_count DESC
					) AS SubQuery',
				$this->prefixed_table,
				$args['start'],
				$args['end']
			)
		);
	}

	public function topUsers( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'page'     => 1,
				'per_page' => 10,
				'start'    => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'      => current_time( 'mysql' ),
			)
		);

		$limit         = (int) $args['per_page'];
		$offset        = (int) ( $args['per_page'] * ( $args['page'] - 1 ) );
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		$output = array();
		$total  = 0;

		$results = $wpdb->get_results(
			$wpdb->prepare(
				'
				SELECT user_id, COUNT(user_id) AS visit_count, AVG(duration) AS average_duration 
				FROM %i
				WHERE (created_at BETWEEN %s AND %s) 
				  AND duration > 0
				  AND user_id IS NOT NULL
				GROUP BY user_id
				ORDER BY COUNT(user_id) DESC
				LIMIT %d OFFSET %d
				',
				$this->prefixed_table,
				$args['start'],
				$args['end'],
				$limit,
				$offset
			)
		);

		if ( count( $results ) != 0 ) {
			$output = $results;
			$total  = $wpdb->get_var(
				$wpdb->prepare(
					'SELECT COUNT(*) total FROM (
                            SELECT COUNT(id) FROM
                            %i
                            WHERE (created_at BETWEEN %s AND %s) AND duration > 0
                            AND user_id IS NOT NULL
                            GROUP BY user_id
                        ) AS SubQuery',
					$this->prefixed_table,
					$args['start'],
					$args['end'],
				)
			);
		}

		return array(
			'total' => (int) $total,
			'data'  => $output,
		);
	}

	public function topBy( $column_name, $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'limit'    => 10,
				'duration' => false,
				'start'    => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'      => current_time( 'mysql' ),
			)
		);

		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		if ( false !== $args['duration'] ) {
			return $wpdb->get_results(
				$wpdb->prepare(
					'SELECT %i, COUNT(*) AS visit_count, AVG(duration) AS average_duration
					 FROM %i
					 WHERE (created_at BETWEEN %s AND %s) AND duration > 0
					 GROUP BY %i
					 ORDER BY COUNT(%i) DESC
					 LIMIT %d',
					$column_name,
					$this->prefixed_table,
					$args['start'],
					$args['end'],
					$column_name,
					$column_name,
					$args['limit']
				)
			);
		}

		return $wpdb->get_results(
			$wpdb->prepare(
				'SELECT %i, COUNT(*) AS visit_count, AVG(duration) AS average_duration
				 FROM %i
				 WHERE (created_at BETWEEN %s AND %s)
				 GROUP BY %i
				 ORDER BY COUNT(%i) DESC
				 LIMIT %d',
				$column_name,
				$this->prefixed_table,
				$args['start'],
				$args['end'],
				$column_name,
				$column_name,
				$args['limit']
			)
		);
	}

	public function totalWatchTimeByDay( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		$results = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT SUM(duration) AS total, created_at AS date_time
				 FROM %i
				 WHERE (created_at >= %s AND created_at <= %s) AND duration > 0
				 GROUP BY DAY(created_at)
				 ORDER BY created_at',
				$this->prefixed_table,
				$args['start'],
				$args['end']
			)
		);

		$average = (float) $wpdb->get_var(
			$wpdb->prepare(
				'SELECT AVG(duration) AS average_watch_time
				 FROM %i
				 WHERE (created_at >= %s AND created_at <= %s) AND duration > 0',
				$this->prefixed_table,
				$args['start'],
				$args['end']
			)
		);

		return array(
			'average' => $average,
			'data'    => $results,
		);
	}

	/**
	 * Video views
	 *
	 * @return void
	 */
	public function views( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		if ( ! $args['video_id'] ) {
			return new \WP_Error( 'missing_parameter', 'You must provide a video id.' );
		}

		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		$result = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT COUNT(*) AS total
				 FROM %i
				 WHERE video_id = %d
				 AND (created_at >= %s AND created_at <= %s) AND duration > 0',
				$this->prefixed_table,
				$args['video_id'],
				$args['start'],
				$args['end']
			)
		);

		return ! empty( $result[0]->total ) ? (int) $result[0]->total : 0;
	}

	/**
	 * Video average watch time
	 *
	 * @param array $args
	 * @return float
	 */
	public function averageWatchTime( $args = array() ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'limit'    => 10,
				'duration' => false,
				'start'    => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'      => current_time( 'mysql' ),
			)
		);

		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		return (float) $wpdb->get_var(
			$wpdb->prepare(
				'SELECT AVG(duration) AS average_watch_time
				 FROM %i
				 WHERE %i.video_id = %d
				 AND (%i.created_at BETWEEN %s AND %s)',
				$this->prefixed_table,
				$this->prefixed_table,
				$args['video_id'],
				$this->prefixed_table,
				$args['start'],
				$args['end']
			)           
		);
	}

	public function timeline( $args = array() ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		if ( ! $args['video_id'] ) {
			return new \WP_Error( 'missing_parameter', 'You must provide a video id.' );
		}

		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		$results = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT COUNT(id) AS total, duration AS watch_time
				 FROM %i
				 WHERE video_id = %d
				 AND (created_at >= %s AND created_at <= %s)
				 GROUP BY duration
				 ORDER BY duration',
				$this->prefixed_table,
				$args['video_id'],
				$args['start'],
				$args['end']
			)
		);

		if ( empty( $results ) || ! is_array( $results ) ) {
			return $results;
		}

		// get the total views on the video
		$total_views = 0;
		foreach ( $results as $result ) {
			$total_views = $total_views + (int) $result->total; // add to total
		}

		// all start at zero
		$output[] = (object) array(
			'total'      => $total_views,
			'watch_time' => 0,
		);

		foreach ( $results as $result ) {
			if ( 0 === (int) $result->watch_time ) {
				continue;
			}
			$total_views = $total_views - $result->total;
			$output[]    = (object) array(
				'total'      => $total_views,
				'watch_time' => (int) $result->watch_time,
			);
		}

		return $output;
	}

	public function totalViewsByDay( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);


		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		$results = $wpdb->get_results(
			$wpdb->prepare(
				'SELECT COUNT(id) AS total, created_at AS date_time
				 FROM %i
				 WHERE (created_at BETWEEN %s AND %s) AND duration > 0
				 GROUP BY DAY(created_at)
				 ORDER BY created_at',
				$this->prefixed_table,
				$args['start'],
				$args['end']
			)
		);

		return ! empty( $results ) ? $results : array();
	}

	/**
	 * Total Video views by particular user
	 *
	 * @return void
	 */
	public function totalVideoViewsByUser( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		if ( ! $args['user_id'] ) {
			return new \WP_Error( 'missing_parameter', 'You must provide a user id.' );
		}

		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date

		return (float) $wpdb->get_var(
			$wpdb->prepare(
				'SELECT COUNT(*) AS total
				 FROM %i
				 WHERE user_id = %d
				 AND (created_at >= %s AND created_at <= %s) AND duration > 0',
				$this->prefixed_table,
				$args['user_id'],
				$args['start'],
				$args['end']
			)
		);
	}

	/**
	 * Average watch time by particular user
	 *
	 * @return void
	 */
	public function videoAverageWatchTimeByUser( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		if ( ! $args['user_id'] ) {
			return new \WP_Error( 'missing_parameter', 'You must provide a user id.' );
		}

		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		return (float) $wpdb->get_var(
			$wpdb->prepare(
				'
				SELECT AVG(duration) AS average_duration
				FROM %i
				WHERE user_id = %d
				AND (created_at >= %s AND created_at <= %s) AND duration > 0
				GROUP BY user_id
				',
				$this->prefixed_table,
				$args['user_id'],
				$args['start'],
				$args['end']
			)
		);
	}


	/**
	 * Total watch time by particular user
	 *
	 * @return void
	 */
	public function videoTotalWatchTimeByUser( $args ) {
		global $wpdb;

		$args = wp_parse_args(
			$args,
			array(
				'start' => date( 'Y-m-d H:i:s', strtotime( '-1 week' ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				'end'   => current_time( 'mysql' ),
			)
		);

		if ( ! $args['user_id'] ) {
			return new \WP_Error( 'missing_parameter', 'You must provide a user id.' );
		}

		// make sure it's the beginning and end of the day.
		$args['start'] = date( 'Y-m-d 00:00:00', strtotime( $args['start'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		$args['end']   = date( 'Y-m-d 23:59:59', strtotime( $args['end'] ) ); // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
		return (float) $wpdb->get_var(
			$wpdb->prepare(
				'
				SELECT SUM(duration) AS total_duration
				FROM %i
				WHERE user_id = %d
				AND (created_at >= %s AND created_at <= %s) AND duration > 0
				GROUP BY user_id
				',
				$this->prefixed_table,
				$args['user_id'],
				$args['start'],
				$args['end']
			)
		);
	}

	public function checkDate( $date ) {
		$mm         = substr( $date, 5, 2 );
		$jj         = substr( $date, 8, 2 );
		$aa         = substr( $date, 0, 4 );
		$valid_date = wp_checkdate( $mm, $jj, $aa, $date );
		if ( ! $valid_date ) {
			return new \WP_Error( 'invalid_date', __( 'Invalid date.', 'presto-player-pro' ), 400 );
		}
	}

	public function deleteOlderThan( $day_string ) {
		// fetch ids of older visits
		$ids = $this->fetch(
			array(
				'date_query' => array(
					'before' => date( 'Y-m-d 00:00:00', strtotime( '-' . sanitize_text_field( $day_string ) ) ), // phpcs:ignore WordPress.DateTime.RestrictedFunctions.date_date
				),
				'fields'     => 'ids',
			)
		);

		if ( ! empty( $ids->data ) ) {
			$deleted = $this->bulkDelete( $ids->data );
			return $deleted;
		}

		return false;
	}
}
