<?php
/**
 * Bunny CDN Block.
 *
 * @package PrestoPlayer\Pro\Blocks
 */

namespace PrestoPlayer\Pro\Blocks;

use PrestoPlayer\Plugin;
use PrestoPlayer\Support\Block;
use PrestoPlayer\Models\CurrentUser;
use PrestoPlayer\Models\Video as PrestoVideo;
use PrestoPlayer\Pro\Support\Utility;
use PrestoPlayer\Pro\Services\Bunny\BunnyService;
use PrestoPlayer\Pro\Models\Bunny\Video;

/**
 * Bunny CDN Block.
 */
class BunnyCDNBlock extends Block {
	/**
	 * Block name
	 *
	 * @var string
	 */
	protected $name = 'bunny';

	/**
	 * Backwards compat.
	 */
	public function __construct() {

		if ( version_compare( Plugin::version(), '0.9', '<' ) ) {
			$this->template_name = 'self-hosting';
		}

		add_filter( 'presto_player/block/default_attributes', array( $this, 'addDefaultBunnyAttributes' ), 10 );
		add_filter( 'presto_player/component/attributes', array( $this, 'addBunnyComponentData' ), 10, 2 );
		add_action( 'init', array( $this, 'registerBlockType' ) );
	}

	/**
	 * Register block type.
	 *
	 * @return void
	 */
	public function registerBlockType() {
		if ( ! defined( 'PRESTO_PLAYER_PLUGIN_DIR' ) ) {
			return;
		}

		// Backwards compat.
		if ( version_compare( \PrestoPlayer\Plugin::version(), '2.0.0-beta1', '<' ) ) {
			add_action(
				'wp_enqueue_scripts',
				function () {
					wp_enqueue_script( 'presto-components' );
				}
			);
			return parent::registerBlockType();
		}

		register_block_type(
			PRESTO_PLAYER_PLUGIN_DIR . 'src/admin/blocks/blocks/bunny',
			array(
				'render_callback' => array( $this, 'html' ),
			)
		);
	}

	/**
	 * Add bunny thumbnail as poster by default.
	 *
	 * @param array $data Data.
	 * @return array
	 */
	public function addDefaultBunnyAttributes( $data ) {
		$attributes = $data['blockAttributes'];

		if ( 'bunny' !== $data['provider'] ) {
			return $data;
		}

		// Maybe set bunny thumbnail as poster.
		if ( empty( $attributes['poster'] ) && ! empty( $attributes['thumbnail'] ) ) {
			// Maybe sign urls.
			if ( ! empty( $attributes['visibility'] ) && 'private' === $attributes['visibility'] ) {
				$data['poster'] = BunnyService::signURL( $attributes['thumbnail'] );
			} else {
				$data['poster'] = $attributes['thumbnail'];
			}
		}

		return $data;
	}

	/**
	 * Add bunny data to component.
	 *
	 * @param array $data Data.
	 * @param array $attributes Attributes.
	 * @return array
	 */
	public function addBunnyComponentData( $data, $attributes ) {
		if ( 'bunny' !== $attributes['provider'] ) {
			return $data;
		}
		return array_merge(
			$data,
			array(
				'bunny',
			)
		);
	}

	/**
	 * Middleware to check for user access.
	 * We will load a blank video here.
	 *
	 * @param array  $attributes Attributes.
	 * @param string $content Content.
	 * @return bool
	 */
	public function middleware( $attributes, $content ) {
		// If private and user cannot access video, don't load.
		if ( ! empty( $attributes['visibility'] ) && 'private' === $attributes['visibility'] ) {
			if ( ! CurrentUser::canAccessVideo( ! empty( $attributes['id'] ) ? $attributes['id'] : 0 ) ) {
				return false;
			}
		}

		return true;
	}

	/**
	 * Override attributes.
	 *
	 * @param array $attributes Attributes.
	 * @return array
	 */
	public function overrideAttributes( $attributes ) {
		$load = $this->middleware( $attributes, '' );

		if ( ! $load ) {
			return array();
		}

		if ( ! empty( $attributes['src'] ) ) {
			if ( Utility::isStream( $attributes['src'] ) ) {
				wp_enqueue_script( 'hls.js' );
			}
		}

		$presto_video   = new PrestoVideo();
		$bunny_video_id = method_exists( $presto_video, 'getExternalId' ) ? $presto_video->getExternalId( $attributes['src'] ) : '';

		if ( ! empty( $bunny_video_id ) ) {
			$visibility     = ! empty( $attributes['visibility'] ) ? $attributes['visibility'] : 'public';
			$video          = new Video( (object) array( 'guid' => $bunny_video_id ), $visibility );
			$bunny_captions = $video->captions()->getCaptions();

			if ( ! empty( $bunny_captions ) ) {
				$existing_tracks = ! empty( $attributes['tracks'] ) && is_array( $attributes['tracks'] )
					? $attributes['tracks']
					: array();

				$attributes['tracks'] = $this->mergeCaptions( $existing_tracks, $bunny_captions );
			}
		}

		if ( ! empty( $attributes['visibility'] ) && 'private' === $attributes['visibility'] ) {
			$attributes['src'] = ! empty( $attributes['src'] ) ? BunnyService::signURL( $attributes['src'] ) : '';
		}

		return $attributes;
	}

	/**
	 * Merge existing block tracks with Bunny captions.
	 *
	 * @param array $existing_tracks Existing tracks from block attributes.
	 * @param array $bunny_captions  Captions fetched from Bunny API.
	 * @return array
	 */
	protected function mergeCaptions( $existing_tracks, $bunny_captions ) {
		$user_tracks = array_filter(
			$existing_tracks,
			function ( $track ) {
				return ( $track['source'] ?? '' ) !== 'bunny';
			}
		);

		return array_values( array_merge( $user_tracks, $bunny_captions ) );
	}
}
