<?php

namespace EGPR;

use EGPR\Data_Structures\Post_Types\Press_Releases;

/**
 * Class Rest_Server
 */
class Server extends \WP_Rest_Controller {

    public $namespace = 'egpr/';
    public $version = 'v1';


    public function init() {
        add_action( 'rest_api_init', array( $this, 'register_routes' ) );
    }


    public function register_routes() {
        $namespace = $this->namespace . $this->version;


        register_rest_route(
            $namespace,
            '/press',
            array(
                array(
                    'methods'  => \WP_REST_Server::CREATABLE,
                    'callback' => array( $this, 'create_press' ),
                    'permission_callback' => array( $this, 'get_users_permission' ),
                    'args' => array(
                        'type' => array(
                            'required' => true,
                            'type'     => 'string',
                        ),
                        'image_url' => array(
                            'required' => false,
                            'type'     => 'string|boolean',
                        ),
                        'image_base64' => array(
                            'required' => false,
                            'type'     => 'string',
                        ),
                        'title' => array(
                            'required' => true,
                            'type'     => 'string',
                        ),
                        'url' => array(
                            'required' => true,
                            'type'     => 'string',
                        ),
                        'source_post_id' => array(
                            'required' => true,
                            'type'     => 'int',
                        ),
                        'date' => array(
                            'required' => true,
                            'type'     => 'string',
                        ),



                    ),
                ),
            )
        );

        register_rest_route(
            $namespace,
            '/press',
            array(
                array(
                    'methods'  => \WP_REST_Server::DELETABLE,
                    'callback' => array( $this, 'trash_actions_press' ),
                    'permission_callback' => array( $this, 'get_users_permission' ),
                    'args' => array(
                        'action' => array(
                            'required' => true,
                            'type'     => 'string',
                        ),
                        'source_post_id' => array(
                            'required' => true,
                            'type'     => 'int',
                        ),



                    ),
                ),
            )
        );



    }

    public function trash_actions_press(\WP_REST_Request $request){
            $source_post_id = $request->get_param( 'source_post_id' );
            $action = $request->get_param( 'action' );
            Utils::log(sprintf('Trash action: %s for source post id of %s'));
            $exists = get_posts([
                'post_type' => Press_Releases::POST_TYPE,
                'meta_query' => [
                    [
                        'key' => 'source_post_id',
                        'value' => $source_post_id,
                    ]
                ]
            ]);
            if(empty($exists)){
                Utils::log('No press release found for source post id of ' . $source_post_id);
            }
            $post_id = $exists[0]->ID;
            Utils::log('Title for the post where trash action ' . $action . ' for post id ' . $post_id . ' with title ' . get_the_title($post_id));

            switch($action):
                case 'trash':
                    Utils::log('Trashing post ' . $post_id);
                    $result = wp_trash_post($post_id);
                    Utils::log('Result of trashing post ' . !!$result ? 'success' : 'failure');
                    break;
                case 'untrash':
                    Utils::log('Untrashing post ' . $post_id);
                    $result = wp_untrash_post($post_id);
                    Utils::log('Result of untrashing post ' . !!$result ? 'success' : 'failure');
                    break;
                case 'delete':
                    $result = wp_delete_post($post_id, true);
                    Utils::log('Result of deleting post ' . !!$result ? 'success' : 'failure');
                    break;
                default:
                    break;

                endswitch;
                $this->send_json_response();

    }

    public function create_press(\WP_REST_Request $request){
        remove_action( 'save_post', array( Press_Releases::class, 'save_post' ), 1, 2 );
        $image_base64 = $request->get_param( 'image_base64' );
        $type = $request->get_param( 'type' );
        $url = $request->get_param( 'url' );
        $image_url = $request->get_param( 'image_url' );
        $title = $request->get_param( 'title' );
        $source_post_id = $request->get_param( 'source_post_id' );
        $date = $request->get_param( 'date' );


        $exists = get_posts([
            'post_type' => Press_Releases::POST_TYPE,
            'meta_query' => [
                [
                    'key' => 'source_post_id',
                    'value' => $source_post_id,
                ]
            ]
        ]);

        if(is_array($exists) && count($exists) > 0){
            // update post
            Utils::log('Updating post ' . $title . ' with source post id of ' . $source_post_id);
            $post_id = $exists[0]->ID;
            wp_update_post(array(
                'ID' => $post_id,
                'post_title' => $title,
                'post_type' => Press_Releases::POST_TYPE,
                'post_status' => 'publish',
                'post_date' => $date,
            ));
            if($image_base64){
                $image_id = self::set_image($image_base64, $title, $image_url);
                set_post_thumbnail($post_id, $image_id);
            } else {
                Utils::log('No image for ' . $title);
            }


            add_action( 'save_post', array( Press_Releases::class, 'save_post' ), 1, 2 );
            return;
        }

        Utils::log('Creating post ' . $title);
        $post_id = wp_insert_post(array(
            'post_title' => $title,
            'post_type' => Press_Releases::POST_TYPE,
            'post_status' => 'publish',
            'post_date' => $date,
        ));
        add_action( 'save_post', array( Press_Releases::class, 'save_post' ), 1, 2 );
        if ( is_wp_error( $post_id ) ) {
            Utils::log('Error creating post ' . $title . ' ' . $post_id->get_error_message());
            return $this->send_json_response($post_id, false);
        }

        if($image_base64){
            $image_id = self::set_image($image_base64, $title, $image_url);
            set_post_thumbnail($post_id, $image_id);
        } else {
            Utils::log('No image for ' . $title);
        }
        Utils::log('Setting meta for ' . $title);
        update_field('source_post_id', $source_post_id, $post_id);
        update_field('url', $url, $post_id);
        update_field('type', $type, $post_id);
        return $this->send_json_response();

    }



    public function send_json_response($results = [], $success = true){
        $response = new \WP_REST_Response(
            array(
                'data' => array(
                    'success'  => $success,
                    'response' => $results,
                ),
            )
        );
        if(!$success){
            $response->set_status(400);
        }
        return $response;
    }

    /**
     * Is the user logged in?
     *
     * @return bool|WP_Error
     */
    public function get_users_permission() {

        return is_user_logged_in() && current_user_can('edit_posts');
    }

    private static function set_image($base64_image, $title, $source_url) {
        $args = array(
            'post_type' => 'attachment',
            'meta_key' => 'image_url',
            'meta_value' => $source_url
        );
        $images = get_posts($args);
        if (count($images) > 0) {
            Utils::log('Image already exists for ' . $title . ' ' . $source_url . ' with meta key image_url and value ' . $source_url);
            return $images[0]->ID;
        }
        Utils::log('Creating image for ' . $title . ' ' . $source_url);
        $image_id = Images_Utils::upload_image($base64_image, $title);
        Utils::log('Image ID for  ' . $image_id . ' ' . $source_url);
        update_field('image_url', $source_url, $image_id);
    }




}
