<?php

namespace EGPR\Data_Structures\Post_Types;

use EGPR\Images_Utils;
use EGPR\Queue\Post_Creator_Manager;
use EGPR\Queue\Post_Creator_Process;
use EGPR\Settings;
use EGPR\Utils;

class Press_Releases {

    const POST_TYPE            = 'eg-press-releases';
    const LOG_KEY = 'egpr_log';



    protected $process_all;


    public function init() {
        $this->attach_hooks();
    }

    public function attach_hooks() {
        add_action('trashed_post', array($this, 'trashed_post'), 10, 1);
        add_action('untrashed_post', array($this, 'untrashed_post'), 10, 1);
        add_action('delete_post', array($this, 'delete_post'), 10, 1);

        add_action( 'init', array( $this, 'register_post_type' ) );
        add_action( 'save_post', array( __CLASS__, 'save_post' ), 1, 2 );
        add_action('acf/include_fields', array($this, 'settings'));
        add_filter('post_type_link', array($this, 'permalink'), 10, 2);
        add_filter('post_link', array($this, 'permalink'), 10, 2);
        add_action('init', array($this, 'image_sizes'));
        add_action( 'init', array($this, 'rewrite' ) );
        add_filter( 'query_vars', array($this, 'query_vars') );

        add_action('pre_get_posts', array($this, 'pre_get_posts'));


    }

    public static function get($const){
        switch ($const) :
            case 'POSTS_PER_PAGE_LINKS_PAGE_1':
                return apply_filters('egpr/POSTS_PER_PAGE_LINKS_PAGE_1', 5);
                break;
            case 'POSTS_PER_PAGE_VIDEOS_PAGE_1':
                return apply_filters('egpr/POSTS_PER_PAGE_VIDEOS_PAGE_1', 2);
                break;
            case 'POSTS_PER_PAGE_BEYOND_1_LINKS':
                return apply_filters('egpr/POSTS_PER_PAGE_BEYOND_1_LINKS', 10);
                break;
            case 'POSTS_PER_PAGE_BEYOND_1_VIDEOS':
                return apply_filters('egpr/POSTS_PER_PAGE_BEYOND_1_VIDEOS', 10);
                break;
            default:
                return null;

        endswitch;

    }



    public function pre_get_posts($query){
        if ( ! is_admin() && $query->is_main_query() && $query->is_post_type_archive( 'eg-press-releases' ) ) {
            $eg_press_type = $query->get( 'eg_press_type' );

            if ( $eg_press_type ) {
                $paged = $query->get( 'paged' ) ?: 1;

                if ( $paged === 1 ) {
                    $posts_per_page = $eg_press_type === 'links' ? self::get('POSTS_PER_PAGE_LINKS_PAGE_1') : self::get('POSTS_PER_PAGE_VIDEOS_PAGE_1');
                } else {
                    $posts_per_page = ($eg_press_type === 'links' ? self::get('POSTS_PER_PAGE_BEYOND_1_LINKS') : self::get('POSTS_PER_PAGE_BEYOND_1_VIDEOS'));
                }

                $offset = $paged <= 1 ? 0 : ($paged - 2) * ($eg_press_type === 'links' ? self::get('POSTS_PER_PAGE_BEYOND_1_LINKS') : self::get('POSTS_PER_PAGE_BEYOND_1_VIDEOS')) + ($eg_press_type === 'links' ? self::get('POSTS_PER_PAGE_LINKS_PAGE_1') : self::get('POSTS_PER_PAGE_VIDEOS_PAGE_1'));

                $query->set( 'posts_per_page', $posts_per_page );
                $query->set( 'offset', $offset );

                $meta_query = array(
                    array(
                        'key' => 'type',
                        'value' => $eg_press_type,
                    ),
                );
                $query->set( 'meta_query', $meta_query );
            }
        }
    }

    public function trashed_post($post_id){
        if(get_post_type($post_id) !== self::POST_TYPE){
            return;
        }
        $this->trash_action_for_post($post_id, 'trash');
    }
    public function untrashed_post($post_id){
        if(get_post_type($post_id) !== self::POST_TYPE){
            return;
        }
        $this->trash_action_for_post($post_id, 'untrash');
    }
    public function delete_post($post_id){
        if(get_post_type($post_id) !== self::POST_TYPE){
            return;
        }
        $this->trash_action_for_post($post_id, 'delete');
    }

    public  function trash_action_for_post($post_id, $action) {

        $endpoints = Utils::get_endpoints();
        if(!is_array($endpoints) || count($endpoints) === 0){
            return;
        }
        $queue = [];
        foreach($endpoints as $endpoint) {
            $queue[] = ( [
                'endpoint' => $endpoint,
                'action' => $action,
                'post_id' => $post_id,
                'method' => 'trash'
            ]);
        }
        $pr = new Post_Creator_Manager($queue);
        $pr->process();

    }

    public function template_redirect(){
        $x=1;
        if ( get_query_var('post_type') === 'eg-press-releases' &&  in_array(get_query_var( 'eg_press_type' ), [ 'links', 'videos' ]) ) {
            return locate_template( 'archive-eg-press-releases.php', true );
            exit;
        }
    }



    public function query_vars($vars){
        $vars[] = 'eg_press_type';
        return $vars;
    }

    public function rewrite(){
        add_rewrite_rule(
            '^press/(videos|links)/?$',
            'index.php?post_type=' . self::POST_TYPE .'&eg_press_type=$matches[1]',
            'top'
        );
        add_rewrite_rule(
            '^press/(videos|links)/page/([0-9]+)/?$',
            'index.php?post_type=' . self::POST_TYPE .'&eg_press_type=$matches[1]&paged=$matches[2]',
            'top'
        );
    }



    public function image_sizes(){
        add_image_size('525x300', 525, 300, true);
        add_image_size('255x205', 255, 205, true);
    }

    public function permalink($permalink, $post){
        if($post->post_type !== self::POST_TYPE){
            return $permalink;
        }
        // Get the meta field value
        return get_field('url', $post->ID) ?: $permalink;


    }

    public function settings(){
        if ( ! function_exists( 'acf_add_local_field_group' ) ) {
            return;
        }

        acf_add_local_field_group( array(
            'key' => 'group_64b720c4a755a',
            'title' => 'Press Releases',
            'fields' => array(
                array(
                    'key' => 'field_64b720c65c7cd',
                    'label' => 'Type',
                    'name' => 'type',
                    'aria-label' => '',
                    'type' => 'select',
                    'instructions' => '',
                    'required' => 0,
                    'conditional_logic' => 0,
                    'wrapper' => array(
                        'width' => '',
                        'class' => '',
                        'id' => '',
                    ),
                    'choices' => array(
                        'links' => 'Link',
                        'videos' => 'Video',
                    ),
                    'default_value' => 'link',
                    'return_format' => 'value',
                    'multiple' => 0,
                    'allow_null' => 0,
                    'ui' => 0,
                    'ajax' => 0,
                    'placeholder' => '',
                ),
                array(
                    'key' => 'field_64b720ee5c7ce',
                    'label' => 'URL',
                    'name' => 'url',
                    'aria-label' => '',
                    'type' => 'url',
                    'instructions' => '',
                    'required' => 0,
                    'conditional_logic' => 0,
                    'wrapper' => array(
                        'width' => '',
                        'class' => '',
                        'id' => '',
                    ),
                    'default_value' => '',
                    'placeholder' => '',
                ),


            ),
            'location' => array(
                array(
                    array(
                        'param' => 'post_type',
                        'operator' => '==',
                        'value' => self::POST_TYPE,
                    ),
                ),
            ),
            'menu_order' => 0,
            'position' => 'normal',
            'style' => 'default',
            'label_placement' => 'top',
            'instruction_placement' => 'label',
            'hide_on_screen' => '',
            'active' => true,
            'description' => '',
            'show_in_rest' => true,
        ) );
    }



    public function register_post_type() {
        register_post_type(
            self::POST_TYPE,
            array(
                'labels'              => array(
                    'name'               => __( 'Press Releases', 'egprep' ),
                    'singular_name'      => __( 'Press Release', 'egprep' ),
                    'add_new'            => __( 'Add New', 'egprep' ),
                    'add_new_item'       => __( 'Add New Press Release', 'egprep' ),
                    'edit_item'          => __( 'Edit Press Release', 'egprep' ),
                    'new_item'           => __( 'New Press Release', 'egprep' ),
                    'all_items'          => __( 'All Press Releases', 'egprep' ),
                    'view_item'          => __( 'View Press Release', 'egprep' ),
                    'search_items'       => __( 'Search Press Releases', 'egprep' ),
                    'not_found'          => __( 'No Press Releases Found', 'egprep' ),
                    'not_found_in_trash' => __( 'No Press Releases found in Trash', 'egprep' ),
                    'menu_name'          => __( 'Press Releases', 'egprep' ),
                ),
                'menu_icon'           => 'dashicons-superhero',
                'query_var'           => true,
                'publicly_queryable'  => true,
                'exclude_from_search' => false,
                'hierarchical'        => false,
                'show_in_menu'        => true,
                'public'              => true,
                'show_ui'             => true,
                'rewrite'             => array(
                    'slug'       => 'press',
                    'with_front' => true,
                ),
                'has_archive'         => apply_filters('egpr/has_archive', true),
                'supports'            => array( 'title', 'thumbnail' ),
                'map_meta_cap'        => true,
                'show_in_rest'        => true,

            )
        );
    }


    public static function save_post( $post_id, $post )
    {

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }
        if (defined('DOING_AJAX') && DOING_AJAX) {
            return;
        }
        if (!current_user_can('edit_post', $post_id)) {
            return;
        }
        if (false !== wp_is_post_revision($post_id)) {
            return;
        }
        if (self::POST_TYPE !== $post->post_type) {
            return;
        }

        $endpoints = Utils::get_endpoints();
        if(!$endpoints || (is_array($endpoints) && count((array)$endpoints)===0)) {
            return;

        }

        $title = get_the_title($post_id);
        $image_url = get_the_post_thumbnail_url($post_id, 'full');
        $url = get_field('url', $post_id);
        $type = get_field('type', $post_id);
        $image_base64 = '';
        if ($image_url) {
            $image_base64 = Images_Utils::url_to_base64($image_url);
        }
        $body = array('image_base64' => $image_base64, 'type' => $type, 'url' => $url, 'image_url' => $image_url, 'title' => $title, 'source_post_id' => $post_id, 'date' => get_post_datetime($post_id)->format('Y-m-d H:i:s'));



        Utils::Log('Getting started for ' . $title . ' with source post id ' . $post_id );
        Utils::log('Found ' . count($endpoints) . ' endpoints');

        Utils::log('Lets get started ' . count($endpoints) . ' endpoints');
        $queue = [];
        foreach($endpoints as $endpoint) {
            $item = ['endpoint' => $endpoint, 'body' => $body, 'method' => 'upsert'];
            Utils::log('adding...');

            $queue[] = $item;
        }
        $pr = new Post_Creator_Manager($queue);
        $pr->process();


    }

    public static function delete_action($args){
        $data = [
            'method' => 'delete',
            'action' => $args['action'],
            'source_post_id' => $args['post_id'],
            'endpoint' => $args['endpoint']
        ];
    }

    public static function upsert_action( $args ){
        $data = [
            'method' => 'upsert',
            'body' => $args['body'],
            'endpoint' => $args['endpoint'],
        ];

    }

    public static function _delete_action($args){

        $args = array_merge(['endpoint' => null, 'action' => null, 'post_id' => null], $args);

        if(!$args['endpoint'] || !$args['action'] || !$args['post_id']){
            Utils::log('Missing required arguments for delete action');
            return;
        }

        $endpoint = $args['endpoint'];
        $action = $args['action'];
        $post_id = $args['post_id'];

        $endpoint['url'] = trailingslashit($endpoint['url']);
        $endpoint['url'] = add_query_arg(['source_post_id' => $post_id, 'action' => $action], $endpoint['url']);
        $credentials = $endpoint['auth_header'];
        $encoded_credentials = base64_encode($credentials);
        $headers = array(
            'Authorization' => 'Basic ' . $encoded_credentials,
            'Content-Type' => 'application/json'
        );



        $response = wp_remote_post($endpoint['url'], array(
            'method' => 'DELETE',
            'headers' => $headers,
            'timeout' => 90,
        ));


        if (!is_wp_error($response)) {
            $response_code = wp_remote_retrieve_response_code($response);
            Utils::log('Response code: ' . $response_code);
            $response_body = wp_remote_retrieve_body($response);

        } else {
            $error_message = $response->get_error_message();
            Utils::log('Error: ' . $error_message);


        }
    }

    public static function _upsert_action($args){
        $args = array_merge(['endpoint' => null, 'body' => null], $args);
        $endpoint = $args['endpoint'];
        $body = $args['body'];

        if(!$endpoint || !$body){
            Utils::log('Endpoint or body is null for upsert action');
        }

        Utils::log('Pushing to ' . $endpoint['url']);
        $endpoint['url'] = trailingslashit($endpoint['url']);
        $credentials = $endpoint['auth_header'];


        $encoded_credentials = base64_encode($credentials);
        $headers = array(
            'Authorization' => 'Basic ' . $encoded_credentials,
            'Content-Type' => 'application/json'
        );


        $response = wp_remote_post($endpoint['url'], array(
            'method' => 'POST',
            'headers' => $headers,
            'body' => wp_json_encode($body),
            'timeout' => 90,

        ));


        if (!is_wp_error($response)) {
            $response_code = wp_remote_retrieve_response_code($response);
            Utils::log('Response code: ' . $response_code);
            $response_body = wp_remote_retrieve_body($response);
            Utils::log('Response body: ' . $response_body);

        } else {
            $error_message = $response->get_error_message();
            Utils::log('Error: ' . $error_message);


        }
    }

    public static function timeout(){
        return 90;
    }


}
