<?php
// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

function monsterinsights_eu_compliance_require_optin() {

	if (  ( function_exists( 'monsterinsights_is_cookiebot_active' ) && monsterinsights_is_cookiebot_active() ) || class_exists( 'Cookie_Notice' ) ) {
		return apply_filters( 'monsterinsights_eu_compliance_require_optin', false );
	}

	return false;
}

// Allow for AMP to wait for AMP consent
function monsterinsights_amp_add_analytics_consent( $analytics ) {
	$consent = monsterinsights_get_option( 'require_amp_consent', false );
	if ( $consent ) {
		$analytics['monsterinsights-googleanalytics']['attributes']['data-block-on-consent'] = apply_filters( 'monsterinsights_amp_add_analytics_consent', "_till_accepted" ); // see https://www.ampproject.org/docs/reference/components/amp-consent#basic-blocking-behaviors for valid options
	}

	return $analytics;
}

add_filter( 'monsterinsights_amp_add_analytics', 'monsterinsights_amp_add_analytics_consent' );

// override demographics to false
add_filter( 'monsterinsights_get_option_demographics', 'monsterinsights_eu_compliance_addon_option_false' );

// override anonymize_ips to true
add_filter( 'monsterinsights_get_option_anonymize_ips', 'monsterinsights_eu_compliance_addon_option_true' );

// override gatracker_compatibility_mode to true
add_filter( 'monsterinsights_get_option_gtagtracker_compatibility_mode', '__return_true' );

// override userID to false
add_filter( 'monsterinsights_get_option_userid', 'monsterinsights_eu_compliance_addon_option_false' );

function monsterinsights_eu_compliance_addon_option_false( $value ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $value;
	}

	return false;
}

function monsterinsights_eu_compliance_addon_option_true( $value ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $value;
	}

	return true;
}

// Force DisplayFeatures off, even if they are turned on in the GA settings (override account settings)
function monsterinsights_eu_compliance_force_displayfeatures_off( $options ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $options;
	}
	$options['demographics'] = "'set', 'displayFeaturesTask', null";

	return $options;
}

add_filter( 'monsterinsights_frontend_tracking_options_analytics_before_pageview', 'monsterinsights_eu_compliance_force_displayfeatures_off' );

// Hide userID and author custom dimension
function monsterinsights_eu_compliance_custom_dimensions( $dimensions ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $dimensions;
	}

	$dimensions['user_id']['enabled'] = false;
	$dimensions['author']['enabled']  = false;

	remove_filter( 'monsterinsights_get_option_custom_dimensions', 'monsterinsights_eu_compliance_custom_dimensions_option' );

	$dimensions_option = monsterinsights_get_option( 'custom_dimensions', array() );
	$show_notification = false;
	foreach ( $dimensions_option as $dimension ) {
		if ( in_array( $dimension['type'], [ 'user_id', 'author' ], true ) ) {
			$show_notification = true;
			break;
		}
	}

	add_filter( 'monsterinsights_get_option_custom_dimensions', 'monsterinsights_eu_compliance_custom_dimensions_option' );

	if ( $show_notification ) {
		$notification = array(
			'id'      => 'eu_compliance_custom_dimensions_disable_author_userid',
			'title'   => __( 'UserID and Author ID Custom Dimensions Disabled', 'google-analytics-for-wordpress' ),
			'type'    => array( 'basic', 'lite', 'master', 'plus', 'pro' ),
			'content' => esc_html__( 'MonsterInsights has automatically disabled the User ID and Author custom dimensions due to the EU Compliance addon being set to enabled.', 'google-analytics-for-wordpress' ) . "\n\n",
		);

		MonsterInsights()->notifications->add( $notification );
	}

	return $dimensions;
}

add_filter( 'monsterinsights_available_custom_dimensions', 'monsterinsights_eu_compliance_custom_dimensions' );

// Remove user_id and author from being used even if already set to be used
function monsterinsights_eu_compliance_custom_dimensions_option( $dimensions ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $dimensions;
	}
	if ( ! empty( $dimensions ) && is_array( $dimensions ) ) {
		foreach ( $dimensions as $key => $row ) {
			if ( ! empty( $row['type'] ) && ( $row['type'] === 'user_id' || $row['type'] === 'author' ) ) {
				unset( $dimensions[ $key ] );
			}
		}
	}

	return $dimensions;
}

add_filter( 'monsterinsights_get_option_custom_dimensions', 'monsterinsights_eu_compliance_custom_dimensions_option' );

// Remove userIDs from the MP calls
function monsterinsights_eu_compliance_mp_v4_api_call_user_id( $body ) {
	if ( monsterinsights_eu_compliance_require_optin() ) {
		return $body;
	}
	unset( $body['user_id'] );

	return $body;
}

add_filter( 'monsterinsights_mp_v4_api_call', 'monsterinsights_eu_compliance_mp_v4_api_call_user_id' );

function monsterinsights_eu_compliance_tracking_analytics_script_attributes( $attributes ) {

	if ( function_exists( 'monsterinsights_is_cookiebot_active' ) && monsterinsights_is_cookiebot_active() ) {
		$attributes['type'] = 'text/plain';
		$attributes['data-cookieconsent'] = 'statistics';
	}

	if ( defined( 'cmplz_plugin' ) || defined( 'cmplz_premium' ) || defined( 'cmplz_free' ) ) {
		$attributes['type']          = 'text/plain';
		$attributes['data-service']  = 'google-analytics';
		$attributes['data-category'] = 'statistics';
	}

	// For CookieYes plugin.
	if ( function_exists( 'cky_define_constants' ) ) {
		$attributes['data-cookieyes'] = 'cookieyes-analytics';
	}

	return $attributes;
}

add_filter( 'monsterinsights_tracking_analytics_script_attributes', 'monsterinsights_eu_compliance_tracking_analytics_script_attributes' );

function monsterinsights_eu_compliance_cookie_notice_integration() {
	if ( ! class_exists( 'Cookie_Notice' ) ) {
		return;
	}
	ob_start();
	?>
	/* Compatibility with Cookie Notice */

	var mi_has_basic_cookie_notice = document.cookie.indexOf( 'cookie_notice_accepted' ) !== -1;
	var mi_has_advanced_cookie_notice = document.cookie.indexOf( 'hu-consent' ) !== -1;

	if ( !mi_has_basic_cookie_notice && !mi_has_advanced_cookie_notice ) {
	mi_track_user      = false;
	mi_no_track_reason = '<?php echo esc_js( __( "Note: You have not accepted the Cookie Notice.", "monsterinsights-eu-compliance" ) ); ?>';
	} else {
	var mi_cn_value = document.cookie;
	var mi_cn_name = mi_has_advanced_cookie_notice ? 'hu-consent' : 'cookie_notice_accepted';
	var mi_cn_starts_at = mi_cn_value.indexOf(" " + mi_cn_name + "=");
	if (mi_cn_starts_at == -1) {
	mi_cn_starts_at = mi_cn_value.indexOf(mi_cn_name + "=");
	}
	if (mi_cn_starts_at == -1) {
	mi_cn_value = null;
	} else {
	mi_cn_starts_at = mi_cn_value.indexOf("=", mi_cn_starts_at) + 1;
	var mi_cn_ends_at = mi_cn_value.indexOf(";", mi_cn_starts_at);
	if (mi_cn_ends_at == -1) {
	mi_cn_ends_at = mi_cn_value.length;
	}
	mi_cn_value = unescape(mi_cn_value.substring(mi_cn_starts_at,mi_cn_ends_at));
	}

	function mi_on_cookie_notice_declined() {
	mi_track_user      = false;
	mi_no_track_reason = '<?php echo esc_js( __( "Note: You declined cookies on the Cookie Notice consent bar.", "monsterinsights-eu-compliance" ) ); ?>';
	}

	if ( mi_has_basic_cookie_notice && mi_cn_value !== 'true' ) {
	mi_on_cookie_notice_declined();
	} else if ( mi_has_advanced_cookie_notice ) {
	var mi_cookie_notice_object = JSON.parse(mi_cn_value);

	if ( mi_cookie_notice_object.consent !== true || mi_cookie_notice_object.consentLevel < 3 ) {
	mi_on_cookie_notice_declined();
	}
	}
	}
	<?php
	$output = ob_get_contents();
	ob_end_clean();
	echo $output; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Injected values are escaped in the script block above
}

add_action( 'monsterinsights_tracking_analytics_frontend_output_after_mi_track_user', 'monsterinsights_eu_compliance_cookie_notice_integration' );
add_action( 'monsterinsights_tracking_gtag_frontend_output_after_mi_track_user', 'monsterinsights_eu_compliance_cookie_notice_integration' );

/**
 * Cookiebot conversion tracking.
 * This function will track if Forms addon and Cookiebot both active.
 * Cookiebot developer doc.: https://www.cookiebot.com/en/developer/
 */
function monsterinsights_eu_compliance_forms_tracking() {
	if ( monsterinsights_skip_tracking() ) {
		return;
	}

	$is_cookie_plugin_active = function_exists( 'monsterinsights_is_cookiebot_active' ) && monsterinsights_is_cookiebot_active();
	$is_forms_addon_active = class_exists( 'MonsterInsights_Forms' );

	if ( ! $is_cookie_plugin_active || ! $is_forms_addon_active ) {
		return;
	}

	$attr_string = function_exists( 'monsterinsights_get_frontend_analytics_script_atts' )
		? monsterinsights_get_frontend_analytics_script_atts()
		: ' type="text/javascript" data-cfasync="false"';

	ob_start();?>
<!-- MonsterInsights Cookiebot Form Tracking -->
<script type="text/javascript" data-cfasync="false">
	function monsterInsightsCookiebotOnAcceptTracking() {
		if ( typeof(__gtagTracker) !== 'undefined' && __gtagTracker ) {
			__gtagTracker( 'event', 'conversion', {
				event_category: 'form',
				event_label: 'CybotCookiebotDialog',
				value: 1,
			} );
		} else {
			setTimeout(monsterInsightsCookiebotOnAcceptTracking, 500);
		}
	}

	function CookiebotCallback_OnAccept() {
		if ( Cookiebot.changed ) {
			monsterInsightsCookiebotOnAcceptTracking();
		}
	}
</script>
<script <?php echo $attr_string; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Handled ?>>
	if ( monsterinsights_forms_record_impression ) {
		monsterinsights_forms_record_impression();
	}
</script>
<!-- End MonsterInsights Cookiebot Form Tracking -->
	<?php
	$output = ob_get_contents();
	ob_end_clean();
	echo $output; // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Handled
}

add_action( 'wp_head', 'monsterinsights_eu_compliance_forms_tracking', 16 );

// Compatibility with CookieYes plugin.
if ( ! function_exists( 'monsterinsights_cookieyes_scripts' ) && defined( 'CLI_PLUGIN_PATH' ) && ! is_admin() ) {
	function monsterinsights_cookieyes_scripts( $tags ) {
		if ( empty( $tags['google-analytics-for-wordpress'] ) ) {
			return $tags;
		}

		$tags['google-analytics-for-wordpress'][] = 'MonsterInsightsDualTracker';
		$tags['google-analytics-for-wordpress'][] = 'monsterinsights';

		return $tags;
	}

	add_filter( 'wt_cli_third_party_scripts', 'monsterinsights_cookieyes_scripts', 11 );
}

/**
 * Handle user journey should track pages by cookie or not.
 * This is necessary for Cookie Notice plugin.
 *
 * @return bool
 */
function compliance_monsterinsights_user_journey_enqueue_js() {

	// For Cookie Notice plugin.
	if ( function_exists( 'cn_cookies_accepted' ) ) {
		return cn_cookies_accepted();
	}

	return true;
}

add_filter( 'monsterinsights_user_journey_enqueue_js', 'compliance_monsterinsights_user_journey_enqueue_js' );

/**
 * Modify user-journey script tag to stop creating cookie.
 * This is necessary for CookieBot and Complianz plugin.
 *
 * @param $tag
 * @param $handle
 * @param $src
 *
 * @return string
 */
function monsterinsights_compliance_user_journey_script_tag( $tag, $handle, $src ) {

	if ( ! function_exists( 'monsterinsights_get_frontend_analytics_script_atts' ) ) {
		return $tag;
	}

	if ( 'monsterinsights-user-journey' === $handle ) {
		$tag = sprintf(
			'<script %s src="%s" id="monsterinsights-user-journey-js"></script>' . PHP_EOL,
			monsterinsights_get_frontend_analytics_script_atts(),
			$src
		);
	}

	return $tag;
}

add_filter( 'script_loader_tag', 'monsterinsights_compliance_user_journey_script_tag', 10, 3 );

/**
 * For CookieYes plugin.
 * If CookieYes activated then we need add our script below CookieYes script.
 */
if ( function_exists( 'cky_define_constants' ) ) {
	remove_action( 'wp_head', 'monsterinsights_tracking_script', 6 );
	add_action( 'wp_head', 'monsterinsights_tracking_script', 20 );
}
