<?php

/**
 * Class MonsterInsights_Consent_Mode_Banner.
 */
class MonsterInsights_Consent_Mode_Banner {
	/**
	 * Holds the class object.
	 *
	 */
	public static $instance;

	/**
	 * EEA countries.
	 *
	 * @var array
	 */
	private $eea_countries = array(
		'AT',
		'BE',
		'BG',
		'CH',
		'CY',
		'CZ',
		'DE',
		'DK',
		'EE',
		'ES',
		'FI',
		'FR',
		'GR',
		'HR',
		'HU',
		'IE',
		'IS',
		'IT',
		'LI',
		'LT',
		'LU',
		'LV',
		'MT',
		'NL',
		'NO',
		'PL',
		'PT',
		'RO',
		'SE',
		'SI',
		'SK',
		'UK',
	);

	/**
	 * Return Singleton instance
	 *
	 */
	public static function get_instance() {
		if ( empty( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	public function __construct() {
		add_action( 'wp_footer', array( $this, 'monsterinsights_show_google_consent_banner' ) );
	}

	/**
	 * This method will print HTML for the google consent banner and the necessary scripts
	 * 
	 * return void
	 */
	public function monsterinsights_show_google_consent_banner() {

		$this->run_country_iso_check();

		if ( isset( $_COOKIE['_monsterinsights_iso'] ) && ! $this->is_eea_country() ) {
			return;
		}

		$stored_consent = isset( $_COOKIE['monsterinsights_consent_mode'] ) ? sanitize_text_field($_COOKIE['monsterinsights_consent_mode']) : null; // phpcs:ignore WordPressVIPMinimum.Variables.RestrictedVariables.cache_constraints___COOKIE

		// By default all the consents are denied.
		$consent = array(
			'ad_personalization' => 'denied',
			'ad_storage'         => 'denied',
			'ad_user_data'       => 'denied',
			'analytics_storage'  => 'denied',
			'wait_for_update'    => 500,
			'regions'            => $this->get_eea_regions(),
		);

		if ($stored_consent === 'granted') {
			$consent = array(
				'ad_personalization' => 'granted',
				'ad_storage'         => 'granted',
				'ad_user_data'       => 'granted',
				'analytics_storage'  => 'granted',
			);
		}

		if ( empty( $stored_consent ) ){
			echo $this->get_banner_template(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped -- Handled in the method.
			echo $this->get_banner_style(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		}
		?>
		<script id='monsterinsights-js-consent-mode'>
			(function() {
				gtag('consent', 'default', <?php echo wp_json_encode( $consent ); ?>);
				<?php
				// If the consent is not stored, then add event listeners to the buttons.
				if ( empty( $stored_consent ) ) { ?>
				document.getElementById('monsterinsights-google-consent-accept')?.addEventListener('click', function(e) {
					e.preventDefault();
					gtag('consent', 'update', {
						'ad_personalization': 'granted',
						'ad_storage': 'granted',
						'ad_user_data': 'granted',
						'analytics_storage': 'granted',
					});
					document.getElementById('monsterinsights-google-consent-banner').remove();
					document.cookie = 'monsterinsights_consent_mode=granted; expires=<?php echo esc_js(time() + YEAR_IN_SECONDS); ?>; path=<?php echo esc_js(COOKIEPATH)?>';
				});

				document.getElementById('monsterinsights-google-consent-deny')?.addEventListener('click', function(e) {
					e.preventDefault();
					const denied = {
						'ad_personalization': 'denied',
						'ad_storage': 'denied',
						'ad_user_data': 'denied',
						'analytics_storage': 'denied',
					};
					gtag('consent', 'update', denied);
					document.cookie = 'monsterinsights_consent_mode=denied; expires=<?php echo esc_js(time() + YEAR_IN_SECONDS); ?>; path=<?php echo esc_js(COOKIEPATH)?>';
					document.getElementById('monsterinsights-google-consent-banner').remove();
				});
				<?php } ?>
				})();
		</script>
		<?php
	}

	/**
	 * First time we load the page we need to check if the user is from EEA region.
	 * We'll run a geolocation check and store id it a cookie.
	 * 
	 * By default the cookie banner is hidden on the first load. if the user is from EEA region, we'll show the banner.
	 */
	private function run_country_iso_check() {
		// cookie already present.
		if ( isset( $_COOKIE['_monsterinsights_iso'] ) ) {
			return;
		}
		?>
		<script>
			(function() {
				window.fetch('https://geo.monsterinsights.com/v3/geolocate/json').then(function(response) {
					return response.json();
				}).then(function(data) {
					document.cookie = '_monsterinsights_iso=' + data.country_iso + '; expires=<?php echo esc_js(time() + YEAR_IN_SECONDS) ?>; path=<?php echo esc_js(COOKIEPATH) ?>';
					
					const eea_countries = <?php echo wp_json_encode( $this->get_eea_regions() ); ?>;

					if ( ! eea_countries.includes( data.country_iso ) ) {
						document.getElementById('monsterinsights-google-consent-banner').style.display = 'hidden';
					}
				});
			})();
		</script>
		<?php
	}

	/**
	 * This method will return the EEA regions ISOs as array.
	 */
	private function get_eea_regions(){
		return $this->eea_countries;
	}

	/**
	 * This method will check if the country is in EEA region.
	 * 
	 * return bool
	 */
	private function is_eea_country() {
		$country_iso = $this->get_country_iso();

		if ( in_array( $country_iso, $this->get_eea_regions() ) ) {
			return true;
		}

		return false;
	}

	/**
	 * This method will try to get the country iso code from the stored cookie.
	 * 
	 * return string|null
	 */
	private function get_country_iso() {
		$iso = isset( $_COOKIE['_monsterinsights_iso'] ) ? sanitize_text_field($_COOKIE['_monsterinsights_iso']) : null; // phpcs:ignore WordPressVIPMinimum.Variables.RestrictedVariables.cache_constraints___COOKIE
		
		return $iso;
	}

	/**
	 * This method will return the banner template.
	 * 
	 * The banner is hidden on the first load and displayed conditionally based on the user's location.
	 * 
	 * return string
	 */
	private function get_banner_template() {
		$style = '';

		if ( ! isset( $_COOKIE['_monsterinsights_iso'] ) ) {
			$style = 'style="display: hidden;"';
		}

		$format = '<div id="monsterinsights-google-consent-banner" %1$s>
			<p>%2$s</p>
			<button id="monsterinsights-google-consent-accept">%3$s</button>
			<button id="monsterinsights-google-consent-deny">%4$s</button>
		</div>';

		return sprintf(
			$format,
			$style,
			__( 'We use cookies to personalize content and ads, to provide social media features, and to analyze our traffic. We also share information about your use of our site with our social media, advertising, and analytics partners.', 'monsterinsights-eu-compliance' ),
			__( 'Allow', 'monsterinsights-eu-compliance' ),
			__( 'Deny', 'monsterinsights-eu-compliance' )
		);
	}

	/**
	 * This method will return the banner style.
	 * 
	 * return string
	 */
	private function get_banner_style() {
		$allow_style = apply_filters( 'monsterinsights_consent_mode_allow_style', true );

		if ( ! $allow_style ) {
			return;
		}

		$format = '<style>
			#monsterinsights-google-consent-banner {
				position: fixed;
				bottom: 0;
				left: 0;
				width: calc(100% - 20px);
				background-color: #f1f1f1;
				color: #333;
				padding: 10px;
				text-align: center;
				z-index: 9999;
			}
			#monsterinsights-google-consent-banner button {
				background-color: #4CAF50;
				color: white;
				padding: 10px 20px;
				border: none;
				cursor: pointer;
				margin: 5px;
			}
			#monsterinsights-google-consent-banner button:hover {
				opacity: 0.8;
			}
		</style>';

		return $format;
	}
}