<?php

namespace Modular\ConnectorDependencies\Illuminate\Console;

use Modular\ConnectorDependencies\Illuminate\Support\Traits\Macroable;
use Modular\ConnectorDependencies\Symfony\Component\Console\Command\Command as SymfonyCommand;
use Modular\ConnectorDependencies\Symfony\Component\Console\Input\InputInterface;
use Modular\ConnectorDependencies\Symfony\Component\Console\Output\OutputInterface;
class Command extends SymfonyCommand
{
    use Concerns\CallsCommands, Concerns\HasParameters, Concerns\InteractsWithIO, Macroable;
    /**
     * The Laravel application instance.
     *
     * @var \Illuminate\Contracts\Foundation\Application
     */
    protected $laravel;
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature;
    /**
     * The console command name.
     *
     * @var string
     */
    protected $name;
    /**
     * The console command description.
     *
     * @var string
     */
    protected $description;
    /**
     * The console command help text.
     *
     * @var string
     */
    protected $help;
    /**
     * Indicates whether the command should be shown in the Artisan command list.
     *
     * @var bool
     */
    protected $hidden = \false;
    /**
     * Create a new console command instance.
     *
     * @return void
     */
    public function __construct()
    {
        // We will go ahead and set the name, description, and parameters on console
        // commands just to make things a little easier on the developer. This is
        // so they don't have to all be manually specified in the constructors.
        if (isset($this->signature)) {
            $this->configureUsingFluentDefinition();
        } else {
            parent::__construct($this->name);
        }
        // Once we have constructed the command, we'll set the description and other
        // related properties of the command. If a signature wasn't used to build
        // the command we'll set the arguments and the options on this command.
        $this->setDescription((string) $this->description);
        $this->setHelp((string) $this->help);
        $this->setHidden($this->isHidden());
        if (!isset($this->signature)) {
            $this->specifyParameters();
        }
    }
    /**
     * Configure the console command using a fluent definition.
     *
     * @return void
     */
    protected function configureUsingFluentDefinition()
    {
        [$name, $arguments, $options] = Parser::parse($this->signature);
        parent::__construct($this->name = $name);
        // After parsing the signature we will spin through the arguments and options
        // and set them on this command. These will already be changed into proper
        // instances of these "InputArgument" and "InputOption" Symfony classes.
        $this->getDefinition()->addArguments($arguments);
        $this->getDefinition()->addOptions($options);
    }
    /**
     * Run the console command.
     *
     * @param  \Symfony\Component\Console\Input\InputInterface  $input
     * @param  \Symfony\Component\Console\Output\OutputInterface  $output
     * @return int
     */
    public function run(InputInterface $input, OutputInterface $output)
    {
        $this->output = $this->laravel->make(OutputStyle::class, ['input' => $input, 'output' => $output]);
        return parent::run($this->input = $input, $this->output);
    }
    /**
     * Execute the console command.
     *
     * @param  \Symfony\Component\Console\Input\InputInterface  $input
     * @param  \Symfony\Component\Console\Output\OutputInterface  $output
     * @return int
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $method = method_exists($this, 'handle') ? 'handle' : '__invoke';
        return (int) $this->laravel->call([$this, $method]);
    }
    /**
     * Resolve the console command instance for the given command.
     *
     * @param  \Symfony\Component\Console\Command\Command|string  $command
     * @return \Symfony\Component\Console\Command\Command
     */
    protected function resolveCommand($command)
    {
        if (!class_exists($command)) {
            return $this->getApplication()->find($command);
        }
        $command = $this->laravel->make($command);
        if ($command instanceof SymfonyCommand) {
            $command->setApplication($this->getApplication());
        }
        if ($command instanceof self) {
            $command->setLaravel($this->getLaravel());
        }
        return $command;
    }
    /**
     * {@inheritdoc}
     *
     * @return bool
     */
    public function isHidden()
    {
        return $this->hidden;
    }
    /**
     * {@inheritdoc}
     *
     * @return static
     */
    public function setHidden(bool $hidden)
    {
        parent::setHidden($this->hidden = $hidden);
        return $this;
    }
    /**
     * Get the Laravel application instance.
     *
     * @return \Illuminate\Contracts\Foundation\Application
     */
    public function getLaravel()
    {
        return $this->laravel;
    }
    /**
     * Set the Laravel application instance.
     *
     * @param  \Illuminate\Contracts\Container\Container  $laravel
     * @return void
     */
    public function setLaravel($laravel)
    {
        $this->laravel = $laravel;
    }
}
