<?php

namespace Modular\ConnectorDependencies\Ares\Framework\Foundation;

use Modular\ConnectorDependencies\Ares\Framework\Foundation\Compatibilities\Compatibilities;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Log;
/**
 * Simulates WordPress admin environment for plugin/theme management operations.
 *
 * This class is used by Manager jobs (ManagerInstallJob, ManagerUpdateJob, ManagerManageItemJob)
 * to simulate the wp-admin environment needed for WordPress upgrader operations.
 *
 * Usage:
 *   ScreenSimulation::simulateAdmin();  // Call at start of job handle()
 *   ScreenSimulation::includeUpgrader(); // Called by Manager services as needed
 *
 * @see \Modular\Connector\Jobs\ManagerInstallJob
 * @see \Modular\Connector\Jobs\ManagerUpdateJob
 * @see \Modular\Connector\Jobs\ManagerManageItemJob
 */
class ScreenSimulation
{
    /**
     * Indicates if the admin environment has been simulated.
     */
    private static bool $adminSimulated = \false;
    /**
     * Simulate WordPress admin environment for plugin/theme management.
     *
     * Call this at the start of Manager jobs that need to install, update,
     * activate, or delete plugins/themes.
     *
     * @param string $path The admin page to simulate (default: update-core.php)
     */
    public static function simulateAdmin(string $path = 'update-core.php'): void
    {
        if (self::$adminSimulated) {
            return;
        }
        self::$adminSimulated = \true;
        Log::debug('ScreenSimulation: Simulating admin environment', ['path' => $path]);
        // Simulate being in wp-admin
        $_SERVER['PHP_SELF'] = '/wp-admin/' . $path;
        if (defined('FORCE_SSL_ADMIN') && \FORCE_SSL_ADMIN) {
            $_SERVER['HTTPS'] = 'on';
            $_SERVER['SERVER_PORT'] = '443';
        }
        if (!isset($GLOBALS['pagenow'])) {
            $GLOBALS['pagenow'] = $path;
        }
        if (!isset($GLOBALS['hook_suffix'])) {
            $GLOBALS['hook_suffix'] = '';
        }
        // Include WordPress screen classes
        if (!class_exists('WP_Screen')) {
            require_once \ABSPATH . 'wp-admin/includes/class-wp-screen.php';
        }
        if (!function_exists('set_current_screen')) {
            require_once \ABSPATH . 'wp-admin/includes/screen.php';
        }
        if (!function_exists('add_meta_box')) {
            require_once \ABSPATH . 'wp-admin/includes/template.php';
        }
        // Apply general compatibility fixes for plugins that interfere with updates
        self::applyCompatibilityFixes();
        // Logout user when request terminates
        \Modular\ConnectorDependencies\app()->terminating(function () {
            ServerSetup::logout();
        });
    }
    /**
     * Makes the necessary WordPress upgrader includes
     * to handle plugin and themes functionality.
     */
    public static function includeUpgrader(): void
    {
        if (!function_exists('wp_update_plugins') || !function_exists('wp_update_themes')) {
            ob_start();
            require_once \ABSPATH . 'wp-admin/includes/update.php';
            ob_end_flush();
            ob_end_clean();
        }
        if (!class_exists('WP_Upgrader')) {
            require_once \ABSPATH . 'wp-admin/includes/class-wp-upgrader.php';
        }
        if (!function_exists('wp_install')) {
            require_once \ABSPATH . 'wp-admin/includes/upgrade.php';
        }
        if (!function_exists('plugins_api')) {
            require_once \ABSPATH . 'wp-admin/includes/plugin-install.php';
        }
        if (empty($GLOBALS['wp_filesystem'])) {
            WP_Filesystem();
        }
        if (empty($GLOBALS['wp_theme_directories'])) {
            register_theme_directory(get_theme_root());
        }
    }
    /**
     * Apply general compatibility fixes for plugins that interfere with updates.
     */
    private static function applyCompatibilityFixes(): void
    {
        $compatibilityFixes = Compatibilities::getCompatibilityFixes();
        array_walk($compatibilityFixes, fn($compatibility) => $compatibility::fix());
    }
}
