<?php

namespace Modular\ConnectorDependencies\Ares\Framework\Foundation\Http;

use Modular\ConnectorDependencies\Illuminate\Support\Facades\Cache;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Log;
use Modular\ConnectorDependencies\Illuminate\Support\Str;
class HttpUtils
{
    /**
     * Get memory limit in MB.
     *
     * @return int
     */
    public static function currentMemoryLimit(): int
    {
        if (function_exists('ini_get')) {
            $memoryLimit = ini_get('memory_limit');
        } else {
            $memoryLimit = '256M';
        }
        if (!$memoryLimit) {
            $memoryLimit = '256M';
        }
        return $memoryLimit !== -1 ? wp_convert_hr_to_bytes($memoryLimit) : $memoryLimit;
    }
    /**
     * @param bool $inMegabytes
     * @return int
     */
    public static function maxMemoryLimit(bool $inMegabytes = \false): int
    {
        $currentMemoryLimit = HttpUtils::currentMemoryLimit();
        $maxMemoryLimit = defined('WP_MAX_MEMORY_LIMIT') ? \WP_MAX_MEMORY_LIMIT : '512M';
        if (!$maxMemoryLimit || intval($maxMemoryLimit) === -1) {
            // Unlimited, set to 3GB.
            $maxMemoryLimit = '3200M';
        }
        $maxMemoryLimit = wp_convert_hr_to_bytes($maxMemoryLimit);
        if ($maxMemoryLimit > $currentMemoryLimit) {
            $currentMemoryLimit = $maxMemoryLimit;
        }
        return $inMegabytes ? $currentMemoryLimit / 1024 / 1024 : $currentMemoryLimit;
    }
    /**
     * Check if this is a valid cron request.
     * Must be from wp-cron.php AND have DOING_CRON defined.
     *
     * @return bool
     */
    public static function isCron(): bool
    {
        if (!defined('DOING_CRON') || !\DOING_CRON) {
            return \false;
        }
        $request = \Modular\ConnectorDependencies\app('request');
        $scriptName = $request->server('SCRIPT_NAME', $request->server('PHP_SELF', ''));
        return Str::endsWith($scriptName, 'wp-cron.php');
    }
    /**
     * Validate that the request comes from wp-load.php only.
     *
     * @return bool
     */
    public static function isValidEntryPoint(): bool
    {
        $request = \Modular\ConnectorDependencies\app('request');
        $scriptName = $request->server('SCRIPT_NAME', $request->server('PHP_SELF', ''));
        if (!Str::endsWith($scriptName, 'wp-load.php')) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - invalid entry point', ['script_name' => $scriptName, 'expected' => 'wp-load.php']);
            return \false;
        }
        return \true;
    }
    /**
     * Validate User-Agent header matches expected pattern.
     *
     * @return bool
     */
    public static function isValidUserAgent(): bool
    {
        $request = \Modular\ConnectorDependencies\app('request');
        $userAgent = $request->userAgent() ?? '';
        if (!Str::is('ModularConnector/* (Linux)', $userAgent)) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - invalid User-Agent', ['user_agent' => $userAgent, 'expected_pattern' => 'ModularConnector/* (Linux)']);
            return \false;
        }
        return \true;
    }
    /**
     * @return bool
     */
    public static function isDirectRequest(): bool
    {
        $request = \Modular\ConnectorDependencies\app('request');
        // Validate entry point is wp-load.php only (prevents wp-cron.php, /wp-json/, etc.)
        if (!self::isValidEntryPoint()) {
            return \false;
        }
        // Validate REAL HTTP method is GET (prevents X-HTTP-Method-Override)
        if ($request->getRealMethod() !== 'GET') {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - invalid HTTP method', ['real_method' => $request->getRealMethod(), 'spoofed_method' => $request->method()]);
            return \false;
        }
        // Must have origin=mo (fast check, most requests will fail here)
        if ($request->get('origin') !== 'mo') {
            return \false;
        }
        // Type: must be in query OR header, but not both
        $hasTypeQuery = $request->has('type');
        $hasTypeHeader = $request->hasHeader('x-mo-type');
        if (!$hasTypeQuery && !$hasTypeHeader) {
            return \false;
        }
        if ($hasTypeQuery && $hasTypeHeader) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - type in both query and header');
            return \false;
        }
        $type = $hasTypeQuery ? $request->get('type') : $request->header('x-mo-type');
        // Validate that type only contains allowed values
        $allowedTypes = ['lb', 'request', 'oauth'];
        if (!in_array($type, $allowedTypes, \true)) {
            return \false;
        }
        $allParams = array_keys($request->query->all());
        // Validate all query parameters are strings (prevents array ?type[]=x)
        foreach ($allParams as $param) {
            $value = $request->get($param);
            if (!is_string($value)) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - parameter is not a string', ['param' => $param, 'type' => gettype($value)]);
                return \false;
            }
        }
        $hasAuthHeader = $request->hasHeader('x-mo-authentication');
        $hasSigParam = $request->has('sig');
        // Mutual exclusivity: cannot have both header and sig
        if ($hasAuthHeader && $hasSigParam) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - cannot use both x-mo-authentication and sig');
            return \false;
        }
        // User-Agent validation only when sig is NOT present (internal loopback requests)
        if (!$hasSigParam && !self::isValidUserAgent()) {
            return \false;
        }
        if ($type === 'request') {
            // type=request: requires either x-mo-authentication header OR sig parameter (not both)
            if (!$hasAuthHeader && !$hasSigParam) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - authentication required (header or sig)', ['type' => $type]);
                return \false;
            }
        } else if (!$hasAuthHeader) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - x-mo-authentication header required', ['type' => $type]);
            return \false;
        }
        // mrid: can be in query OR header, but not both (only for type=request)
        $hasMridQuery = $request->has('mrid');
        $hasMridHeader = $request->hasHeader('x-mo-mrid');
        if ($hasMridQuery && $hasMridHeader) {
            \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - mrid in both query and header');
            return \false;
        }
        // For type=request, mrid is required (in query OR header) and must be UUID
        if ($type === 'request') {
            if (!$hasMridQuery && !$hasMridHeader) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - mrid required for type=request');
                return \false;
            }
            $mrid = $hasMridQuery ? $request->get('mrid') : $request->header('x-mo-mrid');
            if (!Str::isUuid($mrid)) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - mrid must be a valid UUID', ['mrid' => $mrid]);
                return \false;
            }
        }
        // code validation (only for type=oauth)
        $hasCodeQuery = \false;
        if ($type === 'oauth') {
            $hasCodeQuery = $request->has('code');
            $hasCodeHeader = $request->hasHeader('x-mo-code');
            // code: can be in query OR header, but not both
            if ($hasCodeQuery && $hasCodeHeader) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - code in both query and header');
                return \false;
            }
            // code is required (in query OR header)
            if (!$hasCodeQuery && !$hasCodeHeader) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - code required for type=oauth');
                return \false;
            }
        }
        // Define allowed query parameters per type (exclude params that come via header)
        $baseParams = ['origin'];
        if ($hasTypeQuery) {
            $baseParams[] = 'type';
        }
        switch ($type) {
            case 'request':
                $allowedParams = $baseParams;
                if ($hasMridQuery) {
                    $allowedParams[] = 'mrid';
                }
                if ($hasSigParam) {
                    $allowedParams[] = 'sig';
                }
                break;
            case 'lb':
                $allowedParams = array_merge($baseParams, ['lbn']);
                break;
            case 'oauth':
                $allowedParams = $baseParams;
                if ($hasCodeQuery) {
                    $allowedParams[] = 'code';
                }
                $allowedParams[] = 'state';
                break;
            default:
                return \false;
        }
        // Validate all parameters: must be allowed and not empty (except 'state')
        foreach ($allParams as $param) {
            if (!in_array($param, $allowedParams, \true)) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - unexpected parameter', ['param' => $param, 'allowed' => $allowedParams]);
                return \false;
            }
            // 'state' can be empty
            if ($param === 'state') {
                continue;
            }
            $value = $request->get($param);
            if ($value === null || $value === '') {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - empty required parameter', ['param' => $param]);
                return \false;
            }
        }
        // Validate lbn format for loopback requests (32 hex chars)
        if ($type === 'lb') {
            $lbn = $request->get('lbn');
            if (!preg_match('/^[a-f0-9]{32}$/i', $lbn)) {
                \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Rejected - invalid lbn format', ['lbn' => $lbn, 'expected' => '32 hex characters']);
                return \false;
            }
        }
        \Modular\ConnectorDependencies\app('log')->debug('isDirectRequest: Request validated successfully', ['type' => $type]);
        return \true;
    }
    /**
     * @return bool
     */
    public static function isMultisite()
    {
        return is_multisite();
    }
    /**
     * @return bool
     */
    public static function isMuPlugin()
    {
        return \Modular\ConnectorDependencies\data_get($GLOBALS, 'modular_is_mu_plugin', \false);
    }
    /**
     * @return void
     */
    public static function forceCloseConnection(): void
    {
        ignore_user_abort(\true);
        if (\function_exists('fastcgi_finish_request')) {
            fastcgi_finish_request();
        } elseif (\function_exists('litespeed_finish_request')) {
            litespeed_finish_request();
        }
    }
    /**
     * @return void
     */
    public static function restartQueue(int $timestamp)
    {
        Cache::forever('illuminate:queue:restart', $timestamp);
        \Modular\ConnectorDependencies\app('log')->info('Broadcasting queue restart signal.');
    }
}
