<?php

namespace Modular\ConnectorDependencies\Ares\Framework\Foundation;

use Modular\ConnectorDependencies\Ares\Framework\Foundation\Compatibilities\LoginCompatibilities;
use Modular\ConnectorDependencies\Ares\Framework\Foundation\Http\HttpUtils;
use Modular\ConnectorDependencies\Ares\Framework\Foundation\Routing\Router;
use Modular\ConnectorDependencies\Illuminate\Contracts\Foundation\Application as ApplicationContract;
use Modular\ConnectorDependencies\Illuminate\Contracts\Http\Kernel as IlluminateHttpKernel;
use Modular\ConnectorDependencies\Illuminate\Http\Request;
use Modular\ConnectorDependencies\Illuminate\Http\Response;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Facade;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Log;
/**
 * Class Bootloader inspired by Roots (Acorn).
 *
 * @link https://github.com/roots/acorn/blob/4.x/src/Roots/Acorn/Bootloader.php
 */
class Bootloader
{
    /**
     * The Bootloader instance.
     */
    protected static $instance;
    /**
     * @var ApplicationContract|null
     */
    protected ?ApplicationContract $app;
    /**
     * @var string
     */
    protected string $basePath;
    /**
     * @param 1ApplicationContract|null $app
     */
    public function __construct(?ApplicationContract $app = null)
    {
        $this->app = $app;
    }
    /**
     * Get the Bootloader instance.
     */
    public static function getInstance(?ApplicationContract $app = null): self
    {
        return static::$instance ??= new static($app);
    }
    /**
     * Initialize and retrieve the Application instance.
     */
    public function getApplication(): ApplicationContract
    {
        $this->app ??= new Application($this->basePath);
        $this->app->singleton('router', Router::class);
        return $this->app;
    }
    /**
     * @return void
     */
    public function configRequest()
    {
        // Ensure the script continues to run even if the user aborts the connection.
        if (function_exists('set_time_limit')) {
            @set_time_limit(600);
        }
        if (function_exists('ignore_user_abort')) {
            @ignore_user_abort(\true);
        }
        $maxMemoryLimit = HttpUtils::maxMemoryLimit();
        if (function_exists('ini_set')) {
            @ini_set('memory_limit', $maxMemoryLimit);
            @ini_set('display_errors', \false);
        }
    }
    /**
     * Handle the request.
     */
    protected function handleRequest(IlluminateHttpKernel $kernel, Request $request): void
    {
        /**
         * @var IlluminateHttpKernel $kernel
         */
        $response = $kernel->handle($request);
        $body = $response->send();
        $kernel->terminate($request, $body);
        exit((int) $response->isServerError());
    }
    /**
     * Register the request handler.
     *
     * @param IlluminateHttpKernel $kernel
     * @param Request $request
     * @return void
     */
    protected function registerRequestHandler(IlluminateHttpKernel $kernel, Request $request): void
    {
        $isDirectRequest = HttpUtils::isDirectRequest();
        // Handle the request if the route exists
        $hook = $isDirectRequest ? 'wp_loaded' : 'parse_request';
        add_action($hook, fn() => $this->handleRequest($kernel, $request));
    }
    /**
     * Disable WordPress automatic actions during direct requests.
     *
     * Prevents WordPress from:
     * - Redirecting (we use Laravel Response for redirects)
     * - Running auto-updates during our operations
     * - Running wp_cron during our requests
     */
    protected function disableWordPressAutoActions(): void
    {
        // Prevent WordPress redirects (we use Laravel Response)
        add_filter('wp_redirect', '__return_false');
        // Disable auto updates during our operations
        add_filter('auto_update_core', '__return_false', \PHP_INT_MAX);
        add_filter('auto_update_translation', '__return_false', \PHP_INT_MAX);
        add_filter('auto_update_theme', '__return_false', \PHP_INT_MAX);
        add_filter('auto_update_plugin', '__return_false', \PHP_INT_MAX);
        add_filter('automatic_updater_disabled', '__return_true');
        // Prevent wp_cron from running during our requests
        remove_action('init', 'wp_cron');
    }
    /**
     * @param IlluminateHttpKernel $kernel
     * @param Request $request
     * @return void
     * @link https://github.com/deliciousbrains/wp-background-processing/blob/master/classes/wp-async-request.php Method inspired by wp-background-processing
     */
    protected function bootCron(IlluminateHttpKernel $kernel, Request $request): void
    {
        if (!HttpUtils::isCron()) {
            return;
        }
        // When is a cron request, WP takes care of forcing the shutdown to proxy server (nginx, apache)
        remove_action('shutdown', 'wp_ob_end_flush_all', 1);
        add_action('shutdown', fn() => $this->handleRequest($kernel, $request), 100);
    }
    /**
     * Boot the Application for HTTP requests.
     *
     * Only two entry points are allowed:
     * - wp-cron.php with DOING_CRON (cron requests)
     * - wp-load.php with valid params (direct requests)
     *
     * @param IlluminateHttpKernel $kernel
     * @param Request $request
     * @return void
     */
    protected function bootHttp(IlluminateHttpKernel $kernel, Request $request): void
    {
        $isCron = HttpUtils::isCron();
        $isDirectRequest = HttpUtils::isDirectRequest();
        // Only process cron (wp-cron.php) or direct request (wp-load.php)
        if (!$isCron && !$isDirectRequest) {
            return;
        }
        Log::debug('Bootloader: Processing request', ['is_cron' => $isCron, 'is_direct_request' => $isDirectRequest]);
        $this->configRequest();
        // Apply login compatibility fixes early (before plugins_loaded)
        // so they can register their own plugins_loaded actions
        if ($isDirectRequest) {
            $this->disableWordPressAutoActions();
            LoginCompatibilities::beforeLogin();
        }
        add_action('plugins_loaded', function () use ($kernel, $request, $isCron, $isDirectRequest) {
            try {
                $route = apply_filters('ares/routes/match', \false);
                if ($isDirectRequest) {
                    if ($route->getName() === 'default') {
                        \Modular\ConnectorDependencies\abort(404);
                    }
                    add_action('after_setup_theme', fn() => $this->registerRequestHandler($kernel, $request), 0);
                } elseif ($isCron) {
                    $this->bootCron($kernel, $request);
                }
            } catch (\Throwable $e) {
                throw $e;
            }
        });
    }
    public function boot(string $basePath, $callback)
    {
        if (!defined('LARAVEL_START')) {
            define('LARAVEL_START', microtime(\true));
        }
        $this->basePath = $basePath;
        $callback($this->getApplication());
        if ($this->app->hasBeenBootstrapped()) {
            return;
        }
        /**
         * @var IlluminateHttpKernel $kernel
         */
        $kernel = $this->app->make(IlluminateHttpKernel::class);
        $request = Request::capture();
        $this->app->instance('request', $request);
        Facade::clearResolvedInstance('request');
        $kernel->bootstrap();
        $this->bootHttp($kernel, $request);
    }
}
