<?php

namespace Modular\Connector\Providers;

use Modular\Connector\Helper\OauthClient;
use Modular\ConnectorDependencies\Ares\Framework\Foundation\Http\HttpUtils;
use Modular\ConnectorDependencies\GuzzleHttp\Exception\ClientException;
use Modular\ConnectorDependencies\GuzzleHttp\Exception\ServerException;
use Modular\ConnectorDependencies\Illuminate\Foundation\Support\Providers\RouteServiceProvider as ServiceProvider;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Cache;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Log;
use Modular\ConnectorDependencies\Illuminate\Support\Facades\Route;
use Modular\ConnectorDependencies\Illuminate\Support\Str;
use Modular\ConnectorDependencies\Symfony\Component\HttpKernel\Exception\HttpException;
use function Modular\ConnectorDependencies\app;
use function Modular\ConnectorDependencies\base_path;
use function Modular\ConnectorDependencies\request;

class RouteServiceProvider extends ServiceProvider
{
    /**
     * @param string $requestId
     * @return \Modular\SDK\Objects\BaseObject|null
     * @throws \ErrorException
     */
    private function getModularRequest(string $requestId)
    {
        $client = OauthClient::getClient();
        $client->validateOrRenewAccessToken();

        try {
            return $client->wordpress->handleRequest($requestId);
        } catch (ClientException|ServerException $e) {
            throw new HttpException($e->getResponse()->getStatusCode(), $e->getMessage());
        } catch (\Throwable $e) {
            // Silence is golden
            return null;
        }
    }

    /**
     * Get type from request (header XOR query, never both).
     *
     * @param \Illuminate\Http\Request $request
     * @return string|null
     */
    private function getTypeFromRequest($request): ?string
    {
        $hasQuery = $request->has('type');
        $hasHeader = $request->hasHeader('x-mo-type');

        // XOR: one or the other, never both (already validated in isDirectRequest)
        if ($hasQuery) {
            return $request->get('type');
        }

        if ($hasHeader) {
            return $request->header('x-mo-type');
        }

        return null;
    }

    /**
     * Get mrid from request (header XOR query, never both).
     *
     * @param \Illuminate\Http\Request $request
     * @return string|null
     */
    private function getMridFromRequest($request): ?string
    {
        $hasQuery = $request->has('mrid');
        $hasHeader = $request->hasHeader('x-mo-mrid');

        if ($hasQuery) {
            return $request->get('mrid');
        }

        if ($hasHeader) {
            return $request->header('x-mo-mrid');
        }

        return null;
    }

    /**
     * @param $route
     * @param $removeQuery
     * @return \Illuminate\Routing\Route|mixed
     * @throws \Psr\Container\ContainerExceptionInterface
     * @throws \Psr\Container\NotFoundExceptionInterface
     */
    public function bindOldRoutes($removeQuery = false)
    {
        $routes = app('router')->getRoutes();

        $route = $routes->getByName('default');
        $route->bind(request());

        if (!HttpUtils::isDirectRequest()) {
            return $route;
        }

        $request = request();
        $type = $this->getTypeFromRequest($request);

        if ($type === 'request') {
            $mrid = $this->getMridFromRequest($request);

            // Defense in depth: validate mrid is UUID (already validated in isDirectRequest)
            if (!$mrid || !Str::isUuid($mrid)) {
                Log::warning('bindOldRoutes: Invalid mrid format', ['mrid' => $mrid]);

                return $route;
            }

            $modularRequest = Cache::driver('array')->get('modularRequest') ?: $this->getModularRequest($mrid);

            if (!$modularRequest) {
                return $route;
            }

            if (!$removeQuery && !Cache::driver('array')->has('modularRequest')) {
                Cache::driver('array')->set('modularRequest', $modularRequest);
            } elseif ($removeQuery) {
                Cache::driver('array')->forget('modularRequest');
            }

            $routeType = $modularRequest->type;

            /** @var \Illuminate\Routing\Route $routeByName */
            $routeByName = $routes->getByName($routeType);

            if (!$routeByName) {
                Log::warning('bindOldRoutes: Route not found', ['route_type' => $routeType]);

                return $route;
            }

            $route = $routeByName;

            if ($removeQuery) {
                $request->query->remove('origin');
                $request->query->remove('type');
                $request->query->remove('mrid');
            }

            $route = $route->bind($request);

            if ($removeQuery) {
                $route->setParameter('modular_request', $modularRequest);
            }
        } elseif ($type === 'oauth') {
            /**
             * @var \Illuminate\Routing\Route $route
             */
            $route = $routes->getByName('modular-connector.oauth');

            if ($removeQuery) {
                $request->query->remove('origin');
                $request->query->remove('type');
                $request->query->remove('state');
            }

            $route = $route->bind($request);
        } elseif ($type === 'lb') {
            /**
             * @var \Illuminate\Routing\Route $route
             */
            $route = $routes->getByName('schedule.run');

            if ($removeQuery) {
                $request->query->remove('origin');
                $request->query->remove('type');
            }

            $route = $route->bind($request);
        } else {
            // Unknown type - should never reach here if isDirectRequest() works correctly
            Log::warning('bindOldRoutes: Unknown type after isDirectRequest passed', ['type' => $type]);

            return $route;
        }

        return $route;
    }

    /**
     * Define your route model bindings, pattern filters, and other route configuration.
     *
     * @return void
     */
    public function boot()
    {
        $this->routes(function () {
            Route::prefix('/api/modular-connector')
                ->group(base_path('routes/api.php'));
        });

        add_filter('ares/routes/match', [$this, 'bindOldRoutes'], 10, 2);
    }
}
