<?php

declare(strict_types=1);

namespace MailerPress\Services;

use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Kernel;
use MailerPress\Core\Migrations\CustomTableManager;
use MailerPress\Core\Migrations\Manager;
use MailerPress\Core\Migrations\MigrationTracker;
use MailerPress\Core\Migrations\SchemaBuilder;
use MailerPress\Services\DatabaseRepairLogger;

\defined('ABSPATH') || exit;

/**
 * Service for database diagnosis
 * Analyzes the state of the database and detects problems
 */
class DatabaseDiagnostic
{
    protected string $migrationPath;
    protected MigrationTracker $tracker;

    public function __construct()
    {
        $this->migrationPath = Kernel::$config['root'] . '/src/Core/Migrations/migrations';
        $this->tracker = new MigrationTracker();
    }

    /**
     * Performs a complete diagnosis of the database
     */
    public function diagnose(): array
    {
        global $wpdb;

        $issues = [];
        $tables = [];
        $migrationStatus = [];

        // 1. Check the status of migrations
        $migrationStatus = $this->checkMigrationStatus();

        // 2. Analyze all expected tables
        $expectedTables = $this->getExpectedTables();

        foreach ($expectedTables as $tableName) {
            $fullTableName = Tables::get($tableName);
            $tableInfo = $this->analyzeTable($fullTableName, $tableName);

            if (!empty($tableInfo['issues'])) {
                $issues = array_merge($issues, $tableInfo['issues']);
            }

            $tables[$tableName] = [
                'exists' => $tableInfo['exists'],
                'issues' => $tableInfo['issues'],
                'columns' => $tableInfo['columns'],
                'expected_columns' => $tableInfo['expected_columns'] ?? [],
                'missing_columns' => $tableInfo['missing_columns'] ?? [],
                'indexes' => $tableInfo['indexes'],
                'expected_indexes' => $tableInfo['expected_indexes'] ?? [],
                'missing_indexes' => $tableInfo['missing_indexes'] ?? [],
                'foreign_keys' => $tableInfo['foreign_keys'],
                'expected_foreign_keys' => $tableInfo['expected_foreign_keys'] ?? [],
                'missing_foreign_keys' => $tableInfo['missing_foreign_keys'] ?? [],
            ];
        }

        // 3. Check the tracking table of migrations
        $trackerTable = $this->tracker->getTableName();
        $trackerExists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->esc_like($trackerTable))
        ) === $trackerTable;

        if (!$trackerExists) {
            $issues[] = [
                'type' => 'critical',
                'table' => 'mailerpress_migrations',
                'issue' => 'missing_table',
                'message' => __('The migrations tracking table is missing', 'mailerpress'),
            ];
        }

        // 4. Check the failed migrations
        $failedMigrations = $this->tracker->getFailedMigrations();
        if (!empty($failedMigrations)) {
            foreach ($failedMigrations as $migration) {
                $issues[] = [
                    'type' => 'error',
                    'table' => 'migrations',
                    'issue' => 'failed_migration',
                    'message' => sprintf(
                        __('Migration failed: %s - %s', 'mailerpress'),
                        $migration['migration_name'],
                        $migration['error_message'] ?? __('Unknown error', 'mailerpress')
                    ),
                    'migration_file' => $migration['migration_file'],
                    'error_message' => $migration['error_message'],
                ];
            }
        }

        // 5. Check if the migrations are locked
        $manager = new Manager($this->migrationPath, []);
        $status = $manager->getStatus();
        if ($status['is_locked']) {
            $issues[] = [
                'type' => 'warning',
                'table' => 'migrations',
                'issue' => 'locked',
                'message' => __('Migrations are currently locked', 'mailerpress'),
            ];
        }

        return [
            'healthy' => empty($issues),
            'issues' => $issues,
            'tables' => $tables,
            'migration_status' => $migrationStatus,
            'summary' => [
                'total_tables' => count($expectedTables),
                'existing_tables' => count(array_filter($tables, fn($t) => $t['exists'])),
                'missing_tables' => count(array_filter($tables, fn($t) => !$t['exists'])),
                'total_issues' => count($issues),
                'critical_issues' => count(array_filter($issues, fn($i) => $i['type'] === 'critical')),
                'errors' => count(array_filter($issues, fn($i) => $i['type'] === 'error')),
                'warnings' => count(array_filter($issues, fn($i) => $i['type'] === 'warning')),
            ],
        ];
    }

    /**
     * Analyze a specific table and compare with the expected structure
     */
    protected function analyzeTable(string $fullTableName, string $tableName): array
    {
        global $wpdb;

        $result = [
            'exists' => false,
            'issues' => [],
            'columns' => [],
            'expected_columns' => [],
            'missing_columns' => [],
            'indexes' => [],
            'expected_indexes' => [],
            'missing_indexes' => [],
            'foreign_keys' => [],
            'expected_foreign_keys' => [],
            'missing_foreign_keys' => [],
        ];

        // Check if the table exists
        $tableExists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->esc_like($fullTableName))
        ) === $fullTableName;

        $result['exists'] = $tableExists;

        if (!$tableExists) {
            $result['issues'][] = [
                'type' => 'critical',
                'table' => $tableName,
                'issue' => 'missing_table',
                'message' => sprintf(__('Table %s is missing', 'mailerpress'), $tableName),
            ];
            return $result;
        }

        // Get the expected structure from the migrations
        $expectedStructure = $this->getExpectedTableStructure($tableName);

        DatabaseRepairLogger::info(__("Expected structure for {$tableName}", 'mailerpress'), [
            'expected_columns_count' => count($expectedStructure['columns'] ?? []),
            'expected_indexes_count' => count($expectedStructure['indexes'] ?? []),
            'expected_foreign_keys_count' => count($expectedStructure['foreign_keys'] ?? []),
        ]);

        // Analyze the current columns
        try {
            $columns = $wpdb->get_results("SHOW COLUMNS FROM {$fullTableName}", ARRAY_A);
            $actualColumns = array_map(fn($col) => $col['Field'], $columns);
            $result['columns'] = $actualColumns;

            // Compare with the expected columns
            if (!empty($expectedStructure['columns'])) {
                $result['expected_columns'] = $expectedStructure['columns'];
                $missingColumns = array_diff($expectedStructure['columns'], $actualColumns);
                $result['missing_columns'] = array_values($missingColumns);

                DatabaseRepairLogger::info(__("Column comparison for {$tableName}", 'mailerpress'), [
                    'expected' => count($expectedStructure['columns']),
                    'actual' => count($actualColumns),
                    'missing' => count($result['missing_columns']),
                    'missing_list' => $result['missing_columns'],
                ]);

                foreach ($missingColumns as $missingCol) {
                    $result['issues'][] = [
                        'type' => 'critical',
                        'table' => $tableName,
                        'issue' => 'missing_column',
                        'message' => sprintf(__('Missing column in %s: %s', 'mailerpress'), $tableName, $missingCol),
                        'column' => $missingCol,
                    ];
                }
            }
        } catch (\Throwable $e) {
            $result['issues'][] = [
                'type' => 'error',
                'table' => $tableName,
                'issue' => 'column_check_failed',
                'message' => sprintf(__('Unable to check columns: %s', 'mailerpress'), $e->getMessage()),
            ];
        }

        // Analyze the current indexes
        try {
            $indexes = $wpdb->get_results("SHOW INDEX FROM {$fullTableName}", ARRAY_A);
            $indexGroups = [];
            foreach ($indexes as $index) {
                $keyName = $index['Key_name'];
                if ($keyName === 'PRIMARY') {
                    continue; // Ignore the primary key
                }
                if (!isset($indexGroups[$keyName])) {
                    $indexGroups[$keyName] = [];
                }
                $indexGroups[$keyName][] = strtolower($index['Column_name']);
            }
            $result['indexes'] = $indexGroups;

            // Compare with the expected indexes
            if (!empty($expectedStructure['indexes'])) {
                $result['expected_indexes'] = $expectedStructure['indexes'];
                $actualIndexKeys = array_keys($indexGroups);

                DatabaseRepairLogger::info(__("Index comparison for {$tableName}", 'mailerpress'), [
                    'expected' => count($expectedStructure['indexes']),
                    'actual' => count($indexGroups),
                    'expected_details' => $expectedStructure['indexes'],
                    'actual_details' => $indexGroups,
                ]);

                foreach ($expectedStructure['indexes'] as $expectedIndex) {
                    $indexFound = false;
                    $expectedCols = is_array($expectedIndex['columns'])
                        ? array_map('strtolower', $expectedIndex['columns'])
                        : [strtolower($expectedIndex['columns'])];

                    // Sort the expected columns for comparison
                    $expectedColsSorted = $expectedCols;
                    sort($expectedColsSorted);

                    // Check if an index exists with the same columns (same order or different order)
                    foreach ($indexGroups as $indexName => $indexCols) {
                        $normalizedCols = array_map('strtolower', $indexCols);
                        $normalizedColsSorted = $normalizedCols;
                        sort($normalizedColsSorted);

                        // Compare the exact order first (more strict)
                        $exactMatch = (
                            count($normalizedCols) === count($expectedCols) &&
                            $normalizedCols === $expectedCols
                        );

                        // If no exact match, compare the sorted columns (different order but same columns)
                        $sortedMatch = (
                            count($normalizedColsSorted) === count($expectedColsSorted) &&
                            $normalizedColsSorted === $expectedColsSorted
                        );

                        if ($exactMatch || $sortedMatch) {
                            $indexFound = true;
                            DatabaseRepairLogger::info(__("Index found: {$indexName} corresponds to the expected index", 'mailerpress'), [
                                'table' => $tableName,
                                'existing_index' => $indexName,
                                'existing_columns' => $indexCols,
                                'expected_columns' => $expectedIndex['columns'],
                                'exact_match' => $exactMatch,
                                'sorted_match' => $sortedMatch,
                            ]);
                            break;
                        }
                    }

                    if (!$indexFound) {
                        $result['missing_indexes'][] = $expectedIndex;
                        $indexDesc = is_array($expectedIndex['columns'])
                            ? implode(', ', $expectedIndex['columns'])
                            : $expectedIndex['columns'];
                        $result['issues'][] = [
                            'type' => 'error',
                            'table' => $tableName,
                            'issue' => 'missing_index',
                            'message' => sprintf(
                                __('Missing index in %s: %s (%s)', 'mailerpress'),
                                $tableName,
                                $expectedIndex['type'],
                                $indexDesc
                            ),
                            'index' => $expectedIndex,
                        ];
                        DatabaseRepairLogger::warning(__("Missing index detected for {$tableName}", 'mailerpress'), [
                            'index' => $expectedIndex,
                            'expected_columns' => $expectedCols,
                            'existing_indexes' => $indexGroups,
                            'total_missing' => count($result['missing_indexes']),
                        ]);
                    }
                }
            }
        } catch (\Throwable $e) {
            $result['issues'][] = [
                'type' => 'warning',
                'table' => $tableName,
                'issue' => 'index_check_failed',
                'message' => sprintf(__('Unable to check indexes: %s', 'mailerpress'), $e->getMessage()),
            ];
        }

        // Analyze the current foreign keys
        try {
            $foreignKeys = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT CONSTRAINT_NAME, COLUMN_NAME, REFERENCED_TABLE_NAME, REFERENCED_COLUMN_NAME
                     FROM information_schema.KEY_COLUMN_USAGE
                     WHERE TABLE_SCHEMA = DATABASE()
                     AND TABLE_NAME = %s
                     AND REFERENCED_TABLE_NAME IS NOT NULL",
                    $fullTableName
                ),
                ARRAY_A
            );
            $result['foreign_keys'] = $foreignKeys;

            // Compare with the expected foreign keys
            if (!empty($expectedStructure['foreign_keys'])) {
                $result['expected_foreign_keys'] = $expectedStructure['foreign_keys'];
                $actualFKs = [];
                foreach ($foreignKeys as $fk) {
                    $actualFKs[] = [
                        'column' => strtolower($fk['COLUMN_NAME']),
                        'referenced_table' => str_replace($wpdb->prefix, '', $fk['REFERENCED_TABLE_NAME']),
                        'referenced_column' => strtolower($fk['REFERENCED_COLUMN_NAME']),
                    ];
                }

                foreach ($expectedStructure['foreign_keys'] as $expectedFK) {
                    $fkFound = false;
                    foreach ($actualFKs as $actualFK) {
                        if (
                            $actualFK['column'] === strtolower($expectedFK['column']) &&
                            $actualFK['referenced_table'] === $expectedFK['referenced_table'] &&
                            $actualFK['referenced_column'] === strtolower($expectedFK['referenced_column'])
                        ) {
                            $fkFound = true;
                            break;
                        }
                    }

                    if (!$fkFound) {
                        $result['missing_foreign_keys'][] = $expectedFK;
                        $result['issues'][] = [
                            'type' => 'error',
                            'table' => $tableName,
                            'issue' => 'missing_foreign_key',
                            'message' => sprintf(
                                __('Missing foreign key in %s: %s -> %s.%s', 'mailerpress'),
                                $tableName,
                                $expectedFK['column'],
                                $expectedFK['referenced_table'],
                                $expectedFK['referenced_column']
                            ),
                            'foreign_key' => $expectedFK,
                        ];
                    }
                }
            }
        } catch (\Throwable $e) {
            // Ignore the foreign key errors (may not be supported)
        }

        return $result;
    }

    /**
     * Get the expected structure of a table from the migrations
     * Aggregate all modifications from all migrations
     */
    protected function getExpectedTableStructure(string $tableName): array
    {
        global $wpdb;
        $fullTableName = Tables::get($tableName);

        $structure = [
            'columns' => [],
            'indexes' => [],
            'foreign_keys' => [],
        ];

        $files = glob($this->migrationPath . '/*.php');
        if ($files === false) {
            return $structure;
        }

        sort($files);

        // Use keys to avoid duplicates
        $columnsMap = [];
        $indexesMap = [];
        $foreignKeysMap = [];

        foreach ($files as $file) {
            if (!file_exists($file)) {
                continue;
            }

            try {
                $schema = new SchemaBuilder();
                $migration = require $file;

                if (!is_callable($migration)) {
                    continue;
                }

                $migration($schema);

                // Use reflection to access the operations
                $reflection = new \ReflectionClass($schema);
                $operationsProperty = $reflection->getProperty('operations');
                $operationsProperty->setAccessible(true);
                $operations = $operationsProperty->getValue($schema);

                foreach ($operations as $op) {
                    $manager = $op['manager'];
                    if ($manager->getTableName() !== $fullTableName) {
                        continue;
                    }

                    // Extract the columns
                    $managerReflection = new \ReflectionClass($manager);

                    // Columns from columnBuilders
                    $columnBuildersProperty = $managerReflection->getProperty('columnBuilders');
                    $columnBuildersProperty->setAccessible(true);
                    $columnBuilders = $columnBuildersProperty->getValue($manager);

                    foreach ($columnBuilders as $builder) {
                        $columnName = $builder->getName();
                        $columnsMap[$columnName] = true;
                    }

                    // Columns from columns (directly added via addColumn)
                    $columnsProperty = $managerReflection->getProperty('columns');
                    $columnsProperty->setAccessible(true);
                    $columns = $columnsProperty->getValue($manager);

                    foreach (array_keys($columns) as $columnName) {
                        $columnsMap[$columnName] = true;
                    }

                    // Handle the columns to drop (do not include them in the expected structure)
                    $columnsToDropProperty = $managerReflection->getProperty('columnsToDrop');
                    $columnsToDropProperty->setAccessible(true);
                    $columnsToDrop = $columnsToDropProperty->getValue($manager);

                    foreach ($columnsToDrop as $columnToDrop) {
                        unset($columnsMap[$columnToDrop]);
                    }

                    // Extract the indexes
                    $indexesProperty = $managerReflection->getProperty('indexes');
                    $indexesProperty->setAccessible(true);
                    $indexes = $indexesProperty->getValue($manager);

                    foreach ($indexes as $indexClause) {
                        // Parse the index to extract the columns
                        // Possible format: 
                        // - "INDEX (`col1`, `col2`)"
                        // - "UNIQUE (`col1`, `col2`)"  
                        // - "CONSTRAINT `name_unique` UNIQUE (`col1`, `col2`)"

                        $indexType = 'INDEX';
                        if (stripos($indexClause, 'UNIQUE') !== false) {
                            $indexType = 'UNIQUE';
                        }

                        // Extract only the columns between parentheses after UNIQUE or INDEX
                        // Pattern: search UNIQUE or INDEX followed by (possibly a constraint name) then parentheses with the columns
                        if (preg_match('/\((?:`[^`]+`(?:,\s*`[^`]+`)*)\)/', $indexClause, $parenMatch)) {
                            // Extract the columns from the parentheses
                            preg_match_all('/`([^`]+)`/', $parenMatch[0], $matches);
                            $cols = $matches[1] ?? [];
                        } else {
                            // Fallback: extract all columns between backticks
                            preg_match_all('/`([^`]+)`/', $indexClause, $matches);
                            $cols = $matches[1] ?? [];

                            // If it's a UNIQUE index with CONSTRAINT, the first element is the constraint name
                            // We remove it if the name contains "_unique" or if it's the only different element
                            if ($indexType === 'UNIQUE' && stripos($indexClause, 'CONSTRAINT') !== false && count($cols) > 1) {
                                // The first element is probably the constraint name, we remove it
                                $firstCol = $cols[0];
                                if (stripos($firstCol, '_unique') !== false || stripos($firstCol, 'unique') !== false) {
                                    array_shift($cols);
                                }
                            }
                        }

                        // Deduplicate and clean the columns
                        $cols = array_unique(array_map('trim', $cols));
                        $cols = array_values(array_filter($cols, function ($col) {
                            // Ignore names that resemble constraint names
                            return stripos($col, '_unique') === false &&
                                stripos($col, 'unique') === false &&
                                $col !== 'PRIMARY';
                        }));

                        if (!empty($cols)) {
                            // Create a unique key for the index (sort the columns to avoid duplicate order)
                            $colsSorted = $cols;
                            sort($colsSorted);
                            $indexKey = $indexType . '_' . implode('_', $colsSorted);

                            // Do not add if an identical index already exists (same columns, same type)
                            if (!isset($indexesMap[$indexKey])) {
                                $indexesMap[$indexKey] = [
                                    'type' => $indexType,
                                    'columns' => $cols, // Keep the original order
                                ];
                            } else {
                                // If an identical index already exists, keep the one with the original order the most logical
                                DatabaseRepairLogger::info(__("Duplicate index detected (ignored)", 'mailerpress'), [
                                    'table' => $tableName,
                                    'index_key' => $indexKey,
                                    'columns' => $cols,
                                    'existing' => $indexesMap[$indexKey],
                                ]);
                            }
                        }
                    }

                    // Handle the indexes to drop
                    $indexesToDropProperty = $managerReflection->getProperty('indexesToDrop');
                    $indexesToDropProperty->setAccessible(true);
                    $indexesToDrop = $indexesToDropProperty->getValue($manager);

                    foreach ($indexesToDrop as $indexToDrop) {
                        // Remove all indexes that contain this column
                        foreach (array_keys($indexesMap) as $key) {
                            if (stripos($key, $indexToDrop) !== false) {
                                unset($indexesMap[$key]);
                            }
                        }
                    }

                    // Extract the foreign keys
                    $foreignKeysProperty = $managerReflection->getProperty('foreignKeys');
                    $foreignKeysProperty->setAccessible(true);
                    $foreignKeys = $foreignKeysProperty->getValue($manager);

                    foreach ($foreignKeys as $column => $fk) {
                        $fkKey = $column . '_' . $fk['foreignTable'] . '_' . $fk['foreignColumn'];
                        $foreignKeysMap[$fkKey] = [
                            'column' => $column,
                            'referenced_table' => $fk['foreignTable'],
                            'referenced_column' => $fk['foreignColumn'],
                        ];
                    }

                    // Handle the foreign keys to drop
                    $foreignKeysToDropProperty = $managerReflection->getProperty('foreignKeysToDrop');
                    $foreignKeysToDropProperty->setAccessible(true);
                    $foreignKeysToDrop = $foreignKeysToDropProperty->getValue($manager);

                    foreach ($foreignKeysToDrop as $fkToDrop) {
                        // Remove all FKs that use this column
                        foreach (array_keys($foreignKeysMap) as $key) {
                            if (strpos($key, $fkToDrop . '_') === 0) {
                                unset($foreignKeysMap[$key]);
                            }
                        }
                    }
                }
            } catch (\Throwable $e) {
                // Ignore the errors during the analysis of migrations
                continue;
            }
        }

        // Convert the maps to arrays
        $structure['columns'] = array_keys($columnsMap);
        $structure['indexes'] = array_values($indexesMap);
        $structure['foreign_keys'] = array_values($foreignKeysMap);

        DatabaseRepairLogger::info(__("Expected final structure for {$tableName}", 'mailerpress'), [
            'columns_count' => count($structure['columns']),
            'indexes_count' => count($structure['indexes']),
            'foreign_keys_count' => count($structure['foreign_keys']),
            'indexes' => $structure['indexes'],
        ]);

        return $structure;
    }

    /**
     * Get the list of all expected tables from the migrations
     */
    protected function getExpectedTables(): array
    {
        // Use the Tables constants to get all expected tables
        $reflection = new \ReflectionClass(Tables::class);
        $constants = $reflection->getConstants();

        // Tables obsolete to exclude from the diagnosis
        $obsoleteTables = [
            'mailerpress_provider_accounts',
            'mailerpress_provider_contacts',
            'mailerpress_provider_lists',
        ];

        $tables = [];
        foreach ($constants as $name => $value) {
            // Filter only the table constants (which start with MAILERPRESS_)
            if (strpos($name, 'MAILERPRESS_') === 0 || strpos($name, 'CONTACT_') === 0) {
                // Exclude the obsolete tables
                if (!in_array($value, $obsoleteTables, true)) {
                    $tables[] = $value;
                }
            }
        }

        // Add the migrations table
        $tables[] = 'mailerpress_migrations';

        return array_unique($tables);
    }

    /**
     * Check the status of the migrations
     */
    protected function checkMigrationStatus(): array
    {
        global $wpdb;

        $trackerTable = $this->tracker->getTableName();
        $trackerExists = $wpdb->get_var(
            $wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->esc_like($trackerTable))
        ) === $trackerTable;

        if (!$trackerExists) {
            return [
                'tracker_exists' => false,
                'total_migrations' => 0,
                'completed' => 0,
                'failed' => 0,
                'pending' => 0,
                'running' => 0,
            ];
        }

        $total = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$trackerTable}");
        $completed = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$trackerTable} WHERE status = 'completed'");
        $failed = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$trackerTable} WHERE status = 'failed'");
        $pending = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$trackerTable} WHERE status = 'pending'");
        $running = (int) $wpdb->get_var("SELECT COUNT(*) FROM {$trackerTable} WHERE status = 'running'");

        return [
            'tracker_exists' => true,
            'total_migrations' => $total,
            'completed' => $completed,
            'failed' => $failed,
            'pending' => $pending,
            'running' => $running,
        ];
    }

    /**
     * Repair the database by executing the missing migrations
     */
    public function repair(): array
    {
        DatabaseRepairLogger::init();
        DatabaseRepairLogger::info(__('Starting database repair', 'mailerpress'));

        $results = [
            'success' => false,
            'message' => '',
            'actions_taken' => [],
            'errors' => [],
            'warnings' => [],
            'fixed_issues' => [],
        ];

        try {
            // 1. Release the migrations lock if necessary
            // Create the manager in force mode to force execution
            $manager = new Manager($this->migrationPath, [], true); // true = force mode
            $status = $manager->getStatus();

            DatabaseRepairLogger::info(__('Initial migration status', 'mailerpress'), ['status' => $status]);

            if ($status['is_locked']) {
                $manager->forceReleaseLock();
                $results['actions_taken'][] = __('Migration lock released', 'mailerpress');
                DatabaseRepairLogger::info(__('Migration lock released', 'mailerpress'));
            }

            // 2. Reset the failed migrations
            if ($status['failed_count'] > 0) {
                $resetCount = $manager->resetFailed();
                $results['actions_taken'][] = sprintf(
                    __('%d migration(s) failed reset(s)', 'mailerpress'),
                    $resetCount
                );
                DatabaseRepairLogger::info("{$resetCount} migration(s) failed reset(s)");
            }

            // 3. Reset the running migrations (in case of crash)
            if ($status['running_count'] > 0) {
                $runningCount = $this->tracker->resetAllRunningMigrations();
                $results['actions_taken'][] = sprintf(
                    __('%d migration(s) in progress reset(s)', 'mailerpress'),
                    $runningCount
                );
                DatabaseRepairLogger::info("{$runningCount} migration(s) in progress reset(s)");
            }

            // 4. Execute the migrations in force mode
            DatabaseRepairLogger::info(__('Execution of migrations in force mode', 'mailerpress'));
            $manager->runForce();
            $results['actions_taken'][] = __('Migrations executed in force mode', 'mailerpress');
            DatabaseRepairLogger::info(__('Migrations executed in force mode', 'mailerpress'));

            // 5. Repair the structural issues detected (missing indexes, columns, foreign keys)
            DatabaseRepairLogger::info(__('Check and repair the structural issues', 'mailerpress'));

            // Get the current diagnostic to see the problems
            $currentDiagnostic = $this->diagnose();
            DatabaseRepairLogger::info(__('Diagnostic before repair', 'mailerpress'), [
                'total_issues' => $currentDiagnostic['summary']['total_issues'],
                'missing_tables' => $currentDiagnostic['summary']['missing_tables'],
                'issues_details' => array_map(function ($issue) {
                    return [
                        'type' => $issue['type'],
                        'table' => $issue['table'],
                        'issue' => $issue['issue'],
                    ];
                }, array_slice($currentDiagnostic['issues'], 0, 10)), // First 10 problems
            ]);

            // Use the diagnostic data to repair
            $structuralRepairs = $this->repairStructuralIssues($currentDiagnostic);
            DatabaseRepairLogger::info(__('Results of structural repair', 'mailerpress'), [
                'fixed_count' => count($structuralRepairs['fixed']),
                'warnings_count' => count($structuralRepairs['warnings']),
            ]);

            $results['fixed_issues'] = $structuralRepairs['fixed'];
            $results['warnings'] = array_merge($results['warnings'], $structuralRepairs['warnings']);

            if (!empty($structuralRepairs['fixed'])) {
                DatabaseRepairLogger::info(__('Structural issues repaired', 'mailerpress'), [
                    'count' => count($structuralRepairs['fixed']),
                    'details' => $structuralRepairs['fixed']
                ]);
            } else {
                DatabaseRepairLogger::info(__('No structural issues to repair', 'mailerpress'), [
                    'total_issues_detected' => $currentDiagnostic['summary']['total_issues'],
                ]);
            }

            $results['success'] = true;
            $results['message'] = __('Database successfully repaired', 'mailerpress');
            DatabaseRepairLogger::info(__('Database repair completed successfully', 'mailerpress'));
        } catch (\Throwable $e) {
            $errorMessage = $e->getMessage();
            $errorTrace = $e->getTraceAsString();

            DatabaseRepairLogger::exception($e, [
                'repair_step' => 'general',
            ]);

            $results['errors'][] = $errorMessage;
            $results['errors'][] = __('Trace:', 'mailerpress') . "\n" . $errorTrace;

            $results['message'] = sprintf(
                __('Error during database repair: %s', 'mailerpress'),
                $errorMessage
            );
        }

        return $results;
    }

    /**
     * Repair the structural issues detected (missing columns, indexes, foreign keys)
     */
    protected function repairStructuralIssues(?array $diagnostic = null): array
    {
        global $wpdb;

        $fixed = [];
        $warnings = [];

        DatabaseRepairLogger::info(__('Starting structural issues repair', 'mailerpress'));

        // Get the current diagnostic if not provided
        if ($diagnostic === null) {
            $diagnostic = $this->diagnose();
        }

        DatabaseRepairLogger::info(__('Diagnostic obtained', 'mailerpress'), [
            'tables_count' => count($diagnostic['tables']),
            'issues_count' => count($diagnostic['issues']),
            'issues_by_type' => [
                'missing_column' => count(array_filter($diagnostic['issues'], fn($i) => $i['issue'] === 'missing_column')),
                'missing_index' => count(array_filter($diagnostic['issues'], fn($i) => $i['issue'] === 'missing_index')),
                'missing_foreign_key' => count(array_filter($diagnostic['issues'], fn($i) => $i['issue'] === 'missing_foreign_key')),
            ],
        ]);

        // Group the problems by table to facilitate the treatment
        $issuesByTable = [];
        foreach ($diagnostic['issues'] as $issue) {
            if (!isset($issuesByTable[$issue['table']])) {
                $issuesByTable[$issue['table']] = [];
            }
            $issuesByTable[$issue['table']][] = $issue;
        }

        DatabaseRepairLogger::info(__('Problems grouped by table', 'mailerpress'), [
            'tables_with_issues' => array_keys($issuesByTable),
            'issues_per_table' => array_map('count', $issuesByTable),
        ]);

        // Treat the problems from the issues directly
        foreach ($issuesByTable as $tableName => $tableIssues) {
            $tableInfo = $diagnostic['tables'][$tableName] ?? null;
            if (!$tableInfo || !$tableInfo['exists']) {
                DatabaseRepairLogger::info(__("Table {$tableName} does not exist, will be created by migrations", 'mailerpress'));
                continue;
            }

            $fullTableName = Tables::get($tableName);
            DatabaseRepairLogger::info(__("Processing issues for {$tableName}", 'mailerpress'), [
                'issues_count' => count($tableIssues),
            ]);

            foreach ($tableIssues as $issue) {
                try {
                    if ($issue['issue'] === 'missing_index' && isset($issue['index'])) {
                        // Repair a missing index
                        $index = $issue['index'];
                        $columns = is_array($index['columns']) ? $index['columns'] : [$index['columns']];
                        $columnsStr = implode('`, `', $columns);

                        $indexType = $index['type'] === 'UNIQUE' ? 'UNIQUE INDEX' : 'INDEX';
                        $indexName = 'idx_' . $tableName . '_' . implode('_', $columns);

                        // Check if all columns exist
                        $currentCols = $wpdb->get_col("SHOW COLUMNS FROM {$fullTableName}", 0);
                        $missingCols = [];
                        foreach ($columns as $col) {
                            if (!in_array($col, $currentCols, true)) {
                                $missingCols[] = $col;
                            }
                        }

                        if (!empty($missingCols)) {
                            $missingColsStr = implode(', ', $missingCols);
                            DatabaseRepairLogger::warning(__("Unable to create index {$indexName}: missing columns", 'mailerpress'), [
                                'table' => $tableName,
                                'index' => $indexName,
                                'expected_columns' => $columns,
                                'missing_columns' => $missingCols,
                                'existing_columns' => $currentCols,
                            ]);
                            $warnings[] = sprintf(
                                __('Missing index: %s (missing columns: %s)', 'mailerpress'),
                                $indexName,
                                $missingColsStr
                            );
                            continue;
                        }

                        // Check if the index already exists (with a potentially different name)
                        $existingIndexes = $wpdb->get_results("SHOW INDEX FROM {$fullTableName}", ARRAY_A);
                        $indexAlreadyExists = false;
                        $normalizedExpectedCols = array_map('strtolower', $columns);

                        foreach ($existingIndexes as $existingIndex) {
                            if ($existingIndex['Key_name'] === 'PRIMARY') {
                                continue;
                            }

                            // Check if an index with the same columns already exists
                            $existingCols = [];
                            foreach ($existingIndexes as $idx) {
                                if ($idx['Key_name'] === $existingIndex['Key_name']) {
                                    $existingCols[] = strtolower($idx['Column_name']);
                                }
                            }

                            if (
                                count($existingCols) === count($normalizedExpectedCols) &&
                                empty(array_diff($normalizedExpectedCols, $existingCols))
                            ) {
                                $indexAlreadyExists = true;
                                DatabaseRepairLogger::info(__("Index already exists with a different name: {$existingIndex['Key_name']}", 'mailerpress'), [
                                    'table' => $tableName,
                                    'existing_name' => $existingIndex['Key_name'],
                                    'expected_name' => $indexName,
                                    'columns' => $columns,
                                ]);
                                break;
                            }
                        }

                        if ($indexAlreadyExists) {
                            DatabaseRepairLogger::info(__("Index already present (different name), ignored: {$indexName}", 'mailerpress'));
                            continue;
                        }

                        // Create the index
                        $sql = "ALTER TABLE {$fullTableName} ADD {$indexType} `{$indexName}` (`{$columnsStr}`)";
                        DatabaseRepairLogger::info(__("Attempt to create index: {$indexName}", 'mailerpress'), [
                            'table' => $tableName,
                            'sql' => $sql,
                            'columns' => $columns,
                        ]);

                        // Reset the previous errors
                        $wpdb->last_error = '';
                        $result = $wpdb->query($sql);
                        $lastError = $wpdb->last_error;
                        $lastQuery = $wpdb->last_query;

                        if ($result !== false && empty($lastError)) {
                            // Check if the index has been created
                            $verifyIndexes = $wpdb->get_results("SHOW INDEX FROM {$fullTableName} WHERE Key_name = '{$indexName}'", ARRAY_A);
                            if (!empty($verifyIndexes)) {
                                $fixed[] = [
                                    'type' => 'index',
                                    'table' => $tableName,
                                    'name' => $indexName,
                                    'columns' => $columns,
                                ];
                                DatabaseRepairLogger::info(__("Index created successfully: {$indexName}", 'mailerpress'), [
                                    'table' => $tableName,
                                    'index' => $indexName,
                                ]);
                            } else {
                                DatabaseRepairLogger::warning(__("Index created but not found during verification: {$indexName}", 'mailerpress'), [
                                    'table' => $tableName,
                                    'sql' => $sql,
                                ]);
                                $warnings[] = sprintf(
                                    __('Index created but not verified: %s', 'mailerpress'),
                                    $indexName
                                );
                            }
                        } else {
                            // Check if the error is due to an index already existing
                            $errorLower = strtolower($lastError);
                            if (
                                strpos($errorLower, 'duplicate key name') !== false ||
                                strpos($errorLower, 'already exists') !== false ||
                                strpos($errorLower, 'duplicate') !== false
                            ) {
                                DatabaseRepairLogger::info(__("Index already exists (detected by SQL error): {$indexName}", 'mailerpress'), [
                                    'table' => $tableName,
                                    'error' => $lastError,
                                ]);
                                // Do not add a warning, the index already exists
                            } else {
                                DatabaseRepairLogger::error(__("Error creating index {$indexName}", 'mailerpress'), [
                                    'table' => $tableName,
                                    'error' => $lastError,
                                    'sql' => $sql,
                                    'result' => $result,
                                    'last_query' => $lastQuery,
                                ]);
                                $warnings[] = sprintf(
                                    __('Error creating index %s: %s', 'mailerpress'),
                                    $indexName,
                                    $lastError ?: __('Unknown error', 'mailerpress')
                                );
                            }
                        }
                    } elseif ($issue['issue'] === 'missing_foreign_key' && isset($issue['foreign_key'])) {
                        // Repair a missing foreign key
                        $fk = $issue['foreign_key'];
                        $fkName = "fk_{$tableName}_{$fk['column']}";
                        $referencedTable = Tables::get($fk['referenced_table']);

                        // Check if the referenced table exists
                        $refTableExists = $wpdb->get_var(
                            $wpdb->prepare("SHOW TABLES LIKE %s", $wpdb->esc_like($referencedTable))
                        ) === $referencedTable;

                        if (!$refTableExists) {
                            DatabaseRepairLogger::warning(__("Unable to create foreign key {$fkName}: referenced table does not exist", 'mailerpress'), [
                                'table' => $tableName,
                                'fk' => $fkName,
                                'referenced_table' => $fk['referenced_table'],
                            ]);
                            $warnings[] = sprintf(
                                __('Missing foreign key: %s (referenced table does not exist)', 'mailerpress'),
                                $fkName
                            );
                            continue;
                        }

                        // Create the foreign key
                        $sql = "ALTER TABLE {$fullTableName} 
                                ADD CONSTRAINT `{$fkName}` 
                                FOREIGN KEY (`{$fk['column']}`) 
                                REFERENCES `{$referencedTable}` (`{$fk['referenced_column']}`) 
                                ON DELETE RESTRICT ON UPDATE RESTRICT";

                        DatabaseRepairLogger::info(__("Attempt to create foreign key: {$fkName}", 'mailerpress'), [
                            'table' => $tableName,
                            'sql' => $sql,
                        ]);

                        $result = $wpdb->query($sql);
                        $lastError = $wpdb->last_error;

                        if ($result !== false && empty($lastError)) {
                            $fixed[] = [
                                'type' => 'foreign_key',
                                'table' => $tableName,
                                'name' => $fkName,
                                'column' => $fk['column'],
                                'referenced_table' => $fk['referenced_table'],
                                'referenced_column' => $fk['referenced_column'],
                            ];
                            DatabaseRepairLogger::info(__("Foreign key created successfully: {$fkName}", 'mailerpress'));
                        } else {
                            DatabaseRepairLogger::error(__("Error creating foreign key {$fkName}", 'mailerpress'), [
                                'table' => $tableName,
                                'error' => $lastError,
                                'sql' => $sql,
                                'result' => $result,
                            ]);
                            $warnings[] = sprintf(
                                __('Error creating foreign key %s: %s', 'mailerpress'),
                                $fkName,
                                $lastError ?: __('Unknown error', 'mailerpress')
                            );
                        }
                    } elseif ($issue['issue'] === 'missing_column' && isset($issue['column'])) {
                        // Missing columns require a migration, we cannot create them automatically
                        DatabaseRepairLogger::warning(__("Missing column detected: {$tableName}.{$issue['column']}", 'mailerpress'), [
                            'table' => $tableName,
                            'column' => $issue['column'],
                        ]);
                        $warnings[] = sprintf(
                            __('Missing column: %s.%s (requires a migration)', 'mailerpress'),
                            $tableName,
                            $issue['column']
                        );
                    }
                } catch (\Throwable $e) {
                    DatabaseRepairLogger::exception($e, [
                        'table' => $tableName,
                        'issue' => $issue,
                    ]);
                    $warnings[] = sprintf(
                        __('Error during repair: %s', 'mailerpress'),
                        $e->getMessage()
                    );
                }
            }
        }

        DatabaseRepairLogger::info(__('Summary of structural repair', 'mailerpress'), [
            'total_fixed' => count($fixed),
            'total_warnings' => count($warnings),
            'fixed_details' => $fixed,
        ]);

        return [
            'fixed' => $fixed,
            'warnings' => $warnings,
        ];
    }
}
