<?php

declare(strict_types=1);

namespace MailerPress\Jobs;

\defined('ABSPATH') || exit;

use MailerPress\Core\Abstract\BaseJob;
use MailerPress\Core\EmailManager\EmailServiceManager;
use MailerPress\Core\HtmlParser;
use MailerPress\Core\Kernel;
use Throwable;
use WP_Error;

class SendEmailJob extends BaseJob
{
    /**
     * @param array{
     *   to: array<int, array{email:string,variables:array<string,mixed>}>,
     *   subject: string,
     *   sender_to: string,
     *   sender_name: string,
     *   body: string,
     *   webhook_url?: string,
     *   batch_id?: int|string,
     *   transient_key?: string
     * } $data
     */
    public function handle(array $data): void
    {
        $countSuccess = 0;
        $countError = 0;
        $errors = []; // collect per-recipient error detail (optional)

        // We’ll want to know if the whole job failed catastrophically
        $jobException = null;

        try {
            $recipientBatches = $data['to']; // you said already chunked to <= 50 items

            // Acquire shared services once outside loop.
            $container = Kernel::getContainer();
            /** @var HtmlParser $parser */
            $parser = $container->get(HtmlParser::class);
            /** @var EmailServiceManager $manager */
            $manager = $container->get(EmailServiceManager::class);
            $mailer = $manager->getActiveService();

            $apiKey = '';

            $servicesData = get_option('mailerpress_email_services', []);

            if (!empty($servicesData['default_service'])
                && !empty($servicesData['services'][$servicesData['default_service']]['conf']['api_key'])
            ) {
                $defaultService = $servicesData['default_service'];
                $apiKey = $servicesData['services'][$defaultService]['conf']['api_key'];
            }

            // Récupérer les paramètres Reply to depuis les paramètres par défaut
            $defaultSettings = get_option('mailerpress_default_settings', []);
            if (is_string($defaultSettings)) {
                $defaultSettings = json_decode($defaultSettings, true) ?: [];
            }
            
            // Déterminer les valeurs Reply to (utiliser From si Reply to est vide)
            $replyToName = !empty($defaultSettings['replyToName']) 
                ? $defaultSettings['replyToName'] 
                : ($data['sender_name'] ?? '');
            $replyToAddress = !empty($defaultSettings['replyToAddress']) 
                ? $defaultSettings['replyToAddress'] 
                : ($data['sender_to'] ?? '');

            foreach ($recipientBatches as $recipient) {
                try {
                    // Render per‑recipient body
                    $body = $parser->init(
                        $data['body'],
                        $recipient['variables'] ?? []
                    )->replaceVariables();

                    $result = $mailer->sendEmail([
                        'to' => $recipient['email'],
                        'html' => true,
                        'body' => $body,
                        'subject' => $data['subject'],
                        'sender_name' => $data['sender_name'] ?? '',
                        'sender_to' => $data['sender_to'] ?? '',
                        'reply_to_name' => $replyToName,
                        'reply_to_address' => $replyToAddress,
                        'apiKey' => $apiKey,
                    ]);

                    if ($result === true) {
                        ++$countSuccess;
                    } else {
                        ++$countError;

                        $errorMessage = 'Mailer returned false.';
                        if ($result instanceof \WP_Error) {
                            $errorMessage = $result->get_error_message();
                        }

                        $errors[] = [
                            'email' => $recipient['email'],
                            'message' => $errorMessage,
                        ];

                        // --- Log to file per batch ---
                        $batchId = $data['batch_id'] ?? 'unknown';
                        $logDir = WP_CONTENT_DIR . '/mailerpress-logs';
                        $logFile = $logDir . "/batch-{$batchId}.log";

                        if (!is_dir($logDir)) {
                            wp_mkdir_p($logDir);
                        }

                        $logEntry = sprintf(
                            "[%s] Email: %s | Error: %s\n",
                            current_time('mysql'),
                            $recipient['email'],
                            $errorMessage
                        );

                        file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
                    }
                } catch (Throwable $e) {
                    ++$countError;
                    $errors[] = [
                        'email' => $recipient['email'] ?? '',
                        'message' => $e->getMessage(),
                    ];

                    // --- Log to file per batch ---
                    $batchId = $data['batch_id'] ?? 'unknown';

                    $logDir = WP_CONTENT_DIR . '/mailerpress-logs';
                    $logFile = $logDir . "/batch-{$batchId}.log";

                    if (!is_dir($logDir)) {
                        wp_mkdir_p($logDir); // creates dir safely with proper perms
                    }

                    $logEntry = sprintf(
                        "[%s] Email: %s | Error: %s\n",
                        current_time('mysql'),
                        $recipient['email'] ?? 'N/A',
                        $e->getMessage()
                    );

                    file_put_contents($logFile, $logEntry, FILE_APPEND | LOCK_EX);
                }
            }
        } catch (Throwable $e) {
            // This means something big failed (config, container, parsing, etc.)
            $countSuccess = 0;
            $countError = count($data['to']);
        } finally {
            if (
                \array_key_exists('webhook_url', $data)
                && \array_key_exists('batch_id', $data)
            ) {
                $this->sendNotification(
                    $data['webhook_url'],
                    [
                        'action' => 'batch_update',
                        'batch_id' => $data['batch_id'],
                        'countSent' => [
                            'success' => $countSuccess,
                            'error' => $countError,
                        ],
                        // optional: pass transient key for cleanup
                        'transient_key' => $data['transient_key'] ?? null,
                    ]
                );
            }
        }

        // If the job blew up, rethrow so the queue system can mark it failed / retry.
        if ($jobException) {
            throw $jobException; // Action Scheduler will log failure & retry per its rules
        }
    }

    /**
     * Post a JSON webhook and optionally clear transient.
     */
    private function sendNotification(string $url, array $payload): void
    {
        if (!empty($payload['transient_key'])) {
            delete_transient($payload['transient_key']);
        }

        $args = [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($payload),
            'timeout' => 15, // be explicit; default 5s may be too low at scale
        ];

        wp_remote_post($url, $args);
    }
}
