<?php

namespace MailerPress\Core\Migrations;

\defined('ABSPATH') || exit;

class CustomTableManager
{
    protected string $tableName;
    protected string $version = '1.0.1';
    protected string $versionOptionName;
    protected array $columns = [];
    protected array|string|null $primaryKey = null;
    protected array $indexes = [];
    protected array $foreignKeys = [];
    protected array $columnBuilders = [];
    protected array $columnsToDrop = [];


    public function __construct(string $tableName)
    {
        global $wpdb;
        $this->tableName = $wpdb->prefix . $tableName;
        $this->versionOptionName = 'custom_table_' . sanitize_key($tableName) . '_version';
    }

    public function dropColumn(string $name): static
    {
        $this->columnsToDrop[] = $name;
        return $this;
    }

    public function setVersion(string $version): static
    {
        $this->version = $version;
        return $this;
    }

    public function getTableName(): string
    {
        return $this->tableName;
    }

    public function addColumn(string $name, string $definition): static
    {
        $this->columns[$name] = $definition;
        return $this;
    }

    public function setPrimaryKey(string|array $columns): static
    {
        if (is_string($columns)) {
            // Automatically detect comma-separated list and split
            $columns = array_map('trim', explode(',', $columns));
        }

        $this->primaryKey = $columns;
        return $this;
    }

    public function addIndex(string|array $columns, string $type = 'INDEX'): static
    {
        $type = strtoupper($type);

        // Normalize to array
        if (is_string($columns)) {
            $columns = array_map('trim', explode(',', $columns));
        }

        $indexName = implode('_', $columns);
        $key = "{$type}_{$indexName}";

        // Quote columns
        $quotedColumns = implode('`, `', $columns);

        if (!isset($this->indexes[$key])) {
            if ($type === 'UNIQUE') {
                $this->indexes[$key] = "CONSTRAINT `{$indexName}_unique` UNIQUE (`{$quotedColumns}`)";
            } else {
                $this->indexes[$key] = "$type (`{$quotedColumns}`)";
            }
        }

        return $this;
    }

    public function addForeignKey(
        string $column,
        string $foreignTable,
        string $foreignColumn,
        string $onDelete = 'RESTRICT',
        string $onUpdate = 'RESTRICT'
    ): static {
        $this->foreignKeys[$column] = [
            'foreignTable' => $foreignTable,
            'foreignColumn' => $foreignColumn,
            'onDelete' => strtoupper($onDelete),
            'onUpdate' => strtoupper($onUpdate),
        ];
        return $this;
    }

    public function column(string $name, string $type): ColumnBuilder
    {
        $builder = new ColumnBuilder($name, $type);
        $this->columnBuilders[] = $builder;
        return $builder;
    }

    public function id(string $name = 'id'): static
    {
        $this->addColumn($name, 'BIGINT(20) UNSIGNED NOT NULL AUTO_INCREMENT');
        $this->setPrimaryKey($name);
        return $this;
    }

    public function string(string $name, int $length = 255): ColumnBuilder
    {
        return $this->column($name, "VARCHAR($length)");
    }

    public function text(string $name): ColumnBuilder
    {
        return $this->column($name, 'TEXT');
    }

    public function longText(string $name): ColumnBuilder
    {
        return $this->column($name, 'LONGTEXT');
    }

    public function boolean(string $name): ColumnBuilder
    {
        return $this->column($name, 'TINYINT(1)');
    }

    public function integer(string $name): ColumnBuilder
    {
        return $this->column($name, 'INT');
    }

    public function bigInteger(string $name): ColumnBuilder
    {
        return $this->column($name, 'BIGINT(20)');
    }

    public function unsignedBigInteger(string $name): ColumnBuilder
    {
        return $this->bigInteger($name)->unsigned();
    }

    public function decimal(string $name, int $precision = 10, int $scale = 2): ColumnBuilder
    {
        return $this->column($name, "DECIMAL($precision,$scale)");
    }

    public function float(string $name): ColumnBuilder
    {
        return $this->column($name, 'FLOAT');
    }

    public function json(string $name): ColumnBuilder
    {
        return $this->column($name, 'JSON');
    }

    public function enum(string $name, array $values): ColumnBuilder
    {
        $escaped = array_map(fn($v) => "'$v'", $values);
        return $this->column($name, 'enum(' . implode(',', $escaped) . ')');
    }

    public function datetime(string $name): ColumnBuilder
    {
        return $this->column($name, 'DATETIME');
    }

    public function createdAt(): static
    {
        $this->column('created_at', 'DATETIME')->notNull()->default('CURRENT_TIMESTAMP');
        return $this;
    }

    public function deletedAt(): static
    {
        $this->column('deleted_at', 'DATETIME')->nullable();
        return $this;
    }

    protected function tableExists(): bool
    {
        global $wpdb;
        return $wpdb->get_var("SHOW TABLES LIKE '{$this->tableName}'") === $this->tableName;
    }

    protected function getCurrentColumns(): array
    {
        global $wpdb;
        return $wpdb->get_col("SHOW COLUMNS FROM {$this->tableName}", 0);
    }

    protected function getCurrentIndexes(): array
    {
        global $wpdb;
        $results = $wpdb->get_results("SHOW INDEX FROM {$this->tableName}");
        $existing = [];
        foreach ($results as $index) {
            $key = strtoupper($index->Key_name);
            $col = strtolower($index->Column_name);
            $existing[$key][] = $col;
        }
        return $existing;
    }

    protected function createTable(): void
    {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        foreach ($this->columnBuilders as $builder) {
            $this->addColumn($builder->getName(), $builder->getSQL());
            if ($builder->isUnique()) {
                $this->addIndex($builder->getName(), 'UNIQUE');
            }
        }

        $charsetCollate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE {$this->tableName} (\n";
        foreach ($this->columns as $name => $definition) {
            $sql .= "  `$name` $definition,\n";
        }

        if ($this->primaryKey) {
            $columns = is_array($this->primaryKey) ? $this->primaryKey : [$this->primaryKey];
            $sql .= 'PRIMARY KEY (`' . implode('`,`', $columns) . '`),';
        }

        foreach ($this->indexes as $indexClause) {
            $sql .= "  $indexClause,\n";
        }

        foreach ($this->foreignKeys as $column => $fk) {
            $fkName = "fk_{$this->tableName}_$column";
            $sql .= "  CONSTRAINT `$fkName` FOREIGN KEY (`$column`) REFERENCES `{$wpdb->prefix}{$fk['foreignTable']}` (`{$fk['foreignColumn']}`) ON DELETE {$fk['onDelete']} ON UPDATE {$fk['onUpdate']},\n";
        }

        $sql = rtrim($sql, ",\n") . "\n) $charsetCollate;";
        dbDelta($sql);
    }

    protected function getExistingForeignKeys(): array
    {
        global $wpdb;
        $results = $wpdb->get_results("SELECT CONSTRAINT_NAME FROM information_schema.KEY_COLUMN_USAGE WHERE TABLE_NAME = '{$this->tableName}' AND CONSTRAINT_SCHEMA = DATABASE() AND REFERENCED_TABLE_NAME IS NOT NULL");
        return array_map(fn($row) => $row->CONSTRAINT_NAME, $results);
    }

    public function migrate(): bool
    {
        global $wpdb;
        require_once ABSPATH . 'wp-admin/includes/upgrade.php';

        $installedVersion = get_option($this->versionOptionName);
        $tableExists = $this->tableExists();

        if ($installedVersion === $this->version) {
            return false;
        }

        foreach ($this->columnBuilders as $builder) {
            $this->addColumn($builder->getName(), $builder->getSQL());
            if ($builder->isUnique()) {
                $this->addIndex($builder->getName(), 'UNIQUE');
            }
        }

        $charsetCollate = $wpdb->get_charset_collate();

        if (!$tableExists) {
            $this->createTable();
        } else {
            $currentCols = $this->getCurrentColumns();
            $currentIndexes = $this->getCurrentIndexes();
            $existingConstraints = $this->getExistingForeignKeys();
            $alterParts = [];

            foreach ($this->columnBuilders as $builder) {
                $name = $builder->getName();

                if (!in_array($name, $currentCols, true)) {
                    $sql = "ADD COLUMN `$name` {$builder->getSQL()}";

                    if ($builder->getBefore()) {
                        $sql .= " BEFORE `{$builder->getBefore()}`";
                    } elseif ($builder->getAfter()) {
                        $sql .= " AFTER `{$builder->getAfter()}`";
                    }

                    $alterParts[] = $sql;
                }
            }

            foreach ($this->indexes as $key => $clause) {
                preg_match('/`(.+?)`/', $clause, $matches);
                $col = $matches[1] ?? null;
                $indexKey = strtoupper($col);
                if ($col && !isset($currentIndexes[$indexKey])) {
                    $alterParts[] = "ADD $clause";
                }
            }

            foreach ($this->foreignKeys as $column => $fk) {
                $fkName = "fk_{$this->tableName}_$column";
                if (!in_array($fkName, $existingConstraints, true)) {
                    $alterParts[] = "ADD CONSTRAINT `$fkName` FOREIGN KEY (`$column`) REFERENCES `{$wpdb->prefix}{$fk['foreignTable']}` (`{$fk['foreignColumn']}`) ON DELETE {$fk['onDelete']} ON UPDATE {$fk['onUpdate']}";
                }
            }


            foreach ($this->columnsToDrop as $columnName) {
                if (in_array($columnName, $currentCols, true)) {
                    $alterParts[] = "DROP COLUMN `$columnName`";
                }
            }

            if (!empty($alterParts)) {
                $alterSQL = "ALTER TABLE {$this->tableName} " . implode(", ", $alterParts);
                $wpdb->query($alterSQL);
            }
        }

        update_option($this->versionOptionName, $this->version, false);
        return true;
    }

    public function modifyColumn(string $columnName, string $definition): static
    {
        global $wpdb;

        $wpdb->query("
        ALTER TABLE {$this->tableName}
        MODIFY COLUMN `$columnName` $definition
    ");

        return $this;
    }


    public function drop(): void
    {
        global $wpdb;
        $wpdb->query("DROP TABLE IF EXISTS {$this->tableName}");
        delete_option($this->versionOptionName);
    }

    public function generateSQLPreview(): string
    {
        global $wpdb;
        $sql = "CREATE TABLE {$this->tableName} (\n";

        foreach ($this->columnBuilders as $builder) {
            $this->addColumn($builder->getName(), $builder->getSQL());
            if ($builder->isUnique()) {
                $this->addIndex($builder->getName(), 'UNIQUE');
            }
        }

        foreach ($this->columns as $name => $definition) {
            $sql .= "  `$name` $definition,\n";
        }

        if ($this->primaryKey) {
            $columns = is_array($this->primaryKey) ? $this->primaryKey : [$this->primaryKey];
            $quoted = implode('`, `', $columns);
            $sql .= "  PRIMARY KEY (`$quoted`),\n";
        }

        foreach ($this->indexes as $indexClause) {
            $sql .= "  $indexClause,\n";
        }

        foreach ($this->foreignKeys as $column => $fk) {
            $fkName = "fk_{$this->tableName}_$column";
            $sql .= "  CONSTRAINT `$fkName` FOREIGN KEY (`$column`) REFERENCES `{$wpdb->prefix}{$fk['foreignTable']}` (`{$fk['foreignColumn']}`) ON DELETE {$fk['onDelete']} ON UPDATE {$fk['onUpdate']},\n";
        }

        $sql = rtrim($sql, ",\n") . "\n)";
        return $sql;
    }
}
