<?php

declare(strict_types=1);

namespace MailerPress\Core;

\defined('ABSPATH') || exit;

final class HtmlParser
{
    private $htmlContent;
    private $variables;

    /**
     * Définit les variables dynamiques à remplacer.
     *
     * @param array $variables tableau associatif des variables dynamiques et leurs valeurs
     */
    public function init(string $htmlContent, array $variables): static
    {
        $this->htmlContent = $htmlContent;
        $this->variables = $variables;

        if (!empty($variables['TRACK_OPEN'])) {
            $trackingUrl = $variables['TRACK_OPEN'];

            $trackingTable = sprintf(
                '<table role="presentation" border="0" cellpadding="0" cellspacing="0" width="1" height="1" style="display:none;">' .
                    '<tr><td><img src="%s" alt="" width="1" height="1" style="display:block;"/></td></tr>' .
                    '</table>',
                htmlspecialchars($trackingUrl, ENT_QUOTES)
            );

            // Regex to find the entire .footer-email div block
            $pattern = '#(<div[^>]*class=["\'][^"\']*footer-email[^"\']*["\'][^>]*>.*?</div>)#is';

            $this->htmlContent = preg_replace_callback($pattern, function ($matches) use ($trackingTable) {
                // Append the tracking table right after the footer-email div
                return $matches[1] . $trackingTable;
            }, $this->htmlContent);


            // If footer-email was not found, add tracking pixel before closing </body> tag
            if (strpos($this->htmlContent, $trackingTable) === false) {
                // Try to add before </body>
                if (preg_match('#</body>#i', $this->htmlContent)) {
                    $this->htmlContent = preg_replace('#(</body>)#i', $trackingTable . '$1', $this->htmlContent);
                } else {
                    // If no </body> tag, append at the end
                    $this->htmlContent .= $trackingTable;
                }
            }
        }

        return $this;
    }


    /**
     * Supprime les spans d'emoji et garde uniquement l'emoji.
     *
     * @param string $html Le contenu HTML
     * @return string Le contenu HTML avec les spans d'emoji supprimés
     */
    private function removeEmojiSpans(string $html): string
    {
        // Pattern pour détecter les spans avec data-emoji-id et garder uniquement le contenu
        $pattern = '#<span[^>]*data-emoji-id=["\'][^"\']*["\'][^>]*>(.*?)</span>#i';

        return preg_replace_callback($pattern, function ($matches) {
            // Retourner uniquement le contenu du span (l'emoji)
            return $matches[1] ?? '';
        }, $html);
    }

    /**
     * Remplace les variables dynamiques dans le contenu HTML.
     *
     * @return string le contenu HTML avec les variables remplacées par leurs valeurs
     */
    public function replaceVariables(): string
    {
        $replacedContent = $this->htmlContent;

        // 0️⃣ Supprimer les spans d'emoji et garder uniquement l'emoji
        $replacedContent = $this->removeEmojiSpans($replacedContent);

        // 1️⃣ Remove merge-tag spans but keep placeholder inside
        $replacedContent = preg_replace_callback(
            '#<span[^>]*class=["\']merge-tag-span["\'][^>]*>(.*?)</span>#i',
            fn($matches) => $matches[1] ?? '',
            $replacedContent
        );

        // 2️⃣ Replace {{key default="value"}} or {{key}}
        $pattern = '/{{\s*([a-zA-Z0-9_]+)(?:\s+default=["\']([^"\']*)["\'])?\s*}}/';
        $replacedContent = preg_replace_callback($pattern, function ($matches) {
            $key = $matches[1];
            $defaultValue = $matches[2] ?? '';
            $value = $this->variables[$key] ?? '';

            if ($value === '' && $defaultValue !== '') {
                $value = $defaultValue;
            }

            return $value === '' ? '§EMPTY_VAR§' : $value;
        }, $replacedContent);

        // 3️⃣ Replace %VAR% format
        foreach ($this->variables as $key => $value) {
            $patternPercent = sprintf('/%%%s%%/', preg_quote($key, '/'));
            $replacedContent = preg_replace($patternPercent, $value === '' ? '§EMPTY_VAR§' : $value, $replacedContent);
        }

        // 4️⃣ Remove &nbsp; or spaces before empty vars
        $replacedContent = preg_replace('/(&nbsp;|\s)+§EMPTY_VAR§/', '§EMPTY_VAR§', $replacedContent);

        // 5️⃣ Remove the empty var markers
        $replacedContent = str_replace('§EMPTY_VAR§', '', $replacedContent);

        // 6️⃣ Final tidy-up
        $replacedContent = preg_replace('/\s{2,}/', ' ', $replacedContent);
        $replacedContent = preg_replace('/\s+([,.!?])/', '$1', $replacedContent);
        $replacedContent = preg_replace('/>\s+</', '><', $replacedContent);

        return trim($replacedContent);
    }
}
