<?php

declare(strict_types=1);

namespace MailerPress\Blocks;

\defined('ABSPATH') || exit;

use MailerPress\Core\Enums\Tables;

final class TemplatesCategories
{
    private array $categories = [];

    public function __construct()
    {
        $this->categories = [];
    }

    public function registerCategory(array $category): void
    {
        $this->categories = array_merge($this->categories, $category);
    }

    public function getCategories(): array
    {
        global $wpdb;

        $categoriesTable = Tables::get(Tables::MAILERPRESS_CATEGORIES);

        $categories = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT category_id, name AS label
                 FROM {$categoriesTable}
                 WHERE type = %s
                 ORDER BY name ASC",
                'template'
            ),
            ARRAY_A
        );

        // Convert to associative array: category_id => [label => ...]
        $result = [];
        foreach ($categories as $category) {
            $result[$category['category_id']] = [
                'label' => $category['label'],
            ];
        }

        return $result;
    }

    public function getTemplatesGroupByCategories(): array
    {
        global $wpdb;

        $templatesTable = Tables::get(Tables::MAILERPRESS_TEMPLATES);
        $categoriesTable = Tables::get(Tables::MAILERPRESS_CATEGORIES);

        // Fetch all categories where type = 'template'
        $categories = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT category_id, name AS label 
             FROM {$categoriesTable}
             WHERE type = %s",
                'template'
            ),
            OBJECT_K
        );

        // Fetch counts of templates grouped by cat_id
        $entries = $wpdb->get_results("
        SELECT cat_id, COUNT(*) AS total_entries
        FROM {$templatesTable}
        GROUP BY cat_id
    ", OBJECT_K);

        // Build result array
        $result = [];

        // "All templates" count
        $totalAll = array_reduce($entries, static function ($carry, $item) {
            return $carry + (int) $item->total_entries;
        }, 0);

        $result[''] = [
            'label' => __('All templates', 'mailerpress'),
            'total_entries' => $totalAll,
        ];

        $translations = [
            'ecommerce' => __('Ecommerce', 'mailerpress'),
            'newsletter' => __('Newsletter', 'mailerpress'),
            'thank-you' => __('Thank you', 'mailerpress'),
            'welcome' => __('Welcome', 'mailerpress'),
        ];

        foreach ($categories as $category_id => $category) {
            $count = isset($entries[$category_id]) ? (int) $entries[$category_id]->total_entries : 0;

            if ($count > 0) {
                $label = $translations[$category->label] ?? $category->label;

                $result[$category_id] = [
                    'label' => $label,
                    'total_entries' => $count,
                ];
            }
        }

        return $result;
    }
}
