<?php

declare(strict_types=1);

namespace MailerPress\Api;

\defined('ABSPATH') || exit;

use MailerPress\Core\Attributes\Endpoint;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Kernel;

class WebHooks
{
    #[Endpoint(
        'webhook/notify',
        methods: 'POST',
    )]
    public function handleNotification(\WP_REST_Request $request): void
    {
        $data = $request->get_json_params();

        switch ($data['action']) {
            case 'batch_update':
                $this->update_batch_status_increment(
                    $data['batch_id'],
                    $data['countSent'],
                    $data['transient_key']
                );
                break;
            case 'bounce_detected':
                $this->markeEmailAsBounce(
                    $data['batch_id'],
                    $data['email'],
                    $data['bounce_type']
                );
                break;
            case 'mailerpress_batch_update':
                $this->update_batch(
                    $data['wp_batch_id'],
                    $data['status'],
                    $data['sent'],
                    $data['failed'],
                    $data['total'],
                );
                break;
            default:
                break;
        }
    }

    private function update_batch_status_increment($batch_id, $countSent, $transient_key)
    {
        global $wpdb;

        $table_name = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);

        // Récupérer les valeurs actuelles
        $batch = $wpdb->get_row($wpdb->prepare(
            "SELECT sent_emails, error_emails, campaign_id, total_emails FROM {$table_name} WHERE id = %d",
            intval($batch_id)
        ), ARRAY_A);

        if (!$batch) {
            return false;
        }

        // Incrémenter les valeurs
        $new_sent_emails = (int)$batch['sent_emails'] + (int)$countSent['success'];
        $new_error_emails = (int)$batch['error_emails'] + (int)$countSent['error'];

        // Déterminer le nouveau statut si nécessaire
        $new_status = 'in_progress'; // Exemple : vous pouvez ajuster selon la logique

        // Mettre à jour les valeurs dans la table
        $updated = $wpdb->update(
            $table_name,
            [
                'status' => $new_status,
                'sent_emails' => $new_sent_emails,
                'error_emails' => $new_error_emails,
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $batch_id],
            ['%s', '%d', '%d', '%s'], // Formats des colonnes mises à jour
            ['%d'] // Format de la condition
        );


        if (($new_sent_emails + $new_error_emails) === (int)$batch['total_emails']) {
            $campaign = Kernel::getContainer()->get(\MailerPress\Models\Campaigns::class)->find($batch['campaign_id']);
            
            // Determine final status: if all emails failed, mark as failed
            $final_status = ($new_error_emails === (int)$batch['total_emails']) ? 'failed' : 'sent';
            $campaign_status = ($final_status === 'failed') ? 'error' : ($campaign->campaign_type === 'automated' ? 'active' : 'sent');
            
            do_action(
                'mailerpress_batch_event',
                $campaign_status,
                $batch['campaign_id'],
                $batch_id
            );
            $updated = $wpdb->update(
                $table_name,
                [
                    'status' => $final_status,
                ],
                ['id' => $batch_id],
                ['%s'], // Formats des colonnes mises à jour
                ['%d'] // Format de la condition
            );
        } else {
            do_action('mailerpress_batch_event', 'in_progress', $batch['campaign_id'], $batch_id);
        }

        delete_transient($transient_key);

        return false !== $updated;
    }

    private function markeEmailAsBounce(mixed $batch_id, mixed $email, mixed $bounce_type)
    {
    }

    private function update_batch(mixed $wp_batch_id, mixed $status, mixed $sent, mixed $failed, mixed $total)
    {
    }

}
