<?php

namespace MailerPress\Api;

\defined('ABSPATH') || exit;

use MailerPress\Core\Attributes\Endpoint;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Kernel;
use MailerPress\Models\Batch;

class Dashboard
{
    #[Endpoint(
        'dashboard/campaigns',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView'],
    )]
    public function campaignByIntervalDate(\WP_REST_Request $request)
    {
        global $wpdb;

        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $interval = (int) $request->get_param('interval') ?: 30;

        $query = $wpdb->prepare("
        SELECT *
        FROM $table
        WHERE created_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
          AND (status IS NULL OR status != %s)
        ORDER BY created_at DESC
        LIMIT 5
    ", $interval, 'trash');

        $results = $wpdb->get_results($query);

        foreach ($results as &$result) {
            $result->batch = $result->batch_id
                ? Kernel::getContainer()->get(Batch::class)->getById($result->batch_id, true)
                : null;

            $result->statistics = !empty($result->batch_id)
                ? Kernel::getContainer()->get(Batch::class)->getStatistics($result->batch_id)
                : null;
        }

        return rest_ensure_response($results);
    }


    #[Endpoint(
        'dashboard/email-batches-by-date',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView'],
    )]
    public function emailBatchesByDateRange(\WP_REST_Request $request)
    {
        global $wpdb;

        // Define the table for email batches
        $table = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);
        $campaignTable = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Fetch start_date and end_date params from the request
        $startDateParam = $request->get_param('start_date');
        $endDateParam = $request->get_param('end_date');

        // Ensure the date format is compatible with MySQL
        try {
            // Convert to DateTime objects, handling both timestamp and date formats
            $startDate = is_numeric($startDateParam) ? new \DateTime('@' . (int)$startDateParam) : new \DateTime($startDateParam);
            $endDate = is_numeric($endDateParam) ? new \DateTime('@' . (int)$endDateParam) : new \DateTime($endDateParam);

            // Set the time for start date to 00:00:00 (beginning of the day)
            $startDate->setTime(0, 0, 0);

            // Set the time for end date to 23:59:59 (end of the day)
            $endDate->setTime(23, 59, 59);

            // Format the dates to MySQL-compatible format
            $startDateFormatted = $startDate->format('Y-m-d H:i:s');
            $endDateFormatted = $endDate->format('Y-m-d H:i:s');
        } catch (\Exception $e) {
            return rest_ensure_response([
                'message' => 'Invalid date format provided.',
                'error' => $e->getMessage(),
            ]);
        }

        $query = $wpdb->prepare("
    SELECT 
        b.id,
        b.campaign_id,
        b.created_at,
        b.updated_at,
        b.total_emails,
        b.total_open,
        b.sent_emails,
        b.error_emails,
        b.error_message,
        b.scheduled_at,
        b.sender_name,
        b.sender_to,
        b.subject,
        b.offset,
        c.content_html, 
        c.name, 
        c.subject AS campaign_subject, 
        c.status
    FROM $table b
    LEFT JOIN $campaignTable c ON b.campaign_id = c.campaign_id
    WHERE b.scheduled_at >= %s 
      AND b.scheduled_at <= %s
      AND c.status != 'draft'
    ORDER BY b.scheduled_at DESC
", $startDateFormatted, $endDateFormatted);


        // Execute query and get results
        $results = $wpdb->get_results($query);

        foreach ($results as &$result) {
            $result->batch = $result->id ?
                Kernel::getContainer()->get(Batch::class)->getById($result->id, true) :
                null;
            $result->statistics = !empty($result->id) ? Kernel::getContainer()->get(Batch::class)->getStatistics($result->id) : null;
        }


        return rest_ensure_response($results);
    }


    #[Endpoint(
        'dashboard/contacts',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView'],
    )]
    public function contactsByIntervalDate(\WP_REST_Request $request)
    {
        global $wpdb;

        $table = Tables::get(Tables::MAILERPRESS_CONTACT);
        $interval = (int)$request->get_param('interval') ?: 30;

        // 1. Total contacts (global)
        $total_count = (int)$wpdb->get_var("SELECT COUNT(*) FROM $table");

        // 2. Filtered count by updated_at
        $filtered_count_query = $wpdb->prepare("
        SELECT COUNT(*) 
        FROM $table
        WHERE updated_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
    ", $interval);
        $filtered_count = (int)$wpdb->get_var($filtered_count_query);

        // 3. Subscribed contacts count within interval
        $subscribed_count_query = $wpdb->prepare("
        SELECT COUNT(*) 
        FROM $table
        WHERE subscription_status = 'subscribed'
          AND updated_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
    ", $interval);
        $subscribed_count = (int)$wpdb->get_var($subscribed_count_query);

        // 4. Unsubscribed contacts count within interval
        $unsubscribed_count_query = $wpdb->prepare("
        SELECT COUNT(*) 
        FROM $table
        WHERE subscription_status = 'unsubscribed'
          AND updated_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
    ", $interval);
        $unsubscribed_count = (int)$wpdb->get_var($unsubscribed_count_query);

        // 5. Latest 5 contacts updated in interval
        $contacts_query = $wpdb->prepare("
        SELECT contact_id, email, first_name, last_name, subscription_status, updated_at
        FROM $table
        WHERE updated_at >= DATE_SUB(NOW(), INTERVAL %d DAY)
        ORDER BY updated_at DESC
        LIMIT 5
    ", $interval);
        $contacts = $wpdb->get_results($contacts_query);

        return rest_ensure_response([
            'total_count' => $total_count,
            'filtered_count' => $filtered_count,
            'subscribed_count' => $subscribed_count,
            'unsubscribed_count' => $unsubscribed_count,
            'contacts' => $contacts,
        ]);
    }

}