<?php

declare(strict_types=1);

namespace MailerPress\Api;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\Attributes\Endpoint;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Kernel;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

class Contacts
{

    #[Endpoint(
        'export/(?P<export_id>[a-zA-Z0-9-]+)',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageAudience']

    )]
    public static function handleExportDownload(WP_REST_Request $request)
    {
        $export_id = sanitize_text_field($request->get_param('export_id'));
        $token = sanitize_text_field($request->get_param('token'));

        $export_data = get_option("mailerpress_export_{$export_id}");
        if (!$export_data) {
            return new \WP_REST_Response(['message' => 'Export not found.'], 404);
        }

        if ($token !== $export_data['token']) {
            return new \WP_REST_Response(['message' => 'Invalid token.'], 403);
        }

        if (time() > $export_data['expires']) {
            return new \WP_REST_Response(['message' => 'Link expired.'], 410);
        }

        $zip_path = $export_data['zip_path'];
        if (!file_exists($zip_path)) {
            return new \WP_REST_Response(['message' => 'File not found.'], 404);
        }

        header('Content-Type: application/zip');
        header('Content-Disposition: attachment; filename="' . basename($zip_path) . '"');
        header('Content-Length: ' . filesize($zip_path));
        readfile($zip_path);
        exit;
    }

    #[Endpoint(
        'contact-note/(?P<contact_id>\d+)',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function getContactNote(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $contact_id = intval($request['contact_id']);
        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT_NOTE);

        $results = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT note_id, content, created_at, updated_at 
             FROM $table_name 
             WHERE contact_id = %d 
             ORDER BY created_at DESC",
                $contact_id
            ),
            ARRAY_A
        );

        return new WP_REST_Response([
            'success' => true,
            'notes' => $results
        ]);
    }

    #[Endpoint(
        'contact/note',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function addContactNote(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $contact_id = intval($request->get_param('contact_id'));
        $content = sanitize_textarea_field($request->get_param('content'));

        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT_NOTE);

        $result = $wpdb->insert(
            $table_name,
            [
                'contact_id' => $contact_id,
                'content' => $content,
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
            ],
            [
                '%d',
                '%s',
                '%s',
                '%s',
            ]
        );

        if ($result === false) {
            return new WP_Error('insert_failed', 'Failed to insert contact note.', ['status' => 500]);
        }

        $note_id = $wpdb->insert_id;

        // Fetch the newly inserted row
        $new_note = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT note_id, contact_id, content, created_at, updated_at FROM $table_name WHERE note_id = %d",
                $note_id
            ),
            ARRAY_A
        );

        return new WP_REST_Response([
            'success' => true,
            'note' => $new_note,
        ]);
    }

    #[Endpoint(
        'contacts/all',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function all(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $contact_table = Tables::get(Tables::MAILERPRESS_CONTACT);
        $contact_tags_table = Tables::get(Tables::CONTACT_TAGS);
        $tags_table = Tables::get(Tables::MAILERPRESS_TAGS);
        $lists_table = Tables::get(Tables::MAILERPRESS_LIST);
        $contact_lists_table = Tables::get(Tables::MAILERPRESS_CONTACT_LIST);

        $per_page = isset($_GET['perPages']) ? (int)(wp_unslash($_GET['perPages'])) : 20;
        $page = isset($_GET['paged']) ? (int)(wp_unslash($_GET['paged'])) : 1;
        $offset = ($page - 1) * $per_page;
        $search = $request->get_param('search');

        $where = '1=1';
        $params = [];
        $joins = '';

        // Search filter
        if (!empty($search)) {
            $where .= ' AND c.email LIKE %s';
            $params[] = '%' . $wpdb->esc_like($search) . '%';
        }

        // Subscription status filter
        if (!empty($_GET['subscription_status'])) {
            $where .= ' AND c.subscription_status = %s';
            $params[] = sanitize_text_field(wp_unslash($_GET['subscription_status']));
        }

        // Filter by lists
        $listParam = $request->get_param('list');
        $listIds = [];

        if (!empty($listParam) && is_array($listParam)) {
            foreach ($listParam as $list) {
                if (isset($list['id'])) {
                    $listIds[] = (int)$list['id'];
                }
            }
        }

        if (!empty($listIds)) {
            $joins .= " INNER JOIN {$contact_lists_table} cl ON cl.contact_id = c.contact_id ";
            $placeholders = implode(',', array_fill(0, count($listIds), '%d'));
            $where .= " AND cl.list_id IN ($placeholders)";
            $params = array_merge($params, $listIds);
        }

        // Filter by tags
        $tagParam = $request->get_param('tag');
        $tagIds = [];

        if (!empty($tagParam) && is_array($tagParam)) {
            foreach ($tagParam as $tag) {
                if (isset($tag['id'])) {
                    $tagIds[] = (int)$tag['id'];
                }
            }
        }

        if (!empty($tagIds)) {
            $joins .= " INNER JOIN {$contact_tags_table} ct_filter ON ct_filter.contact_id = c.contact_id ";
            $placeholders = implode(',', array_fill(0, count($tagIds), '%d'));
            $where .= " AND ct_filter.tag_id IN ($placeholders)";
            $params = array_merge($params, $tagIds);
        }

        // Order
        $orderBy = 'c.contact_id DESC';
        if (!empty($request->get_param('orderby')) && !empty($request->get_param('order'))) {
            $orderBy = \sprintf(
                'c.%s %s',
                esc_sql($request->get_param('orderby')),
                esc_sql($request->get_param('order'))
            );
        }

        // Query contacts
        $contacts = $wpdb->get_results($wpdb->prepare("
        SELECT
            c.*,
            c.contact_id as id
        FROM {$contact_table} c
        {$joins}
        WHERE {$where}
        ORDER BY {$orderBy}
        LIMIT %d OFFSET %d", [...$params, $per_page, $offset]));

        $contact_ids = array_map(static function ($contact) {
            return $contact->contact_id;
        }, $contacts);

        $placeholders = implode(',', array_fill(0, \count($contact_ids), '%d'));

        // Fetch tags
        $tags_by_contact = [];
        if (!empty($contact_ids)) {
            $tags_results = $wpdb->get_results(
                $wpdb->prepare("
                SELECT
                    ct.contact_id,
                    t.tag_id,
                    t.name
                FROM {$contact_tags_table} ct
                INNER JOIN {$tags_table} t ON ct.tag_id = t.tag_id
                WHERE ct.contact_id IN ({$placeholders})
            ", ...$contact_ids)
            );

            foreach ($tags_results as $tag) {
                $tags_by_contact[$tag->contact_id][] = [
                    'tags_id' => $tag->tag_id,
                    'tag_name' => $tag->name,
                ];
            }
        }

        // Fetch lists
        $lists_by_contact = [];
        if (!empty($contact_ids)) {
            $lists_results = $wpdb->get_results(
                $wpdb->prepare("
                SELECT
                    cl.contact_id,
                    cl.list_id,
                    l.name as list_name
                FROM {$contact_lists_table} cl
                INNER JOIN {$lists_table} l ON cl.list_id = l.list_id
                WHERE cl.contact_id IN ({$placeholders})
            ", ...$contact_ids)
            );

            foreach ($lists_results as $list) {
                $lists_by_contact[$list->contact_id][] = [
                    'list_id' => $list->list_id,
                    'list_name' => $list->list_name,
                ];
            }
        }

        // Merge all
        foreach ($contacts as &$contact) {
            $contact->tags = $tags_by_contact[$contact->contact_id] ?? [];
            $contact->contact_lists = $lists_by_contact[$contact->contact_id] ?? [];
        }

        // Count for pagination
        $total_count = $wpdb->get_var($wpdb->prepare("
        SELECT COUNT(DISTINCT c.contact_id)
        FROM {$contact_table} c
        {$joins}
        WHERE {$where}", $params));

        $total_pages = ceil($total_count / $per_page);

        return new \WP_REST_Response([
            'posts' => $contacts,
            'pages' => $total_pages,
            'count' => $total_count,
        ], 200);
    }

    #[Endpoint(
        'contact',
        methods: 'POST'
    )]
    public function add(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);
        $email = sanitize_email($request->get_param('contactEmail'));
        $first_name = sanitize_text_field($request->get_param('contactFirstName'));
        $last_name = sanitize_text_field($request->get_param('contactLastName'));
        $subscription_status = sanitize_text_field($request->get_param('contactStatus'));
        $contactTags = $request->get_param('tags');
        $contactLists = $request->get_param('lists');
        $optinSource = $request->get_param('opt_in_source') ?? 'unknown';
        $optinDetails = $request->get_param('optin_details') ?? '';

        // Vérifier si l'email existe déjà
        $existing_contact = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table_name} WHERE email = %s", $email));

        if ($existing_contact) {
            $wpdb->update(
                $table_name,
                [
                    'first_name' => $first_name,
                    'last_name'  => $last_name,
                    'updated_at' => current_time('mysql'),
                ],
                ['contact_id' => $existing_contact->contact_id],
                ['%s', '%s', '%s'],
                ['%d']
            );

            $contactId = $existing_contact->contact_id;
            do_action('mailerpress_contact_updated', $contactId);

            return new \WP_REST_Response([
                'success' => true,
                'message' => 'Contact updated successfully.',
                'data' => [
                    'contact_id' => $contactId,
                    'updated' => true,
                ],
            ]);
        }

        // Générer un token unique pour unsubscribe
        $unsubscribe_token = wp_generate_uuid4();
        $singUpConfirmation = get_option('mailerpress_signup_confirmation', wp_json_encode([
            'enableSignupConfirmation' => true,
            'emailSubject' => 'Confirm your subscription to [site:title]',
            'emailContent' => 'Hello [contact:firstName] [contact:lastName],

You have received this email regarding your subscription to [site:title]. Please confirm it to receive emails from us:

[activation_link]Click here to confirm your subscription[/activation_link]

If you received this email in error, simply delete it. You will no longer receive emails from us if you do not confirm your subscription using the link above.

Thank you,

<a target="_blank" href=" [site:homeURL]">[site:title]</a>'
        ]));

        if (is_string($singUpConfirmation)) {
            $singUpConfirmation = json_decode($singUpConfirmation, true);
        }

        // Insérer le contact dans la base de données
        $result = $wpdb->insert(
            $table_name,
            [
                'email' => $email,
                'first_name' => $first_name,
                'last_name' => $last_name,
                'subscription_status' => $optinSource !== 'manual' ? (!empty($singUpConfirmation) && true === $singUpConfirmation['enableSignupConfirmation']) ? 'pending' : $subscription_status : $subscription_status,
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
                'unsubscribe_token' => $unsubscribe_token,
                'opt_in_source' => $optinSource,
                'opt_in_details' => $optinDetails,
                'access_token' => bin2hex(random_bytes(32))
            ],
            [
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
                '%s',
            ]
        );

        if (false === $result) {
            return new \WP_Error('db_error', 'Could not insert contact into the database.', ['status' => 500]);
        }

        $contactId = $wpdb->insert_id;

        do_action('mailerpress_contact_created', $contactId);

        if (is_array($contactTags)) {
            foreach ($contactTags as $tag) {
                $wpdb->insert(Tables::get(Tables::CONTACT_TAGS), [
                    'contact_id' => $contactId,
                    'tag_id' => $tag['id'],
                ]);

                do_action('mailerpress_contact_tag_added', $contactId, $tag['id']);
            }
        }


        if (is_array($contactLists)) {
            foreach ($contactLists as $list) {
                $wpdb->insert(Tables::get(Tables::MAILERPRESS_CONTACT_LIST), [
                    'contact_id' => $contactId,
                    'list_id' => $list['id'],
                ]);

                do_action('mailerpress_contact_list_added', $contactId, $list['id']);
            }
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Contact added successfully.',
            'data' => [
                'contact_id' => $wpdb->insert_id,
            ],
        ]);
    }

    #[Endpoint(
        'contacts',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function edit(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);
        $tags_table = Tables::get(Tables::CONTACT_TAGS);
        $lists_table = Tables::get(Tables::MAILERPRESS_CONTACT_LIST);

        $newStatus = $request->get_param('newStatus');
        $ids = $request->get_param('ids');
        $tags = $request->get_param('tags') ?? [];
        $lists = $request->get_param('lists') ?? [];
        $removeTags = $request->get_param('removeTags') ?? [];
        $removeLists = $request->get_param('removeLists') ?? [];

        if (null === $ids) {
            if (!empty($newStatus)) {
                $updatedRows = $wpdb->query($wpdb->prepare(
                    "UPDATE {$table_name} SET subscription_status = %s", esc_html($newStatus)
                ));

                if ($updatedRows === false) {
                    return new \WP_REST_Response([
                        'success' => false,
                        'message' => 'Database update failed for subscription status.',
                        'error' => $wpdb->last_error,
                    ], 500);
                }
            }

            // Add tags to all contacts
            if (!empty($tags)) {
                foreach ($tags as $tag) {
                    // Insert tag for all contacts, ignoring duplicates
                    $inserted = $wpdb->query(
                        $wpdb->prepare(
                            "INSERT IGNORE INTO {$tags_table} (contact_id, tag_id)
                     SELECT contact_id, %d FROM {$table_name}",
                            $tag['id']
                        )
                    );
                    if ($inserted === false) {
                        return new \WP_REST_Response([
                            'success' => false,
                            'message' => 'Failed to add tags to all contacts.',
                            'error' => $wpdb->last_error,
                        ], 500);
                    }

                    // Trigger action after tag is added to all contacts
                    do_action('mailerpress_contact_tag_added_bulk', $tag['id']);
                }
            }

            // Add lists to all contacts
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Insert list for all contacts, ignoring duplicates
                    $inserted = $wpdb->query(
                        $wpdb->prepare(
                            "INSERT IGNORE INTO {$lists_table} (contact_id, list_id)
                     SELECT contact_id, %d FROM {$table_name}",
                            $list['id']
                        )
                    );
                    if ($inserted === false) {
                        return new \WP_REST_Response([
                            'success' => false,
                            'message' => 'Failed to add lists to all contacts.',
                            'error' => $wpdb->last_error,
                        ], 500);
                    }

                    // Trigger action after list is added to all contacts
                    do_action('mailerpress_contact_list_added_bulk', $list['id']);
                }
            }

            // Optionally, trigger an action after all contacts have been updated
            do_action('mailerpress_all_contacts_updated');


        } else {
            foreach ($ids as $id) {
                if (!empty($newStatus)) {
                    $wpdb->update(
                        $table_name,
                        ['subscription_status' => esc_html($newStatus)],
                        ['contact_id' => esc_html($id)],
                        ['%s'],
                        ['%d']
                    );
                }

                // Add new tags
                foreach ($tags as $tag) {
                    $wpdb->insert(
                        $tags_table,
                        [
                            'contact_id' => $id,
                            'tag_id' => $tag['id']
                        ],
                        ['%d', '%d']
                    );
                    do_action('mailerpress_contact_tag_added', $id, $tag['id']);
                }

                // Remove tags
                foreach ($removeTags as $tag) {
                    $wpdb->delete(
                        $tags_table,
                        [
                            'contact_id' => $id,
                            'tag_id' => $tag['id']
                        ],
                        ['%d', '%d']
                    );
                    do_action('mailerpress_contact_tag_removed', $id, $tag['id']);
                }

                // Add new lists
                foreach ($lists as $list) {
                    $wpdb->insert(
                        $lists_table,
                        [
                            'contact_id' => $id,
                            'list_id' => $list['id']
                        ],
                        ['%d', '%d']
                    );
                    do_action('mailerpress_contact_list_added', $id, $list['id']);
                }

                // Remove lists
                foreach ($removeLists as $list) {
                    $wpdb->delete(
                        $lists_table,
                        [
                            'contact_id' => $id,
                            'list_id' => $list['id']
                        ],
                        ['%d', '%d']
                    );
                    do_action('mailerpress_contact_list_removed', $id, $list['id']);
                }
            }
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Contacts updated successfully.',
        ], 200);
    }

    #[Endpoint(
        'contact/(?P<id>\d+)',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function editSingle(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);
        $tags_table = Tables::get(Tables::CONTACT_TAGS);
        $lists_table = Tables::get(Tables::MAILERPRESS_CONTACT_LIST);

        $id = (int)$request->get_param('id');
        $newStatus = $request->get_param('newStatus');
        $tags = $request->get_param('tags') ?? [];
        $lists = $request->get_param('lists') ?? [];
        $removeTags = $request->get_param('removeTags') ?? [];
        $removeLists = $request->get_param('removeLists') ?? [];

        if (empty($id)) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Missing contact ID.',
            ], 400);
        }

        if (!empty($newStatus)) {
            $wpdb->update(
                $table_name,
                ['subscription_status' => esc_html($newStatus)],
                ['contact_id' => $id],
                ['%s'],
                ['%d']
            );
        }

        // Add tags
        foreach ($tags as $tag) {
            $wpdb->insert(
                $tags_table,
                [
                    'contact_id' => $id,
                    'tag_id' => $tag['id']
                ],
                ['%d', '%d']
            );
            do_action('mailerpress_contact_tag_added', $id, $tag['id']);
        }

        // Remove tags
        foreach ($removeTags as $tag) {
            $wpdb->delete(
                $tags_table,
                [
                    'contact_id' => $id,
                    'tag_id' => $tag['id']
                ],
                ['%d', '%d']
            );
            do_action('mailerpress_contact_tag_removed', $id, $tag['id']);
        }

        // Add lists
        foreach ($lists as $list) {
            $wpdb->insert(
                $lists_table,
                [
                    'contact_id' => $id,
                    'list_id' => $list['id']
                ],
                ['%d', '%d']
            );
            do_action('mailerpress_contact_list_added', $id, $list['id']);
        }

        // Remove lists
        foreach ($removeLists as $list) {
            $wpdb->delete(
                $lists_table,
                [
                    'contact_id' => $id,
                    'list_id' => $list['id']
                ],
                ['%d', '%d']
            );
            do_action('mailerpress_contact_list_removed', $id, $list['id']);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Contact updated successfully.',
        ], 200);
    }

    #[Endpoint(
        '/contact',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function delete(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Get the contact IDs from the request (expects an array)
        $contact_ids = $request->get_param('ids'); // Assuming 'ids' is an array of contact IDs

        // Validate that the input is an array
        if (!\is_array($contact_ids) || empty($contact_ids)) {
            return new \WP_Error(
                'invalid_input',
                __('Contact IDs must be an array and cannot be empty', 'mailerpress'),
                ['status' => 400]
            );
        }

        // Verify if the contacts exist
        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);
        $placeholders = implode(',', array_fill(0, \count($contact_ids), '%d'));
        $existing_contacts = $wpdb->get_results($wpdb->prepare(
            "SELECT contact_id FROM {$table_name} WHERE contact_id IN ({$placeholders})",
            ...$contact_ids
        ));

        // If any contact does not exist, return an error
        if (\count($existing_contacts) !== \count($contact_ids)) {
            return new \WP_Error(
                'contact_not_found',
                __('One or more contacts were not found.', 'mailerpress'),
                ['status' => 404]
            );
        }

        // Delete the contacts
        $deleted = $wpdb->query($wpdb->prepare("DELETE FROM {$table_name} WHERE contact_id IN ({$placeholders})", ...
            $contact_ids));

        if ($deleted) {
            return new \WP_REST_Response([
                'success' => true,
                'message' => __('Contacts deleted successfully', 'mailerpress'),
                'deleted_contacts' => $contact_ids,
            ], 200);
        }

        return new \WP_Error(
            'delete_failed',
            __('Failed to delete the contacts.', 'mailerpress'),
            ['status' => 500]
        );
    }


    #[Endpoint(
        '/contact/list/(?P<id>\d+)',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function deleteContactList(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Route param: contact ID from URL
        $contact_id = (int)$request->get_param('id');
        // Query param: list ID from ?listId=123
        $list_id = (int)$request->get_param('listId');

        if (!$contact_id || !$list_id) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Missing contact_id or list_id.',
            ], 400);
        }

        $table = Tables::get(Tables::MAILERPRESS_CONTACT_LIST);

        $deleted = $wpdb->delete(
            $table,
            [
                'contact_id' => $contact_id,
                'list_id' => $list_id,
            ],
            [
                '%d',
                '%d',
            ]
        );

        if ($deleted === false) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Database error occurred.',
            ], 500);
        }

        if ($deleted === 0) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'No matching record found.',
            ], 404);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Contact removed from the list.',
        ], 200);
    }


    #[Endpoint(
        '/contact/tag/(?P<id>\d+)',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canManageLists'],
    )]
    public function deleteContactTag(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Route param: contact ID from URL
        $contact_id = (int)$request->get_param('id');
        // Query param: list ID from ?listId=123
        $tag_id = (int)$request->get_param('tagId');

        if (!$tag_id || !$contact_id) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Missing contact_id or tag_id.',
            ], 400);
        }

        $table = Tables::get(Tables::CONTACT_TAGS);

        $deleted = $wpdb->delete(
            $table,
            [
                'contact_id' => $contact_id,
                'tag_id' => $tag_id,
            ],
            [
                '%d',
                '%d',
            ]
        );

        if ($deleted === false) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'Database error occurred.',
            ], 500);
        }

        if ($deleted === 0) {
            return new \WP_REST_Response([
                'success' => false,
                'message' => 'No matching record found.',
            ], 404);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Contact removed from the tag.',
        ], 200);
    }


    #[Endpoint(
        '/contact/all',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canManageAudience'],
    )]
    public function deleteAll(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;
        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);

        // Delete all rows in the table
        $result = $wpdb->query("DELETE FROM {$table_name}");

        if (false === $result) {
            return new \WP_REST_Response(['message' => 'Failed to delete contacts.'], 500);
        }

        return new \WP_REST_Response(
            ['message' => 'All contacts have been deleted successfully.', 'deleted' => $result],
            200
        );
    }

    #[Endpoint(
        'contacts/bactches/pending',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function contactBatchImport(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Replace `wp_contact_batches` with your actual table name
        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT_BATCHES);

        // Query all batches with status "pending"
        $results = $wpdb->get_results($wpdb->prepare(
            "SELECT * FROM `{$table_name}` WHERE `status` = %s",
            'pending'
        ), ARRAY_A);

        if (\is_array($results) && !empty($results)) {
            return new \WP_REST_Response(
                $results,  // This will be returned as a valid JSON array
                200
            );
        }

        return new \WP_REST_Response(
            ['message' => 'No pending records found'],
            200
        );
    }

    #[Endpoint(
        'contacts',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView'],
        args: [
            'tags' => [
                'required' => false,
                'type' => 'array',
            ],
        ],
    )]
    public function response(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $tags = $request->get_param('tags') ?? '';
        $lists = $request->get_param('lists') ?? '';

        return new \WP_REST_Response(
            Kernel::getContainer()->get(\MailerPress\Models\Contacts::class)->getContactsWithTagsAndLists(
                $lists,
                $tags
            ),
            200
        );
    }

    #[Endpoint(
        'contact/export',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function export(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $table_name = Tables::get(Tables::MAILERPRESS_CONTACT);
        $contact_ids = $request->get_param('contact_ids');
        $email = sanitize_email($request->get_param('email'));
        $export_id = uniqid(); // utilisé pour stocker les fichiers temporairement
        $batch_size = 200;
        $i = 0;

        if (is_array($contact_ids) && !empty($contact_ids)) {
            // Mode ciblé : on divise les contact_ids en batchs
            $chunks = array_chunk($contact_ids, $batch_size);

            foreach ($chunks as $index => $chunk) {
                as_schedule_single_action(
                    time() + $index,
                    'mailerpress_export_contact_batch',
                    [
                        'export_id' => $export_id,
                        'status' => null,
                        'offset' => null,
                        'contact_ids' => $chunk,
                    ],
                    'mailerpress'
                );
            }

            $total_batches = count($chunks);
        } else {
            // Mode complet : on récupère tous les contacts par status
            $count = (int)$wpdb->get_var("SELECT COUNT(*) FROM $table_name");

            if ($count === 0) {
                return new \WP_REST_Response([
                    'success' => false,
                    'message' => 'Aucun contact à exporter.',
                ], 200);
            }

            $statuses = ['subscribed', 'unsubscribed'];
            $total_batches = 0;

            foreach ($statuses as $status) {
                $status_count = (int)$wpdb->get_var($wpdb->prepare(
                    "SELECT COUNT(*) FROM $table_name WHERE subscription_status = %s",
                    $status
                ));

                if ($status_count === 0) {
                    continue;
                }

                $offsets = [];
                for ($offset = 0; $offset < $status_count; $offset += $batch_size) {
                    $offsets[] = $offset;
                }

                foreach ($offsets as $offset) {
                    as_schedule_single_action(
                        time() + $i,
                        'mailerpress_export_contact_batch',
                        [
                            'export_id' => $export_id,
                            'status' => $status,
                            'offset' => $offset,
                        ],
                        'mailerpress'
                    );
                    $i++;
                }

                $total_batches += count($offsets);
            }
        }

        // Planifie la finalisation une fois tous les batchs théoriquement terminés
        as_schedule_single_action(
            time() + ($total_batches * 2),
            'mailerpress_finalize_export_contact_zip',
            ['export_id' => $export_id, 'email' => $email],
            'mailerpress'
        );

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Export démarré',
            'export_id' => $export_id,
        ]);
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'contacts/import',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function batchImport(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;
        $data = $request->get_param('data') ?? [];

        $batch_data = [
            'tags' => wp_json_encode($data['tags']),
            'lists' => wp_json_encode($data['lists']),
            'count' => \count($data['mapping']),
            'subscription_status' => $data['status'],
        ];

        $wpdb->insert(
            Tables::get(Tables::MAILERPRESS_CONTACT_BATCHES),
            $batch_data
        );

        $batch_id = $wpdb->insert_id;

        if ($batch_id) {
            // Split contacts into chunks (e.g., 500 per chunk)
            $chunks = array_chunk($data['mapping'], 100);

            foreach ($chunks as $chunk) {
                // Save chunk to the temporary table
                $wpdb->insert(Tables::get(Tables::MAILERPRESS_IMPORT_CHUNKS), [
                    'batch_id' => $batch_id,
                    'chunk_data' => wp_json_encode($chunk),
                ]);

                $chunk_id = $wpdb->insert_id;

                if (!wp_next_scheduled('process_import_chunk', [$chunk_id])) {
                    // Planifie un événement unique dans 1 minute
                    as_schedule_single_action(
                        time() + 10,
                        'process_import_chunk',
                        [$chunk_id, $data['forceUpdate']]
                    );
                }
            }
        }

        return new \WP_REST_Response(
            [],
            200
        );
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'contact/import',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function importContact(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;
        $contact = $request->get_param('item');
        $status = sanitize_text_field($request->get_param('status')) ?? 'pending';
        $contactTags = $request->get_param('tags');
        $contactLists = $request->get_param('lists');
        $forceUpdate = $request->get_param('forceUpdate');
        $contactTable = Tables::get(Tables::MAILERPRESS_CONTACT);

        $contact_id = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT contact_id FROM {$contactTable} WHERE email = %s LIMIT 1",
                trim($contact['email'], '"')
            )
        );


        if (null === $contact_id) {
            $contact_data = [
                'email' => sanitize_email($contact['email'] ?? ''),
                'first_name' => sanitize_text_field(trim($contact['first_name'] ?? '', '"')),
                'last_name' => sanitize_text_field(trim($contact['last_name'] ?? '', '"')),
                'subscription_status' => sanitize_text_field($status),
                'unsubscribe_token' => wp_generate_uuid4(),
                'created_at' => current_time('mysql'),
                'updated_at' => current_time('mysql'),
                'opt_in_source' => 'batch_import_file',
                'access_token' => bin2hex(random_bytes(32))
            ];

            // Insérer les données dans `wp_mailerpress_contact`
            $result = $wpdb->insert($contactTable, $contact_data);

            if (false !== $result) {
                $contactId = $wpdb->insert_id;

                do_action('mailerpress_contact_created', $contactId);

                foreach ($contactTags as $tag) {
                    $wpdb->insert(Tables::get(Tables::CONTACT_TAGS), [
                        'contact_id' => $contactId,
                        'tag_id' => $tag['id'],
                    ]);

                    do_action('mailerpress_contact_tag_added', $contactId, $tag['id']);
                }

                foreach ($contactLists as $list) {
                    $wpdb->insert(Tables::get(Tables::MAILERPRESS_CONTACT_LIST), [
                        'contact_id' => $contactId,
                        'list_id' => $list['id'],
                    ]);

                    do_action('mailerpress_contact_list_added', $contactId, $list['id']);
                }

                return new \WP_REST_Response(
                    $result,
                    200
                );
            }
        } else {
            if (true === $forceUpdate || '1' === $forceUpdate) {
                $result = $wpdb->update(
                    $contactTable,
                    [
                        'subscription_status' => $status,
                        'updated_at' => current_time('mysql'),
                        'first_name' => sanitize_text_field(trim($contact['first_name'] ?? '', '"')),
                        'last_name' => sanitize_text_field(trim($contact['last_name'] ?? '', '"')),
                    ], // Data to update
                    ['contact_id' => $contact_id] // Where condition
                );

                if (false !== $result) {
                    if (is_array($contactTags)) {
                        foreach ($contactTags as $tag) {
                            $wpdb->insert(Tables::get(Tables::CONTACT_TAGS), [
                                'contact_id' => $contact_id,
                                'tag_id' => $tag['id'],
                            ]);

                            do_action('mailerpress_contact_tag_added', $contact_id, $tag['id']);
                        }
                    }

                    if (is_array($contactLists)) {
                        foreach ($contactLists as $list) {
                            $wpdb->insert(Tables::get(Tables::MAILERPRESS_CONTACT_LIST), [
                                'contact_id' => $contact_id,
                                'list_id' => $list['id'],
                            ]);

                            do_action('mailerpress_contact_list_added', $contact_id, $list['id']);
                        }
                    }


                    return new \WP_REST_Response(
                        $result,
                        200
                    );
                }
            }
        }

        usleep(100000);

        return new \WP_REST_Response(
            [],
            400
        );
    }
}
