<?php

declare(strict_types=1);

namespace MailerPress\Api;

\defined('ABSPATH') || exit;

use ActionScheduler_Store;
use DateTime;
use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\Attributes\Endpoint;
use MailerPress\Core\Capabilities;
use MailerPress\Core\EmailManager\EmailServiceManager;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\HtmlParser;
use MailerPress\Core\Interfaces\ContactFetcherInterface;
use MailerPress\Core\Kernel;
use MailerPress\Models\Batch;
use MailerPress\Models\Contacts;
use MailerPress\Services\ClassicContactFetcher;
use MailerPress\Services\SegmentContactFetcher;
use WP_Error;
use WP_REST_Request;
use WP_REST_Response;

class Campaigns
{
    #[Endpoint(
        'batch-opened-contacts',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function batchOpenedContacts(\WP_REST_Request $request): \WP_REST_Response
    {
        global $wpdb;

        $batch_id = absint($request->get_param('batch_id'));
        if (!$batch_id) {
            return new \WP_REST_Response(['error' => 'No batch_id provided'], 400);
        }

        $paged = max(1, (int)$request->get_param('paged') ?? 1);
        $per_page = max(1, (int)($request->get_param('perPage') ?? 10));
        $search = $request->get_param('search');

        $tracking_table = Tables::get(Tables::MAILERPRESS_EMAIL_TRACKING);
        $contact_table = Tables::get(Tables::MAILERPRESS_CONTACT);

        $offset = ($paged - 1) * $per_page;

        $query_params = [$batch_id];

        // Base WHERE clause
        $where = "WHERE t.batch_id = %d AND t.opened_at IS NOT NULL";

        if (!empty($search)) {
            $where .= " AND (c.email LIKE %s OR c.first_name LIKE %s OR c.last_name LIKE %s)";
            $like_search = '%' . $wpdb->esc_like($search) . '%';
            $query_params[] = $like_search;
            $query_params[] = $like_search;
            $query_params[] = $like_search;
        }

        // Count total rows
        $count_sql = "
        SELECT COUNT(*)
        FROM {$tracking_table} AS t
        INNER JOIN {$contact_table} AS c ON t.contact_id = c.contact_id
        {$where}
    ";
        $total_rows = (int)$wpdb->get_var($wpdb->prepare($count_sql, ...$query_params));
        $total_pages = ceil($total_rows / $per_page);

        // Main SELECT query
        $query_sql = "
        SELECT 
            t.contact_id,
            t.opened_at,
            t.clicks,
            c.email,
            c.first_name,
            c.last_name
        FROM {$tracking_table} AS t
        INNER JOIN {$contact_table} AS c ON t.contact_id = c.contact_id
        {$where}
        ORDER BY t.opened_at DESC
        LIMIT %d OFFSET %d
    ";

        // Append LIMIT and OFFSET
        $query_params[] = $per_page;
        $query_params[] = $offset;

        $results = $wpdb->get_results($wpdb->prepare($query_sql, ...$query_params), ARRAY_A);

        return new \WP_REST_Response([
            'posts' => $results,
            'count' => $total_rows,
            'pages' => $total_pages,
            'current_page' => $paged,
        ], 200);
    }

    #[Endpoint(
        'campaign-status',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function campaignStatus(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $ids = $request->get_param('ids');
        if (!$ids) {
            return new WP_REST_Response(['error' => 'No IDs provided'], 400);
        }

        $ids = array_map('absint', explode(',', $ids));
        if (empty($ids)) {
            return new WP_REST_Response(['error' => 'Invalid IDs provided'], 400);
        }

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $batches_table = $wpdb->prefix . 'mailerpress_email_batches';

        $query = $wpdb->prepare("
        SELECT 
            c.campaign_id, 
            c.name AS title, 
            c.subject, 
            c.status, 
            c.batch_id AS batch, 
            c.updated_at,
            c.content_html,
            c.config
        FROM {$table} AS c
        LEFT JOIN {$batches_table} AS b ON c.batch_id = b.id
        WHERE c.campaign_id IN ($placeholders)
    ", ...$ids);

        $results = $wpdb->get_results($query);

        foreach ($results as &$result) {
            $result->content_html = !empty($result->content_html) ? json_decode($result->content_html, true) : null;
            $result->config = !empty($result->config) ? json_decode($result->config, true) : null;

            $result->batch = $result->batch ? Kernel::getContainer()->get(Batch::class)->getById(
                $result->batch,
                true
            ) : null;

            $result->statistics = !empty($result->batch['id']) ? Kernel::getContainer()->get(Batch::class)->getStatistics($result->batch['id']) : null;
        }

        return new WP_REST_Response($results, 200);
    }

    #[Endpoint(
        'campaign-status-lock',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function campaignStatusLock(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $ids = $request->get_param('ids');
        if (!$ids) {
            return new WP_REST_Response(['error' => 'No IDs provided'], 400);
        }

        $ids = array_map('absint', explode(',', $ids));
        if (empty($ids)) {
            return new WP_REST_Response(['error' => 'Invalid IDs provided'], 400);
        }

        $placeholders = implode(',', array_fill(0, count($ids), '%d'));
        $table = $wpdb->prefix . 'mailerpress_campaigns';

        $query = $wpdb->prepare("
        SELECT 
            campaign_id,
            editing_user_id,
            editing_started_at
        FROM {$table}
        WHERE campaign_id IN ($placeholders)
    ", ...$ids);

        $results = $wpdb->get_results($query);

        $now = current_time('timestamp'); // Unix timestamp in site’s timezone

        $LOCK_TIMEOUT = 2 * 60; // 5 minutes

        foreach ($results as &$campaign) {
            $campaign->locked = false;
            $campaign->locked_by = null;

            if (!empty($campaign->editing_started_at) && $campaign->editing_user_id) {
                $editing_time = strtotime($campaign->editing_started_at);
                $elapsed = $now - $editing_time;
                if ($elapsed > $LOCK_TIMEOUT) {
                    // Only unlock if lock is stale
                    $wpdb->query(
                        $wpdb->prepare(
                            "UPDATE {$table} 
                             SET editing_user_id = NULL, editing_started_at = NULL 
                             WHERE campaign_id = %d",
                            $campaign->campaign_id
                        )
                    );
                    $campaign->editing_user_id = null;
                    $campaign->editing_started_at = null;
                } else {
                    // Lock is valid
                    $campaign->locked = true;
                    $user = get_userdata($campaign->editing_user_id);
                    $campaign->locked_by = $user ? $user->display_name : null;
                }
            }
        }

        return new WP_REST_Response($results, 200);
    }


    #[Endpoint(
        'campaign/batches',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function getBatchesForCampaign(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');
        $paged = max((int)($request->get_param('paged') ?? 1), 1);
        $per_page = max((int)($request->get_param('perPage') ?? 10), 1);
        $offset = ($paged - 1) * $per_page;

        if (!$campaign_id) {
            return new \WP_Error('invalid_campaign', __('Invalid campaign ID', 'mailerpress'), ['status' => 400]);
        }

        $table_campaigns = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $table_batches = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);

        $campaign = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$table_campaigns} WHERE campaign_id = %d", $campaign_id)
        );

        if (!$campaign) {
            return new \WP_Error('campaign_not_found', __('Campaign not found', 'mailerpress'), ['status' => 404]);
        }

        $total_rows = (int)$wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM {$table_batches} WHERE campaign_id = %d", $campaign_id)
        );

        $total_pages = (int)ceil($total_rows / $per_page);

        $batches = $wpdb->get_results(
            $wpdb->prepare(
                "SELECT id FROM {$table_batches} WHERE campaign_id = %d ORDER BY created_at DESC LIMIT %d OFFSET %d",
                $campaign_id,
                $per_page,
                $offset
            )
        );

        $batchService = Kernel::getContainer()->get(Batch::class);
        $results = [];

        foreach ($batches as $batch) {
            $batch_id = (int)$batch->id;
            $batch_data = $batchService->getById($batch_id, true);
            $statistics = $batchService->getStatistics($batch_id);

            $results[] = [
                'batch' => $batch_data,
                'created_at' => get_date_from_gmt($batch_data['created_at'], 'c'),
                'statistics' => $statistics,
            ];
        }

        return new \WP_REST_Response([
            'posts' => $results,
            'pages' => $total_pages,
            'count' => $total_rows,
            'current_page' => $paged,
            'per_page' => $per_page,
        ], 200);
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaigns',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function response(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $paged = $request->get_param('paged') ?? 1;
        $posts_per_page = $request->get_param('perPages') ?? 10;
        $search = $request->get_param('search');
        $statusParam = $request->get_param('status');

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        $statuses = [
            'draft',
            'mine',
            'sent',
            'in_progress',
            'scheduled',
            'pending',
            'error',
            'active',
            'inactive',
        ];


        if (!empty($statusParam)) {
            if (is_string($statusParam)) {
                $statuses = array_map('trim', explode(',', $statusParam));
            } elseif (is_array($statusParam)) {
                $statuses = array_map('trim', $statusParam);
            }
        }


        // Initialize query params early
        $query_params = [];

        // Base query
        $query = "
        SELECT c.campaign_id AS id, 
               c.user_id,
               c.name AS title, 
               c.subject, 
               c.status, 
               c.batch_id AS batch, 
               c.updated_at,
               c.created_at,
               c.campaign_type,
                c.editing_user_id,
                c.editing_started_at
        FROM {$table_name} AS c
        LEFT JOIN {$wpdb->prefix}mailerpress_email_batches AS b 
            ON c.batch_id = b.id
        WHERE 1=1
    ";

        $countQuery = "
        SELECT COUNT(c.campaign_id)
        FROM {$table_name} AS c
        LEFT JOIN {$wpdb->prefix}mailerpress_email_batches AS b 
            ON c.batch_id = b.id
        WHERE 1=1
    ";

        // Search filter
        if (!empty($search)) {
            $query .= ' AND c.name LIKE %s';
            $countQuery .= ' AND c.name LIKE %s';
            $query_params[] = '%' . $wpdb->esc_like($search) . '%';
        }

        // Status filter
        // Status filter
        if (!empty($statuses)) {
            $hasDraft = in_array('draft', $statuses, true);
            $hasMine = in_array('mine', $statuses, true);
            $filteredStatuses = array_filter($statuses, fn($status) => !in_array($status, ['draft', 'mine'], true));

            $query .= ' AND (';
            $countQuery .= ' AND (';

            $statusParts = [];

            if ($hasDraft) {
                $statusParts[] = "(c.status = 'draft' AND b.status IS NULL)";
            }

            if ($hasMine) {
                $currentUserId = get_current_user_id();
                $statusParts[] = $wpdb->prepare("(c.user_id = %d AND c.status != 'trash')", $currentUserId);
            }

            if (!empty($filteredStatuses)) {
                $placeholders = implode(',', array_fill(0, count($filteredStatuses), '%s'));
                $statusParts[] = "c.status IN ($placeholders)";
                $query_params = array_merge($query_params, $filteredStatuses);
            }

            // Combine all conditions with OR
            $query .= implode(' OR ', $statusParts) . ')';
            $countQuery .= implode(' OR ', $statusParts) . ')';
        }

        // Campaign type filter
        $campaignTypesRaw = $request->get_param('campaign_type');
        $campaignTypes = [];

        if (is_array($campaignTypesRaw)) {
            foreach ($campaignTypesRaw as $entry) {
                if (isset($entry['id']) && is_string($entry['id'])) {
                    $campaignTypes[] = sanitize_text_field($entry['id']);
                }
            }
        }

        if (!empty($campaignTypes)) {
            $placeholders = implode(',', array_fill(0, count($campaignTypes), '%s'));
            $query .= " AND c.campaign_type IN ($placeholders)";
            $countQuery .= " AND c.campaign_type IN ($placeholders)";
            $query_params = array_merge($query_params, $campaignTypes);
        }

        // Ordering
        if (!empty($request->get_param('orderby')) && !empty($request->get_param('order'))) {
            $query .= sprintf(
                ' ORDER BY c.%s %s',
                esc_sql($request->get_param('orderby')),
                esc_sql($request->get_param('order'))
            );
        } else {
            $query .= ' ORDER BY c.updated_at DESC';
        }

        // Pagination
        $offset = ($paged - 1) * $posts_per_page;
        $query .= " LIMIT {$offset}, {$posts_per_page}";

        // Final execution
        $results = $wpdb->get_results($wpdb->prepare($query, ...$query_params));

        foreach ($results as &$result) {
            $campaign = $wpdb->get_row(
                $wpdb->prepare("SELECT content_html, config FROM {$table_name} WHERE campaign_id = %d", $result->id)
            );

            $result->content_html = !empty($campaign->content_html) ? json_decode($campaign->content_html, true) : null;
            $result->config = !empty($campaign->config) ? json_decode($campaign->config, true) : null;

            $result->batch = $result->batch ? Kernel::getContainer()->get(Batch::class)->getById(
                $result->batch,
                true
            ) : null;

            $result->statistics = !empty($result->batch['id']) ? Kernel::getContainer()->get(Batch::class)->getStatistics($result->batch['id']) : null;

            if (!empty($result->user_id)) {
                $user = get_user_by('id', $result->user_id);
                $result->author = $user ? [
                    'name' => $user->display_name,
                    'email' => $user->user_email,
                    'avatar' => get_avatar_url($user->ID, ['size' => 256, 'default' => 'mystery'])
                ] : null;
            } else {
                $result->author = null;
            }

            // ✅ Lock info
            $canEdit = false;

            if ((int)$result->user_id === get_current_user_id()) {
                // User is the author → check own edit capability
                $canEdit = current_user_can(Capabilities::MANAGE_CAMPAIGNS);
            } else {
                // User is not the author → need edit_others capability
                $canEdit = current_user_can(Capabilities::EDIT_OTHERS_CAMPAIGNS);
            }

            $result->canEdit = $canEdit;

            $result->locked = !empty($result->editing_user_id);
            $result->locked = !empty($result->editing_user_id);
            $user = get_userdata($result->editing_user_id);

            if ($result->editing_user_id) {
                $result->locked_by = $user->display_name;
                $result->locked_since = $result->editing_started_at;

                $result->locked_avatar = get_avatar_url($user->ID, ['size' => 256, 'default' => 'mystery']);
            } else {
                $result->locked_by = null;
                $result->locked_since = null;
                if ($user instanceof WP_User) {
                    $result->locked_avatar = get_avatar_url($user->ID, [
                        'size' => 256,
                        'default' => 'mystery',
                    ]);
                } else {
                    $result->locked_avatar = get_avatar_url(0, [
                        'size' => 256,
                        'default' => 'mystery',
                    ]);
                }
            }
        }

        $total_rows = $wpdb->get_var($wpdb->prepare($countQuery, ...$query_params));
        $total_pages = ceil($total_rows / $posts_per_page);

        return new \WP_REST_Response([
            'posts' => $results,
            'pages' => $total_pages,
            'count' => $total_rows,
            'current_page' => $paged,
        ], 200);
    }

    #[Endpoint(
        'campaign/(?P<id>\d+)',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canView']
    )]
    public function getCampaignById(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Récupérer l'ID de la campagne depuis les paramètres de la requête
        $campaign_id = (int)$request->get_param('id');

        // Nom de la table des campagnes
        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Vérifier si la campagne existe
        $campaign = $wpdb->get_row(
            $wpdb->prepare("SELECT * FROM {$table_name} WHERE campaign_id = %d", $campaign_id),
            ARRAY_A
        );

        if (!$campaign) {
            return new \WP_Error('not_found', __('Campaign not found.', 'mailerpress'), ['status' => 404]);
        }

        // Décoder les champs JSON pour les rendre utilisables
        $campaign['content_html'] = !empty($campaign['content_html']) ? json_decode(
            $campaign['content_html'],
            true
        ) : null;
        $campaign['config'] = !empty($campaign['config']) ? json_decode($campaign['config'], true) : null;

        // Retourner la campagne en réponse
        return new \WP_REST_Response(
            [
                'title' => $campaign['name'],
                'status' => $campaign['status'],
                'json' => $campaign['content_html'],
                'config' => $campaign['config'],
                'batch' => '',
            ],
            200
        );
    }

    #[Endpoint(
        'campaigns',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function post(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;
        $name = esc_attr($request->get_param('title'));
        $meta = $request->get_param('meta');

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Prepare data for insertion
        $data = [
            'user_id' => get_current_user_id(),
            'name' => $name,
            'subject' => $meta['emailConfig']['campaignSubject'] ?? '',
            'status' => 'draft',
            'email_type' => $meta['emailConfig']['email_type'] ?? 'html',
            'content_html' => $meta['json'] ? wp_json_encode($meta['json']) : null,
            'config' => !empty($meta['emailConfig']) ? wp_json_encode($meta['emailConfig']) : null,
            'created_at' => current_time('mysql'),
            'updated_at' => current_time('mysql'),
        ];

        // Insert data into the database
        $inserted = $wpdb->insert($table_name, $data);

        if (false === $inserted) {
            return new \WP_Error('db_insert_error', __('Failed to create campaign.', 'mailerpress'), ['status' => 500]);
        }

        do_action('mailerpress_campaign_created', $inserted);

        // Return success response
        return new \WP_REST_Response($wpdb->insert_id, 201);
    }

    #[Endpoint(
        'campaign',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canDeleteCampaigns'],
    )]
    public function delete(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        if (!current_user_can(Capabilities::DELETE_EMAIL_CAMPAIGNS)) {
            return new \WP_Error(
                'forbidden',
                __('You do not have permission to delete campaigns.', 'mailerpress'),
                ['status' => 403]
            );
        }

        // Récupérer les IDs de campagnes depuis la requête
        $campaign_ids = $request->get_param('ids'); // Attendez un tableau d'IDs, exemple : [1, 2, 3]

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $batchTable = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);

        // Vérifier que le tableau d'IDs n'est pas vide
        if (empty($campaign_ids)) {
            return new \WP_Error(
                'no_ids_provided',
                __('No campaign IDs provided.', 'mailerpress'),
                ['status' => 400]
            );
        }

        $placeholders = implode(',', array_fill(0, \count($campaign_ids), '%d'));

        $query = $wpdb->prepare(
            "DELETE FROM {$table_name} WHERE campaign_id IN ({$placeholders})",
            ...$campaign_ids
        );

        $deleted = $wpdb->query($query);

        if (false === $deleted) {
            return new \WP_Error(
                'db_delete_error',
                __('Failed to delete the campaigns.', 'mailerpress'),
                ['status' => 500]
            );
        }

        $batch_deleted = $wpdb->query(
            $wpdb->prepare(
                "DELETE FROM {$batchTable} WHERE campaign_id IN ({$placeholders})",
                ...$campaign_ids
            )
        );

        if (false === $batch_deleted) {
            return new \WP_Error(
                'db_delete_error',
                __('Failed to delete the batches.', 'mailerpress'),
                ['status' => 500]
            );
        }

        return new \WP_REST_Response(
            [
                'message' => __('Campaigns successfully deleted.', 'mailerpress'),
                'ids' => $campaign_ids,
            ],
            200
        );
    }

    #[Endpoint(
        'campaign/all',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canDeleteCampaigns'],
    )]
    public function deleteAll(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {

        if (!current_user_can(Capabilities::DELETE_EMAIL_CAMPAIGNS)) {
            return new \WP_Error(
                'forbidden',
                __('You do not have permission to delete campaigns.', 'mailerpress'),
                ['status' => 403]
            );
        }

        global $wpdb;
        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $tableBatch = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);

        $campaign_types = $request->get_param('campaign_type'); // array of ['id'=>..., 'name'=>...]

        if (empty($campaign_types)) {
            $campaign_types = [
                ['id' => 'newsletter', 'name' => 'Newsletter'],
                ['id' => 'automated', 'name' => 'Automated'],
            ];
        } elseif (!is_array($campaign_types)) {
            return new \WP_REST_Response([
                'message' => 'campaign_type parameter must be an array.'
            ], 400);
        }

        $campaign_type_ids = array_map(fn($ct) => $ct['id'], $campaign_types);

        if (empty($campaign_type_ids)) {
            return new \WP_REST_Response(['message' => 'No campaign_type IDs found.'], 400);
        }

        $placeholders = implode(',', array_fill(0, count($campaign_type_ids), '%s'));

        // Delete batches linked to campaigns of these types AND with trash status
        $delete_batches_query = "
        DELETE FROM {$tableBatch}
        WHERE campaign_id IN (
            SELECT id FROM {$table_name} 
            WHERE campaign_type IN ($placeholders) AND status = 'trash'
        )
    ";
        $delete_batches_query_prepared = $wpdb->prepare($delete_batches_query, ...$campaign_type_ids);
        $deleted_batches = $wpdb->query($delete_batches_query_prepared);

        // Delete campaigns of these types AND with trash status
        $delete_campaigns_query = "
        DELETE FROM {$table_name} 
        WHERE campaign_type IN ($placeholders) AND status = 'trash'
    ";
        $delete_campaigns_query_prepared = $wpdb->prepare($delete_campaigns_query, ...$campaign_type_ids);
        $deleted_campaigns = $wpdb->query($delete_campaigns_query_prepared);

        if ($deleted_batches === false || $deleted_campaigns === false) {
            return new \WP_REST_Response(['message' => 'Failed to delete campaigns or batches.'], 500);
        }

        return new \WP_REST_Response(
            [
                'message' => "Deleted campaigns and related batches for campaign_type IDs (trash only): " . implode(
                    ', ',
                    $campaign_type_ids
                ),
                'deleted_campaigns' => $deleted_campaigns,
                'deleted_batches' => $deleted_batches,
            ],
            200
        );
    }

    #[Endpoint(
        'campaign/status',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function update_status(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $ids = $request->get_param('id');
        $status = sanitize_text_field($request->get_param('status'));
        $campaign_type = sanitize_text_field($request->get_param('campaign_type'));

        // Validate status
        $allowed_statuses = ['draft', 'scheduled', 'sending', 'sent', 'paused', 'cancelled', 'trash'];
        if (empty($status) || !in_array($status, $allowed_statuses, true)) {
            return new \WP_Error(
                'invalid_status',
                __('Invalid or empty campaign status.', 'mailerpress'),
                ['status' => 400]
            );
        }

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Prepare extra SQL for trash
        $extra_set = $status === 'trash' ? ', batch_id = NULL' : '';

        // Handle "all" case
        if ($ids === 'all') {
            $where = '';
            $params = [$status, current_time('mysql')];

            if (!empty($campaign_type)) {
                $where = 'WHERE type = %s';
                $params[] = $campaign_type;
            }

            $updated = $wpdb->query(
                $wpdb->prepare(
                    "UPDATE {$table_name} SET status = %s, updated_at = %s{$extra_set} {$where}",
                    $params
                )
            );

            if (false === $updated) {
                return new \WP_Error(
                    'db_update_error',
                    __('Failed to update campaign statuses.', 'mailerpress'),
                    ['status' => 500]
                );
            }

            return new \WP_REST_Response(
                [
                    'success' => true,
                    'message' => sprintf(
                        /* translators: %d number of campaigns */
                        __('%d campaign(s) status updated successfully.', 'mailerpress'),
                        $updated
                    ),
                    'updated_ids' => 'all',
                    'new_status' => $status,
                    'campaign_type' => $campaign_type,
                ],
                200
            );
        }

        // Otherwise normalize IDs: ensure array
        if (!is_array($ids)) {
            $ids = [$ids];
        }
        $ids = array_map('intval', $ids);
        $ids = array_filter($ids);

        if (empty($ids)) {
            return new \WP_Error('missing_id', __('No campaign ID(s) provided.', 'mailerpress'), ['status' => 400]);
        }

        // Build placeholders for IN clause
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        // Check existence
        $existing_ids = $wpdb->get_col(
            $wpdb->prepare(
                "SELECT campaign_id FROM {$table_name} WHERE campaign_id IN ($placeholders)",
                $ids
            )
        );

        if (empty($existing_ids)) {
            return new \WP_Error('not_found', __('No matching campaign(s) found.', 'mailerpress'), ['status' => 404]);
        }

        // Update all in one query
        $updated = $wpdb->query(
            $wpdb->prepare(
                "UPDATE {$table_name} SET status = %s, updated_at = %s{$extra_set} WHERE campaign_id IN ($placeholders)",
                array_merge([$status, current_time('mysql')], $ids)
            )
        );

        if (false === $updated) {
            return new \WP_Error(
                'db_update_error',
                __('Failed to update campaign status.', 'mailerpress'),
                ['status' => 500]
            );
        }

        return new \WP_REST_Response(
            [
                'success' => true,
                'message' => sprintf(
                    /* translators: %d number of campaigns */
                    __('%d campaign(s) status updated successfully.', 'mailerpress'),
                    $updated
                ),
                'updated_ids' => $existing_ids,
                'new_status' => $status,
            ],
            200
        );
    }

    #[Endpoint(
        'campaign/delete',
        methods: 'DELETE',
        permissionCallback: [Permissions::class, 'canDeleteCampaigns'],
    )]
    public function delete_campaign(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        if (!current_user_can(Capabilities::DELETE_EMAIL_CAMPAIGNS)) {
            return new \WP_Error(
                'forbidden',
                __('You do not have permission to delete campaigns.', 'mailerpress'),
                ['status' => 403]
            );
        }

        global $wpdb;

        $ids = $request->get_param('id'); // "all" or array/single ID
        $campaign_type = sanitize_text_field($request->get_param('campaign_type'));
        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // --- HANDLE "ALL" ---
        if ($ids === 'all') {
            $sql = "DELETE FROM {$table_name} WHERE status = %s";
            $params = ['trash'];

            if (!empty($campaign_type)) {
                $sql .= " AND type = %s";
                $params[] = $campaign_type;
            }

            // Spread the array safely
            $deleted = $wpdb->query($wpdb->prepare($sql, ...$params));

            if (false === $deleted) {
                return new \WP_Error(
                    'db_delete_error',
                    __('Failed to delete campaigns.', 'mailerpress'),
                    ['status' => 500]
                );
            }

            return new \WP_REST_Response([
                'success' => true,
                'message' => sprintf(__('All (%d) campaign(s) permanently deleted.', 'mailerpress'), $deleted),
                'deleted_ids' => 'all',
            ], 200);
        }

        // --- HANDLE SPECIFIC IDS ---
        if (!is_array($ids)) {
            $ids = [$ids];
        }

        $ids = array_map('intval', $ids);
        $ids = array_filter($ids);

        if (empty($ids)) {
            return new \WP_Error('missing_id', __('No campaign ID(s) provided.', 'mailerpress'), ['status' => 400]);
        }

        // Build placeholders for IN clause
        $placeholders = implode(',', array_fill(0, count($ids), '%d'));

        // Only select campaigns that are in trash
        $sql = "SELECT campaign_id FROM {$table_name} WHERE campaign_id IN ($placeholders) AND status = %s";
        $prepare_params = array_merge($ids, ['trash']); // merge before unpacking
        $existing_ids = $wpdb->get_col($wpdb->prepare($sql, ...$prepare_params));

        if (empty($existing_ids)) {
            return new \WP_Error('not_found', __('No campaign(s) in trash found.', 'mailerpress'), ['status' => 404]);
        }

        // Delete the selected campaigns
        $placeholders_existing = implode(',', array_fill(0, count($existing_ids), '%d'));
        $sql_delete = "DELETE FROM {$table_name} WHERE campaign_id IN ($placeholders_existing)";
        $deleted = $wpdb->query($wpdb->prepare($sql_delete, ...$existing_ids));

        if (false === $deleted) {
            return new \WP_Error(
                'db_delete_error',
                __('Failed to delete campaign(s).', 'mailerpress'),
                ['status' => 500]
            );
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => sprintf(__('Campaign(s) permanently deleted: %d', 'mailerpress'), $deleted),
            'deleted_ids' => $existing_ids,
        ], 200);
    }


    #[Endpoint(
        'campaign/(?P<id>\d+)/rename',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canEdit'],
    )]
    public function rename(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');
        $title = sanitize_text_field($request->get_param('title'));

        if (empty($title)) {
            return new \WP_Error('invalid_title', __('Title cannot be empty.', 'mailerpress'), ['status' => 400]);
        }

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $campaign = $wpdb->get_row($wpdb->prepare(
            "SELECT campaign_id, status FROM {$table_name} WHERE campaign_id = %d",
            $campaign_id
        ));

        if (!$campaign) {
            return new \WP_Error('not_found', __('Campaign not found.', 'mailerpress'), ['status' => 404]);
        }

        // Ne pas mettre à jour updated_at si la campagne est programmée (scheduled)
        // pour éviter de déclencher l'envoi prématurément
        $data_to_update = ['name' => $title];
        if ($campaign->status !== 'scheduled') {
            $data_to_update['updated_at'] = current_time('mysql');
        }

        $updated = $wpdb->update(
            $table_name,
            $data_to_update,
            [
                'campaign_id' => $campaign_id,
            ]
        );

        if (false === $updated) {
            return new \WP_Error('db_update_error', __('Failed to rename campaign.', 'mailerpress'), ['status' => 500]);
        }

        return new \WP_REST_Response(
            [
                'success' => true,
                'message' => __('Campaign renamed successfully.', 'mailerpress'),
                'campaign_id' => $campaign_id,
                'new_title' => $title,
            ],
            200
        );
    }


    #[Endpoint(
        'campaign/(?P<id>\d+)',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canEdit'],
        args: [
            'id' => [
                'required' => true,
                'validate_callback' => [ArgsValidator::class, 'validateId'],
            ],
        ]
    )]
    public function edit(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');
        $name = esc_attr($request->get_param('title'));
        $meta = $request->get_param('meta');

        // Vérifiez si la campagne existe
        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $campaign = $wpdb->get_row($wpdb->prepare("SELECT * FROM {$table_name} WHERE campaign_id = %d", $campaign_id));

        if (!$campaign) {
            return new \WP_Error('not_found', __('Campaign not found.', 'mailerpress'), ['status' => 404]);
        }

        $current_user_id = get_current_user_id();


        // Préparer les données pour la mise à jour
        $data = [
            'name' => $name ?: $campaign->name, // Si "title" est vide, garder l'ancien
            'subject' => !empty($meta['emailConfig']['campaignSubject']) ? $meta['emailConfig']['campaignSubject'] : $campaign->subject,
            'status' => !empty($meta['status']) ? esc_attr($meta['status']) : $campaign->status,
            'email_type' => !empty($meta['emailConfig']['email_type']) ? esc_attr($meta['emailConfig']['email_type']) : $campaign->email_type,
            'content_html' => !empty($meta['json']) ? wp_json_encode($meta['json']) : wp_json_encode($campaign->content_html),
            'config' => !empty($meta['emailConfig']) ? wp_json_encode($meta['emailConfig']) : $campaign->config,
            'updated_at' => current_time('mysql'),
        ];

        if (empty($campaign->editing_user_id) || (int)$campaign->editing_user_id === $current_user_id) {
            $data['editing_user_id'] = $current_user_id;
            $data['editing_started_at'] = current_time('mysql');
        }

        // Mettre à jour les données dans la base de données
        $updated = $wpdb->update($table_name, $data, ['campaign_id' => $campaign_id]);

        if (false === $updated) {
            return new \WP_Error('db_update_error', __('Failed to update campaign.', 'mailerpress'), ['status' => 500]);
        }

        // Retourner une réponse de succès
        return new \WP_REST_Response(
            [
                'success' => true,
                'message' => __('Campaign updated successfully.', 'mailerpress'),
                'campaign_id' => $campaign_id,
                'updated_data' => $data,
            ],
            200
        );
    }

    #[Endpoint(
        'campaign/save-content/(?P<id>\d+)',
        methods: 'PUT',
        permissionCallback: [Permissions::class, 'canEdit'],
        args: [
            'id' => [
                'required' => true,
                'validate_callback' => [ArgsValidator::class, 'validateId'],
            ],
        ]
    )]
    public function saveCampaignContent(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');
        $current_user = get_current_user_id();
        $content = $request->get_param('content');

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $campaign = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$table_name} WHERE campaign_id = %d",
            $campaign_id
        ));

        if (!$campaign) {
            return new \WP_Error('not_found', __('Campaign not found.', 'mailerpress'), ['status' => 404]);
        }

        // ✅ Check if campaign is locked by someone else
        if ($campaign->editing_user_id && (int)$campaign->editing_user_id !== (int)$current_user) {
            return new \WP_Error(
                'locked',
                __('This campaign is currently locked by another user.', 'mailerpress'),
                ['status' => 423] // 423 Locked
            );
        }

        // Prepare data for update
        $data = [
            'content_html' => wp_json_encode($content),
        ];

        $updated = $wpdb->update($table_name, $data, ['campaign_id' => $campaign_id]);

        if (false === $updated) {
            return new \WP_Error('db_update_error', __('Failed to update campaign.', 'mailerpress'), ['status' => 500]);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => __('Campaign updated successfully.', 'mailerpress'),
            'campaign_id' => $campaign_id,
            'updated_data' => $data,
        ], 200);
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/html',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canView'],
    )]
    public function formatHTML(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $html = $request->get_param('html');

        return new \WP_REST_Response(
            Kernel::getContainer()->get(HtmlParser::class)->init(
                $html,
                [
                    'UNSUB_LINK' => home_url('/unsubsribe'),
                ]
            )->replaceVariables(),
            200
        );
    }

    #[Endpoint(
        'campaign/contact/preview/',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canView'],
    )]
    public function previewEmailByContact(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $contactId = esc_html($request->get_param('contact'));
        $html = $request->get_param('html');

        if (!empty($contactId && !empty($html))) {
            $contactEntity = Kernel::getContainer()->get(Contacts::class)->get((int)$contactId);

            // Générer l'HTML personnalisé pour ce contact
            $parsed_html = Kernel::getContainer()->get(HtmlParser::class)->init(
                $html,
                [
                    'UNSUB_LINK' => wp_unslash(
                        \sprintf(
                            '%s&data=%s&cid=%s&batchId=%s',
                            mailerpress_get_page('unsub_page'),
                            esc_attr($contactEntity->unsubscribe_token),
                            esc_attr($contactEntity->access_token),
                            ''
                        )
                    ),
                    'MANAGE_SUB_LINK' => wp_unslash(
                        \sprintf(
                            '%s&cid=%s',
                            mailerpress_get_page('manage_page'),
                            esc_attr($contactEntity->access_token)
                        )
                    ),
                    'CONTACT_NAME' => esc_html($contactEntity->first_name) . ' ' . esc_html($contactEntity->last_name),
                    'TRACK_OPEN' => get_rest_url(
                        null,
                        \sprintf(
                            'mailerpress/v1/campaign/track-open?token=%s',
                            $this->generateTrackingTokenForPreview($contactEntity->access_token, 0)
                        )
                    ),
                    'contact_name' => \sprintf(
                        '%s %s',
                        esc_html($contactEntity->first_name),
                        esc_html($contactEntity->last_name)
                    ),
                    'contact_email' => \sprintf('%s', esc_html($contactEntity->email)),
                    'contact_first_name' => \sprintf('%s', esc_html($contactEntity->first_name)),
                    'contact_last_name' => \sprintf('%s', esc_html($contactEntity->last_name)),
                ]
            )->replaceVariables();

            return new \WP_REST_Response($parsed_html);
        }

        return new \WP_REST_Response(
            'error',
            400
        );
    }

    #[Endpoint(
        'campaign/create_batch',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canPublishCampaign'],
    )]
    public function createBatch(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $contacts = $request->get_param('contacts');
        $sendType = $request->get_param('sendType');
        $post = $request->get_param('post');
        $html = $request->get_param('htmlContent');
        $config = $request->get_param('config');
        $scheduledAt = $request->get_param('scheduledAt');

        $status = ('future' === $sendType) ? 'scheduled' : 'pending';

        $wpdb->insert(
            Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
            [
                'status' => $status,
                'total_emails' => count($contacts),
                'sender_name' => $config['fromName'],
                'sender_to' => $config['fromTo'],
                'subject' => $config['subject'],
                'scheduled_at' => $scheduledAt,
                'campaign_id' => $post,
            ]
        );

        $batch_id = $wpdb->insert_id;

        if (!$batch_id || is_wp_error($batch_id)) {
            return new \WP_REST_Response(null, 400);
        }

        // Get frequency sending option with default fallback
        $frequencySending = get_option('mailerpress_frequency_sending', [
            "webHost" => "",
            "frequency" => "recommended",
            "settings" => [
                "numberEmail" => 25,
                "config" => [
                    "value" => 5,
                    "unit" => "minutes",
                ],
            ],
        ]);

        if (is_string($frequencySending)) {
            $decoded = json_decode($frequencySending, true);
            if (is_array($decoded)) {
                $frequencySending = $decoded;
            } else {
                // fallback to default if decode fails
                $frequencySending = [
                    "webHost" => "",
                    "frequency" => "recommended",
                    "settings" => [
                        "numberEmail" => 25,
                        "config" => [
                            "value" => 5,
                            "unit" => "minutes",
                        ],
                    ],
                ];
            }
        }

        // Extract numberEmail and config properly from settings
        $numberEmail = $frequencySending['settings']['numberEmail'] ?? 25;
        $frequencyConfig = $frequencySending['settings']['config'] ?? ['value' => 5, 'unit' => 'minutes'];

        $contact_chunks = array_chunk($contacts, $numberEmail);

        $now = time();

        $unit_multipliers = [
            'seconds' => 1,
            'minutes' => MINUTE_IN_SECONDS,
            'hours' => HOUR_IN_SECONDS,
        ];

        $interval_seconds = ($frequencyConfig['value'] ?? 5) * ($unit_multipliers[$frequencyConfig['unit']] ?? MINUTE_IN_SECONDS);

        foreach ($contact_chunks as $chunk_index => $contact_chunk) {

            $hook_name = 'mailerpress_process_contact_chunk';

            // Generate a unique transient key for this chunk
            $transient_key = 'mailerpress_chunk_' . $batch_id . '_' . $chunk_index;

            $datetime = new DateTime($scheduledAt, wp_timezone());
            $scheduledAt = $datetime->format('Y-m-d H:i:s');

            set_transient($transient_key, [
                'html' => $html,
                'subject' => $config['subject'],
                'contacts' => $contact_chunk,
                'scheduled_at' => $datetime,
                'webhook_url' => get_rest_url(null, 'mailerpress/v1/webhook/notify'),
                'sendType' => $sendType,
            ], 12 * HOUR_IN_SECONDS);

            $scheduled_time = $now + ($chunk_index * $interval_seconds);

            as_schedule_single_action(
                $scheduled_time,
                $hook_name,
                [$batch_id, $transient_key],
                'mailerpress'
            );
        }

        do_action('mailerpress_batch_event', $status, $post, $batch_id);

        return new \WP_REST_Response($batch_id, 200);
    }

    #[Endpoint(
        'campaign/create_batch_V2',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canPublishCampaign'],
    )]
    public function createBatchV2(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        if (!current_user_can(Capabilities::PUBLISH_CAMPAIGNS)) {
            return new WP_Error(
                'mailerpress_no_permission',
                __('You do not have permission to create a campaign batch.', 'mailerpress'),
                ['status' => 403]
            );
        }

        $sendType = $request->get_param('sendType');
        $post = $request->get_param('postEdit');
        $html = $request->get_param('html');
        $config = $request->get_param('config');
        $scheduledAt = $request->get_param('scheduledAt');
        $recipientTargeting = $request->get_param('recipientTargeting') ?? null;
        $lists = $request->get_param('lists') ?? [];
        $tags = $request->get_param('tags') ?? [];
        $segment = $request->get_param('segment') ?? [];

        update_option('mailerpress_batch_' . $post . '_html', $html, false);

        // Get subject from config or fallback to campaign title
        $subject = $config['subject'] ?? '';
        if (empty($subject) && !empty($post)) {
            $campaign = get_post($post);
            $subject = $campaign ? $campaign->post_title : '';
        }

        // Calculate total number of contacts before creating the batch
        $total_emails = 0;
        try {
            $fetcher = $this->getContactFetcher($recipientTargeting, $lists, $tags, $segment);
            if ($fetcher) {
                // Fetch contacts in chunks to count them
                $chunk_size = 1000;
                $offset = 0;
                do {
                    $contacts = $fetcher->fetch($chunk_size, $offset);
                    $total_emails += count($contacts);
                    $offset += $chunk_size;
                } while (count($contacts) === $chunk_size);
            }
        } catch (\Exception $e) {
            // If counting fails, use 0 (will be updated later in MailerPressEmailBatch)
            error_log("[MailerPress] Failed to count contacts: " . $e->getMessage());
        }

        // Create batch immediately so it can be displayed in the UI
        $status = ('future' === $sendType) ? 'scheduled' : 'pending';
        $wpdb->insert(
            Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
            [
                'status' => $status,
                'total_emails' => $total_emails,
                'sender_name' => $config['fromName'] ?? '',
                'sender_to' => $config['fromTo'] ?? '',
                'subject' => $subject,
                'scheduled_at' => $scheduledAt,
                'campaign_id' => $post,
            ]
        );

        $batch_id = $wpdb->insert_id;
        if (!$batch_id) {
            return new \WP_Error(
                'mailerpress_batch_creation_failed',
                __('Failed to create batch', 'mailerpress'),
                ['status' => 500]
            );
        }

        // Calculate the scheduled time for the action
        $scheduled_time = time() + 5; // Default: 5 seconds from now for immediate sending
        if ('future' === $sendType && !empty($scheduledAt)) {
            // Convert scheduledAt to timestamp
            $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone(wp_timezone_string());
            try {
                $dt = new \DateTime($scheduledAt, $tz);
                $scheduled_timestamp = $dt->getTimestamp();
                // Only use scheduled time if it's in the future
                if ($scheduled_timestamp > time()) {
                    $scheduled_time = $scheduled_timestamp;
                }
            } catch (\Exception $e) {
                // If parsing fails, fallback to default (time() + 5)
                error_log("[MailerPress] Failed to parse scheduledAt: " . $e->getMessage());
            }
        }

        as_schedule_single_action(
            $scheduled_time,
            'mailerpress_batch_email',
            [
                $sendType,
                $post,
                $config,
                $scheduledAt,
                $recipientTargeting,
                $lists,
                $tags,
                $segment,
            ],
            'mailerpress'
        );

        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Définir le statut correct selon le type d'envoi
        // Si sendType est 'future', la campagne est programmée, sinon elle est en attente
        $campaign_status = ('future' === $sendType) ? 'scheduled' : 'pending';

        $wpdb->update(
            $table_name,
            [
                'status' => $campaign_status,
                'batch_id' => $batch_id,
                'updated_at' => current_time('mysql'), // Set to the current timestamp
            ],
            ['campaign_id' => intval($post)], // Where condition
            ['%s', '%d', '%s'], // Data format: string for status, integer for batch_id, string for timestamp
            ['%d']        // Where condition format: integer for campaign_id
        );

        $wpdb->update(
            $table_name,
            [
                'editing_user_id' => null,
                'editing_started_at' => null
            ],
            ['campaign_id' => $post,]
        );

        // Remove all pending unlock requests
        delete_transient("campaign_{$post}_unlock_requests");


        return new \WP_REST_Response(['pending'], 200);
    }


    #[Endpoint(
        'campaign/update_automated_campaign',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function updateAutomatedCampaign(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaignId = (int)$request->get_param('campaignId');
        $html = $request->get_param('html');
        $data = $request->get_param('data');

        // Validate inputs
        if (!$campaignId || empty($html)) {
            return new \WP_Error(
                'invalid_parameters',
                'Missing or invalid campaignId or html',
                ['status' => 400]
            );
        }

        $table = $wpdb->prefix . 'mailerpress_campaigns';

        // Check if campaign exists
        $exists = $wpdb->get_var(
            $wpdb->prepare("SELECT COUNT(*) FROM $table WHERE campaign_id = %d", $campaignId)
        );

        if (!$exists) {
            return new \WP_Error(
                'campaign_not_found',
                'Campaign not found',
                ['status' => 404]
            );
        }

        // Update the content_html in the database
        $updated = $wpdb->update(
            $table,
            ['content_html' => json_encode($data)],
            ['campaign_id' => $campaignId],
            ['%s'],
            ['%d']
        );

        if ($updated === false) {
            return new \WP_Error(
                'update_failed',
                'Failed to update campaign HTML content',
                ['status' => 500]
            );
        }

        // Update the HTML version in the WordPress options
        $optionKey = 'mailerpress_batch_' . $campaignId . '_html';
        if (get_option($optionKey)) {
            update_option($optionKey, $html);
        }

        return new \WP_REST_Response([
            'success' => true,
            'message' => 'Campaign HTML content and option updated successfully',
        ]);
    }


    /**
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/create_automated_campaign',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function createAutomatedCampaign(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $sendType = $request->get_param('sendType');
        $post = intval($request->get_param('postEdit'));
        $html = $request->get_param('html');
        $config = $request->get_param('config');
        $scheduledAt = $request->get_param('scheduledAt');
        $recipientTargeting = $request->get_param('recipientTargeting') ?? null;
        $lists = $request->get_param('lists') ?? [];
        $tags = $request->get_param('tags') ?? [];
        $segment = $request->get_param('segment') ?? [];
        $automateSettings = $request->get_param('automateSettings') ?? null;

        // Store HTML separately
        update_option('mailerpress_batch_' . $post . '_html', $html, false);

        // Get existing config from DB
        $table_name = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $existing = $wpdb->get_row($wpdb->prepare("SELECT config FROM {$table_name} WHERE campaign_id = %d", $post));

        $currentConfig = [];
        if ($existing && $existing->config) {
            $currentConfig = json_decode($existing->config, true) ?? [];
        }

        // Merge automateSettings into config
        if ($automateSettings) {
            $currentConfig['automateSettings'] = $automateSettings;
        }

        // Update the campaign
        $wpdb->update(
            $table_name,
            [
                'status' => 'active',
                'campaign_type' => 'automated',
                'updated_at' => current_time('mysql'),
                'config' => wp_json_encode($currentConfig),
            ],
            ['campaign_id' => $post],
            ['%s', '%s', '%s', '%s'],
            ['%d']
        );

        // Schedule the first run of the automation
        mailerpress_schedule_automated_campaign(
            $post,
            $sendType,
            $config,
            $scheduledAt,
            $recipientTargeting,
            $lists,
            $tags,
            $segment,
        );

        return new \WP_REST_Response(['pending'], 200);
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/send_test',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canPublishCampaign'],
    )]
    public function sendTest(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $contacts = $request->get_param('contacts');
        $body = $request->get_param('htmlContent');
        $subject = esc_attr($request->get_param('subject'));

        $mailer = Kernel::getContainer()->get(EmailServiceManager::class)->getActiveService();
        $config = $mailer->getConfig();

        if (
            empty($config['conf']['default_email'])
            || empty($config['conf']['default_name'])
        ) {
            $globalSender = get_option('mailerpress_default_settings');

            if ($globalSender) {
                if (is_string($globalSender)) {
                    $globalSender = json_decode($globalSender, true);
                }

                if (is_array($globalSender)) {
                    $config['conf']['default_email'] = $globalSender['fromAddress'] ?? '';
                    $config['conf']['default_name'] = $globalSender['fromName'] ?? '';
                }
            }
        }

        // Parser le HTML pour supprimer les spans d'emoji et traiter les variables
        $container = Kernel::getContainer();
        /** @var HtmlParser $parser */
        $parser = $container->get(HtmlParser::class);
        $parsedBody = $parser->init($body, [])->replaceVariables();

        // Récupérer les paramètres Reply to depuis les paramètres par défaut
        $defaultSettings = get_option('mailerpress_default_settings', []);
        if (is_string($defaultSettings)) {
            $defaultSettings = json_decode($defaultSettings, true) ?: [];
        }

        // Déterminer les valeurs Reply to (utiliser From si Reply to est vide)
        $replyToName = !empty($defaultSettings['replyToName'])
            ? $defaultSettings['replyToName']
            : ($config['conf']['default_name'] ?? '');
        $replyToAddress = !empty($defaultSettings['replyToAddress'])
            ? $defaultSettings['replyToAddress']
            : ($config['conf']['default_email'] ?? '');

        $success = [];
        $errors = [];

        foreach ($contacts as $contact) {
            try {
                $mailer->sendEmail([
                    'to' => $contact,
                    'html' => true,
                    'body' => $parsedBody,
                    'subject' => /* translators: %s is the subject of the email */ sprintf(__(
                        '[MailerPress TEST] - %s',
                        'mailerpress'
                    ), $subject),
                    'sender_name' => $config['conf']['default_name'],
                    'sender_to' => $config['conf']['default_email'],
                    'reply_to_name' => $replyToName,
                    'reply_to_address' => $replyToAddress,
                    'apiKey' => $config['conf']['api_key'] ?? '',
                ]);
                $success[] = $contact;
            } catch (\Exception $e) {
                $errors[] = [
                    'contact' => $contact,
                    'message' => $e->getMessage()
                ];
            }
        }

        return new \WP_REST_Response([
            'status' => empty($errors) ? 'success' : 'partial',
            'sent' => $success,
            'errors' => $errors,
        ], empty($errors) ? 200 : 207); // 207: Multi-Status (partial success)
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/pause_batch',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function mailerpress_cancel_batch_actions(\WP_REST_Request $request)
    {
        global $wpdb;

        $batch_id = (int)$request->get_param('batchId');
        $campaign_id = (int)$request->get_param('campaignId');

        if (!$batch_id) {
            return new \WP_REST_Response(['error' => 'Missing batchId'], 400);
        }

        // Get all actions for this batch (you can pass a reduced status list if preferred)
        $asActions = $this->mailerpress_get_chunk_actions_for_batch($batch_id);

        $store = \ActionScheduler_Store::instance();

        foreach ($asActions as $action_id => $action) {
            $args = $action->get_args();

            // Arg[1] is our transient key: 'mailerpress_chunk_{batch_id}_{chunk_index}'
            if (isset($args[1])) {
                delete_transient($args[1]);
            }

            // Cancel first (safe; marks it as canceled and prevents execution)
            try {
                $store->cancel_action($action_id);
            } catch (\Exception $e) {
                error_log("MailerPress: Failed to cancel AS action {$action_id}: " . $e->getMessage());
            }

            // Hard delete (optional). Comment out if you want history.
            try {
                $store->delete_action($action_id);
            } catch (\Exception $e) {
                error_log("MailerPress: Failed to delete AS action {$action_id}: " . $e->getMessage());
            }
        }

        // Set campaign as draft and remove the batch_id
        if ($campaign_id) {
            $wpdb->update(
                Tables::get(Tables::MAILERPRESS_CAMPAIGNS),
                [
                    'status' => 'draft',
                    'batch_id' => null
                ],
                ['campaign_id' => $campaign_id],
                ['%s', 'NULL'],
                ['%d']
            );
        }

        // Delete the batch record
        $wpdb->delete(
            Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
            ['id' => $batch_id],
            ['%d']
        );


        return new \WP_REST_Response([
            'batchId' => $batch_id,
            'campaignId' => $campaign_id,
            'canceled' => array_keys($asActions),
            'status' => 'draft',
        ], 200);
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/(?P<id>\d+)/deactivate',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function mailerpress_deactivate_automated_campaign(\WP_REST_Request $request)
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');

        if (!$campaign_id) {
            return new \WP_REST_Response(['error' => 'Missing campaign ID'], 400);
        }

        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        $campaign = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT campaign_id, campaign_type, status FROM $table WHERE campaign_id = %d",
                $campaign_id
            ),
            ARRAY_A
        );

        if (!$campaign) {
            return new \WP_REST_Response(['error' => 'Campaign not found'], 404);
        }

        if ($campaign['campaign_type'] !== 'automated') {
            return new \WP_REST_Response(['error' => 'Only automated campaigns can be deactivated'], 400);
        }

        $wpdb->update(
            $table,
            ['status' => 'inactive'],
            ['campaign_id' => $campaign_id],
            ['%s'],
            ['%d']
        );

        return new \WP_REST_Response([
            'campaignId' => $campaign_id,
            'status' => 'inactive',
            'message' => 'Campaign deactivated successfully',
        ], 200);
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/(?P<id>\d+)/activate',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function mailerpress_activate_automated_campaign(\WP_REST_Request $request)
    {
        global $wpdb;

        $campaign_id = (int)$request->get_param('id');

        if (!$campaign_id) {
            return new \WP_REST_Response(['error' => 'Missing campaign ID'], 400);
        }

        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        $campaign = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT campaign_id, campaign_type, status FROM $table WHERE campaign_id = %d",
                $campaign_id
            ),
            ARRAY_A
        );

        if (!$campaign) {
            return new \WP_REST_Response(['error' => 'Campaign not found'], 404);
        }

        if ($campaign['campaign_type'] !== 'automated') {
            return new \WP_REST_Response(['error' => 'Only automated campaigns can be activated'], 400);
        }

        // Optional: Only allow activation if not already active
        if ($campaign['status'] === 'scheduled') {
            return new \WP_REST_Response(['message' => 'Campaign is already active'], 200);
        }

        $wpdb->update(
            $table,
            ['status' => 'active'],
            ['campaign_id' => $campaign_id],
            ['%s'],
            ['%d']
        );

        return new \WP_REST_Response([
            'campaignId' => $campaign_id,
            'status' => 'active',
            'message' => 'Campaign activated successfully',
        ], 200);
    }


    /**
     * Return all AS actions for the given MailerPress batch (chunk sends).
     *
     * @param int $batch_id
     * @param array|null $statuses Optional list of AS statuses to include.
     * @return array [ action_id => ActionScheduler_Action ]
     */
    private function mailerpress_get_chunk_actions_for_batch($batch_id, $statuses = null)
    {
        if (null === $statuses) {
            $statuses = [
                ActionScheduler_Store::STATUS_PENDING,
                ActionScheduler_Store::STATUS_COMPLETE,
                ActionScheduler_Store::STATUS_RUNNING,   // include in-progress
                ActionScheduler_Store::STATUS_FAILED,    // include failures
                ActionScheduler_Store::STATUS_CANCELED,  // include canceled
            ];
        }

        $store = ActionScheduler_Store::instance();
        $found = [];
        $limit = 100; // page size; tune as needed

        foreach ($statuses as $status) {
            $offset = 0;

            do {
                $ids = $store->query_actions([
                    'hook' => 'mailerpress_process_contact_chunk',
                    'group' => 'mailerpress',
                    'status' => $status,
                    'per_page' => $limit,
                    'offset' => $offset,
                ]);

                if (empty($ids)) {
                    break;
                }

                foreach ($ids as $id) {
                    $action = $store->fetch_action($id);
                    if (!$action) {
                        continue;
                    }

                    $args = $action->get_args();
                    // Our scheduled actions use [ $batch_id, $transient_key ]
                    if (isset($args[0]) && (int)$args[0] === (int)$batch_id) {
                        $found[$id] = $action;
                    }
                }

                $offset += $limit;
            } while (count($ids) === $limit);
        }

        return $found;
    }


    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint(
        'campaign/resume_batch',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function resumeBatch(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $batch_id = $request->get_param('batchId');

        $wpdb->update(
            "{$wpdb->prefix}mailerpress_email_batches",
            ['status' => 'pending'],
            ['id' => $batch_id],
            ['%s'],    // Format de la valeur du champ 'status' (NULL est traité comme une chaîne vide)
            ['%d']     // Format de la condition (id)
        );

        return new \WP_REST_Response([], 200);
    }

    /**
     * Generate a secure tracking token from access_token and batch_id
     */
    private function generateTrackingTokenForPreview(string $accessToken, int $batchId): string
    {
        // Use HMAC to create a secure token that includes batch_id
        $secret = defined('AUTH_SALT') ? AUTH_SALT : 'mailerpress-tracking-secret';
        $data = $accessToken . '|' . $batchId;
        $token = hash_hmac('sha256', $data, $secret);

        // Encode the token and batch_id together (base64url safe)
        $payload = base64_encode($token . '|' . $batchId);
        return rtrim(strtr($payload, '+/', '-_'), '=');
    }

    /**
     * Decode a secure tracking token and return contact_id and batch_id
     */
    private function decodeTrackingToken(string $token): ?array
    {
        global $wpdb;

        // Decode base64url
        $payload = base64_decode(strtr($token, '-_', '+/') . str_repeat('=', 3 - (3 + strlen($token)) % 4));

        if (false === $payload) {
            return null;
        }

        // Extract token hash and batch_id
        $parts = explode('|', $payload, 2);
        if (count($parts) !== 2) {
            return null;
        }

        [$tokenHash, $batchId] = $parts;
        $batchId = (int) $batchId;

        // batch_id can be 0 for previews, so we only check if it's a valid integer
        if ($batchId < 0) {
            return null;
        }

        // Find contact by matching the token hash
        // We limit the search to contacts in this batch for better performance
        $secret = defined('AUTH_SALT') ? AUTH_SALT : 'mailerpress-tracking-secret';
        $contactTable = Tables::get(Tables::MAILERPRESS_CONTACT);

        // For batch_id 0 (preview), search all contacts
        // Otherwise, limit to contacts in the batch for better performance
        if ($batchId === 0) {
            $contacts = $wpdb->get_results(
                "SELECT contact_id, access_token FROM {$contactTable} WHERE access_token IS NOT NULL AND access_token != '' LIMIT 1000"
            );
        } else {
            $queueTable = Tables::get(Tables::MAILERPRESS_EMAIL_QUEUE);

            // Get contacts that are in this batch to limit the search
            $contacts = $wpdb->get_results(
                $wpdb->prepare(
                    "SELECT c.contact_id, c.access_token 
                     FROM {$contactTable} c
                     INNER JOIN {$queueTable} q ON c.contact_id = q.contact_id
                     WHERE q.batch_id = %d 
                     AND c.access_token IS NOT NULL 
                     AND c.access_token != ''",
                    $batchId
                )
            );

            // If no contacts found in queue, fallback to all contacts (for backward compatibility)
            if (empty($contacts)) {
                $contacts = $wpdb->get_results(
                    "SELECT contact_id, access_token FROM {$contactTable} WHERE access_token IS NOT NULL AND access_token != '' LIMIT 1000"
                );
            }
        }

        foreach ($contacts as $contact) {
            $data = $contact->access_token . '|' . $batchId;
            $expectedHash = hash_hmac('sha256', $data, $secret);

            if (hash_equals($expectedHash, $tokenHash)) {
                return [
                    'contact_id' => (int) $contact->contact_id,
                    'batch_id' => $batchId,
                ];
            }
        }

        return null;
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws \Exception
     */
    #[Endpoint('campaign/track-open', methods: 'GET')]
    public function trackOpen(\WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        // Support both old format (contactId/batchId) and new secure format (token)
        $token = $request->get_param('token');

        if (!empty($token)) {
            // Decode the secure token
            $decoded = $this->decodeTrackingToken($token);
            if (!$decoded) {
                return new \WP_Error('invalid_token', 'Invalid tracking token.', ['status' => 400]);
            }
            $batch_id = $decoded['batch_id'];
            $contact_id = $decoded['contact_id'];
        } else {
            // Fallback to old format for backward compatibility
            $contact_id = $request->get_param('contactId');
            $batch_id = $request->get_param('batchId');
        }

        // Validate input
        // batch_id can be 0 for previews, so we only check if contact_id is valid
        if (empty($contact_id) || $batch_id < 0) {
            return new \WP_Error('invalid_input', 'Contact ID and valid Batch ID are required.', ['status' => 400]);
        }

        // Only track opens for real batches (not previews with batch_id = 0)
        if ($batch_id > 0) {
            $table = Tables::get(Tables::MAILERPRESS_EMAIL_TRACKING);

            $existing = $wpdb->get_var(
                $wpdb->prepare(
                    "SELECT id FROM {$table} WHERE contact_id = %d AND batch_id = %d",
                    $contact_id,
                    $batch_id
                )
            );

            if (empty($existing)) {
                // Prepare the data
                $data = [
                    'batch_id' => $batch_id,
                    'contact_id' => $contact_id,
                    'opened_at' => current_time('mysql'), // Record the time the email was opened
                    'clicks' => 0,                      // Default: no clicks yet
                    'unsubscribed_at' => null,           // Default: not unsubscribed
                ];

                $format = ['%d', '%d', '%s', '%d', '%s'];

                $row_exists = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT id FROM {$table} WHERE batch_id = %d AND contact_id = %d",
                        $batch_id,
                        $contact_id
                    )
                );

                if ($row_exists) {
                    $wpdb->update(
                        $table,
                        [
                            'opened_at' => $data['opened_at'],
                            'clicks' => $data['clicks'],
                            'unsubscribed_at' => $data['unsubscribed_at'],
                        ],
                        [
                            'batch_id' => $batch_id,
                            'contact_id' => $contact_id,
                        ],
                        ['%s', '%d', '%s'], // Formats for the fields to be updated
                        ['%d', '%d'] // Formats for batch_id and contact_id
                    );
                } else {
                    // If the row doesn't exist, insert it
                    $wpdb->insert($table, $data, $format);
                }
            }
        }

        // Send a transparent 1x1 pixel image as the response
        header('Content-Type: image/png');

        // Base64 string for a 1x1 transparent PNG.
        $base64_image = 'iVBORw0KGgoAAAANSUhEUgAAAA...'; // Truncated for brevity

        $image_data = base64_decode($base64_image, true);

        if (false === $image_data) {
            // Handle error.
            esc_html_e('Invalid Base64 string.', 'mailerpress');

            exit;
        }

        echo '<img src="data:image/png;base64,' . esc_attr($base64_image) . '" alt="" />';

        exit;
    }

    #[Endpoint(
        'campaign/(?P<id>\d+)/lock',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function mailerpress_lock_campaign(WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $campaign_id = intval($request['id']);
        $user_id = get_current_user_id();
        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        global $wpdb;

        $lock = $wpdb->get_row($wpdb->prepare(
            "SELECT editing_user_id, editing_started_at FROM $table WHERE campaign_id = %d",
            $campaign_id
        ));

        $lock_timeout = strtotime('-5 minutes');

        if ($lock && $lock->editing_user_id && $lock->editing_user_id != $user_id) {
            if (strtotime($lock->editing_started_at) > $lock_timeout) {
                return new WP_REST_Response([
                    'success' => false,
                    'message' => 'Cette campagne est en cours d’édition par un autre utilisateur.'
                ], 423); // 423 Locked
            }
        }

        $wpdb->update(
            $table,
            [
                'editing_user_id' => $user_id,
                'editing_started_at' => current_time('mysql')
            ],
            ['campaign_id' => $campaign_id]
        );

        return new WP_REST_Response(['success' => true]);
    }


    #[Endpoint(
        'campaign/(?P<id>\d+)/unlock-requests',
        methods: 'GET',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function mailerpress_unlock_requests_campaign(
        WP_REST_Request $request
    ): \WP_Error|\WP_HTTP_Response|\WP_REST_Response {
        $campaign_id = intval($request['id']);
        $requests = get_transient("campaign_{$campaign_id}_unlock_requests") ?: [];
        return new WP_REST_Response(['requests' => $requests]);
    }

    #[Endpoint(
        'campaign/(?P<id>\d+)/add-unlock-request',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function mailerpress_add_unlock_request_campaign(
        WP_REST_Request $request
    ): \WP_Error|\WP_HTTP_Response|\WP_REST_Response {
        $campaign_id = intval($request['id']);
        $user_id = get_current_user_id();
        add_unlock_request($campaign_id, $user_id);
        return new WP_REST_Response(['success' => true]);
    }

    #[Endpoint(
        'campaign/(?P<id>\d+)/deny-unlock-request',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function mailerpress_deny_unlock_request_campaign(
        WP_REST_Request $request
    ): \WP_Error|\WP_HTTP_Response|\WP_REST_Response {
        $campaign_id = intval($request['id']);
        $user_id = intval($request['new_user_id']);
        remove_unlock_request($campaign_id, $user_id);
        return new WP_REST_Response(['success' => true]);
    }


    #[Endpoint(
        'campaign/(?P<id>\d+)/unlock',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function mailerpress_unlock_campaign(WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $campaign_id = intval($request['id']);
        $current_user_id = get_current_user_id();
        $new_user_id = $request->get_param('new_user_id'); // optional new locker
        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        global $wpdb;

        // Reset the current lock
        $wpdb->update(
            $table,
            [
                'editing_user_id' => null,
                'editing_started_at' => null
            ],
            ['campaign_id' => $campaign_id, 'editing_user_id' => $current_user_id]
        );

        // Remove all pending unlock requests
        delete_transient("campaign_{$campaign_id}_unlock_requests");

        // Optionally assign new locker
        if (!empty($new_user_id) && is_numeric($new_user_id)) {
            $wpdb->update(
                $table,
                [
                    'editing_user_id' => intval($new_user_id),
                    'editing_started_at' => current_time('mysql')
                ],
                ['campaign_id' => $campaign_id]
            );
        }

        return new WP_REST_Response(['success' => true]);
    }


    #[Endpoint(
        'campaign/(?P<id>\d+)/refresh-lock',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function mailerpress_refresh_lock(WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        global $wpdb;

        $campaign_id = intval($request['id']);
        $user_id = get_current_user_id();
        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);
        $wpdb->update(
            $table,
            ['editing_started_at' => current_time('mysql')],
            ['campaign_id' => $campaign_id, 'editing_user_id' => $user_id]
        );

        return new WP_REST_Response(['success' => true]);
    }

    #[Endpoint(
        'campaign/(?P<id>\d+)/status',
        permissionCallback: [Permissions::class, 'canManageCampaign']
    )]
    public function campaingStatusLock(WP_REST_Request $request): \WP_Error|\WP_HTTP_Response|\WP_REST_Response
    {
        $campaign_id = intval($request['id']);
        global $wpdb;
        $table = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        $lock = $wpdb->get_row($wpdb->prepare(
            "SELECT editing_user_id, editing_started_at FROM $table WHERE campaign_id = %d",
            $campaign_id
        ));

        if (!$lock || !$lock->editing_user_id) {
            return new WP_REST_Response(['locked' => false]);
        }

        $user = get_userdata($lock->editing_user_id);
        return new WP_REST_Response([
            'locked' => true,
            'user_id' => $lock->editing_user_id,
            'user_name' => $user ? $user->display_name : '',
            'locked_avatar' => get_avatar_url($lock->editing_user_id, ['size' => 256, 'default' => 'mystery']),
            'timestamp' => $lock->editing_started_at
        ]);
    }

    #[Endpoint(
        'video-preview',
        methods: 'POST',
        permissionCallback: [Permissions::class, 'canManageCampaign'],
    )]
    public function generateVideoPreview(\WP_REST_Request $request): \WP_REST_Response
    {
        $videoUrl = esc_url_raw($request->get_param('url'));
        if (empty($videoUrl)) {
            return new \WP_REST_Response(['error' => 'Missing video url'], 400);
        }

        $parsed = $this->parseVideoUrl($videoUrl);
        if (!$parsed || empty($parsed['thumbnail'])) {
            return new \WP_REST_Response(['error' => 'Unsupported video url'], 400);
        }

        // 🔹 If Dailymotion, fetch high-res thumbnail
        if ($parsed['type'] === 'dailymotion') {
            $videoId = $parsed['id'];
            $parsed['thumbnail'] = "https://www.dailymotion.com/thumbnail/video/$videoId?size=1280";
            $oEmbedUrl = "https://www.dailymotion.com/services/oembed?url=https://www.dailymotion.com/video/$videoId";
            $response = wp_remote_get($oEmbedUrl);
            if (!is_wp_error($response)) {
                $data = json_decode(wp_remote_retrieve_body($response), true);
                if (!empty($data['thumbnail_url'])) {
                    $parsed['thumbnail'] = $data['thumbnail_url'];
                }
            }
        }

        $thumbnailUrl = $parsed['thumbnail'];

        $uploadDir = wp_upload_dir();
        $previewDir = $uploadDir['basedir'] . '/mailerpress-previews/';
        $previewUrlBase = $uploadDir['baseurl'] . '/mailerpress-previews/';

        if (!file_exists($previewDir)) {
            wp_mkdir_p($previewDir);
        }

        $filename = 'preview-' . $parsed['type'] . '-' . preg_replace('/[^a-zA-Z0-9_-]/', '', $parsed['id']) . '.jpg';
        $outputPath = $previewDir . $filename;
        $previewUrl = $previewUrlBase . $filename;

        // Return cached version if it exists
        if (file_exists($outputPath)) {
            return new \WP_REST_Response([
                'url' => $previewUrl,
                'type' => $parsed['type'],
                'id' => $parsed['id'],
            ]);
        }

        if (!function_exists('download_url')) {
            require_once ABSPATH . 'wp-admin/includes/file.php';
        }

        $tmpFile = download_url($thumbnailUrl);
        if (is_wp_error($tmpFile)) {
            return new \WP_REST_Response(['error' => 'Failed to fetch thumbnail'], 400);
        }

        try {
            $image = new \Imagick($tmpFile);
            unlink($tmpFile);

            $width = $image->getImageWidth();
            $height = $image->getImageHeight();

            // 🔹 Dark overlay for contrast
            $overlay = new \Imagick();
            $overlay->newImage($width, $height, new \ImagickPixel('rgba(0,0,0,0.3)'));
            $overlay->setImageFormat('png');
            $image->compositeImage($overlay, \Imagick::COMPOSITE_OVER, 0, 0);
            $overlay->destroy();

            // 🔹 Draw play button (circle + triangle)
            $draw = new \ImagickDraw();
            $draw->setStrokeAntialias(true);

            $centerX = $width / 2;
            $centerY = $height / 2;
            $circleRadius = min($width, $height) * 0.08; // 8% of image width

            $draw->setFillColor(new \ImagickPixel('rgba(255,255,255,0.85)'));
            $draw->circle($centerX, $centerY, $centerX + $circleRadius, $centerY);

            $triangleSize = $circleRadius * 0.8;
            $triangle = [
                ['x' => $centerX - $triangleSize / 2, 'y' => $centerY - $triangleSize / 1.8],
                ['x' => $centerX - $triangleSize / 2, 'y' => $centerY + $triangleSize / 1.8],
                ['x' => $centerX + $triangleSize / 1.5, 'y' => $centerY]
            ];

            $draw->setFillColor(new \ImagickPixel('black'));
            $draw->polygon($triangle);

            $image->setImageMatte(true);
            $image->drawImage($draw);

            // Save final image
            $image->setImageFormat('jpeg');
            $image->setImageCompressionQuality(90);
            $image->writeImage($outputPath);
            $image->destroy();

            return new \WP_REST_Response([
                'url' => $previewUrl,
                'type' => $parsed['type'],
                'id' => $parsed['id'],
            ]);
        } catch (\Exception $e) {
            @unlink($tmpFile);
            return new \WP_REST_Response(['error' => $e->getMessage()], 500);
        }
    }

    private function parseVideoUrl(string $url): ?array
    {
        // YouTube
        if (preg_match('/(?:youtube\.com\/watch\?v=|youtu\.be\/)([a-zA-Z0-9_-]+)/', $url, $m)) {
            return [
                'type' => 'youtube',
                'id' => $m[1],
                // Use maxresdefault if available, else fallback to hqdefault
                'thumbnail' => "https://img.youtube.com/vi/{$m[1]}/maxresdefault.jpg",
            ];
        }

        // Vimeo
        if (preg_match('/vimeo\.com\/(\d+)/', $url, $m)) {
            // Vimeo doesn't have a direct URL pattern for high-res thumbnails,
            // need to fetch via API for best quality
            $vimeoId = $m[1];
            $thumbnail = "https://vumbnail.com/{$vimeoId}.jpg"; // default
            // Optional: fetch JSON for better resolution
            $json = @file_get_contents("https://vimeo.com/api/v2/video/{$vimeoId}.json");
            if ($json) {
                $data = json_decode($json, true);
                if (!empty($data[0]['thumbnail_large'])) {
                    $thumbnail = $data[0]['thumbnail_large'];
                }
            }
            return [
                'type' => 'vimeo',
                'id' => $vimeoId,
                'thumbnail' => $thumbnail,
            ];
        }

        // Dailymotion
        if (preg_match('/dailymotion\.com\/video\/([a-zA-Z0-9]+)/', $url, $m)) {
            return [
                'type' => 'dailymotion',
                'id' => $m[1],
                'thumbnail' => "https://www.dailymotion.com/thumbnail/video/{$m[1]}",
                // Dailymotion only provides small size by default; for higher-res you’d need API
            ];
        }

        return null;
    }

    /**
     * Get contact fetcher based on recipient targeting type
     * 
     * @param string|null $recipientTargeting
     * @param array $lists
     * @param array $tags
     * @param array $segment
     * @return ContactFetcherInterface|null
     */
    private function getContactFetcher(
        ?string $recipientTargeting,
        array $lists,
        array $tags,
        array $segment
    ): ?ContactFetcherInterface {
        $type = $recipientTargeting ?? 'classic';

        return match ($type) {
            'classic' => new ClassicContactFetcher($lists, $tags),
            'segment' => new SegmentContactFetcher(is_array($segment) ? $segment[0] : $segment),
            default => null
        };
    }
}
