<?php

declare(strict_types=1);

namespace MailerPress\Actions\Setup;

\defined('ABSPATH') || exit;

use MailerPress\Core\Attributes\Action;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Kernel;
use MailerPress\Core\Migrations\Manager;

class TableManager
{
    #[Action('plugins_loaded')]
    public function createOrUpdateTable(): void
    {
        $current_version = defined('MAILERPRESS_VERSION_DEV')
            ? MAILERPRESS_VERSION_DEV   // dev override
            : MAILERPRESS_VERSION;      // release version

        $installedVersion = get_option('mailerpress_plugin_version');

        if ($installedVersion === $current_version) {
            return;
        }

        // Run migrations
        $manager = new Manager(Kernel::$config['root'] . '/src/Core/Migrations/migrations');
        $manager->run();

        // Add default data
        $this->addDefaultList();
        $this->addDefaultCategories();

        // Update stored version
        update_option('mailerpress_plugin_version', $current_version);
    }


    private function addDefaultList(): void
    {
        global $wpdb;

        // Vérifie si l'action a déjà été exécutée
        if (get_option('mailerpress_default_list_added')) {
            return;
        }

        $table_name = Tables::get(Tables::MAILERPRESS_LIST);
        $default_list_name = __('Default list', 'mailerpress');

        if ($wpdb->get_var("SHOW TABLES LIKE '{$table_name}'") === $table_name) {
            $exists = $wpdb->get_var($wpdb->prepare(
                "SELECT COUNT(*) FROM {$table_name} WHERE name = %s",
                $default_list_name
            ));

            if (!$exists) {
                $wpdb->insert(
                    $table_name,
                    [
                        'name' => $default_list_name,
                        'created_at' => current_time('mysql'),
                        'updated_at' => current_time('mysql'),
                    ],
                    [
                        '%s', // name
                        '%s', // created_at
                        '%s', // updated_at
                    ]
                );
            }

            // Marquer l'action comme réalisée
            add_option('mailerpress_default_list_added', true);
        }
    }

    private function addDefaultCategories(): void
    {
        global $wpdb;

        if (get_option('mailerpress_default_categories_added')) {
            return;
        }

        $categoriesTable = Tables::get(Tables::MAILERPRESS_CATEGORIES);

        // Define your default categories for each type
        $defaultCategories = [
            'template' => [
                'Ecommerce',
                'Newsletter',
            ],
            'pattern' => [
                'Header',
                'Footer',
                'Call to action',
                'Banners',
                'Text',
            ],
        ];

        foreach ($defaultCategories as $type => $categories) {
            foreach ($categories as $categoryName) {
                // Check if category already exists to avoid duplicates
                $exists = $wpdb->get_var(
                    $wpdb->prepare(
                        "SELECT category_id FROM {$categoriesTable} WHERE name = %s AND type = %s LIMIT 1",
                        $categoryName,
                        $type
                    )
                );

                if (!$exists) {
                    $wpdb->insert(
                        $categoriesTable,
                        [
                            'name' => $categoryName,
                            'slug' => sanitize_title($categoryName),
                            'type' => $type,
                            'created_at' => current_time('mysql'),
                        ],
                        [
                            '%s',
                            '%s',
                            '%s',
                            '%s'
                        ]
                    );
                }
            }
        }

        add_option('mailerpress_default_categories_added', true);

    }

}
