<?php

namespace MailerPress\Actions\Admin;

\defined('ABSPATH') || exit;

use MailerPress\Core\Attributes\Action;
use MailerPress\Core\Capabilities;
use MailerPress\Core\CapabilitiesManager;
use WP_Roles;

class AdminMenu
{
    private array|string $options = [];

    public static function mailerpressRoot(): void
    {

        // 3️⃣ Determine required capability for this page
        $path = $_GET['path'] ?? '';
        $capability = match ($path) {
            '/home/settings', '/home/integrations' => Capabilities::MANAGE_SETTINGS,
            '/home/contacts' => match ($_GET['activeView'] ?? '') {
                'Segmentation' => Capabilities::MANAGE_CONTACT_SEGMENTATION,
                'Contact Lists' => Capabilities::MANAGE_LISTS,
                'Contact Tags' => Capabilities::MANAGE_TAGS,
                default => Capabilities::MANAGE_CONTACTS,
            },
            '/home/templates' => Capabilities::MANAGE_TEMPLATES,
            default => Capabilities::MANAGE_CAMPAIGNS,
        };

        // 4️⃣ Only check capabilities if NOT first load after update
        if (!current_user_can($capability)) :
            ?>
            <div class="wrap">
                <div class="mp-error-page"
                     style="display:flex;flex-direction:column;align-items:center;justify-content:center;min-height:300px;text-align:center;padding:50px 20px;">
                    <h2 style="font-size:28px;font-weight:400;margin-bottom:10px;color:#222;">
                        <?php esc_html_e('Access Denied', 'mailerpress'); ?>
                    </h2>
                    <p style="font-size:16px;color:#555;max-width:400px;margin-bottom:30px;line-height:1.5;">
                        <?php esc_html_e('Sorry, you do not have the necessary permissions to access this page. Please contact your administrator if you believe this is an error.', 'mailerpress'); ?>
                    </p>
                    <a href="<?php echo esc_url(admin_url('admin.php?page=mailerpress%2Fcampaigns.php&path=%2Fhome')); ?>"
                       class="button button-primary">
                        <?php esc_html_e('Return to Dashboard', 'mailerpress'); ?>
                    </a>
                </div>
            </div>
            <?php
            return;
        endif;

        // 5️⃣ Render MailerPress app
        ?>
        <div id="mailerpress"></div>
        <div id="toast-root"></div>
        <div id="modal-root"></div>
        <?php
    }

    public static function mailpressCampaigns(): void
    {
        ?>
        <div id="mailerpress-root"></div>
        <div id="toast-root"></div>
        <div id="modal-root"></div>
        <?php
    }


    #[Action('admin_menu')]
    public function adminMenu(): void
    {
        // Always load options fresh
        $options = apply_filters('mailerpress_white_label_options', []);

        if (!is_array($options) || !is_plugin_active('mailerpress-pro/mailerpress-pro.php')) {
            $options = [];
        }

        $this->options = $options; // Keep available in class if needed

        // Base default labels
        $labels = [
                'main' => __('MailerPress', 'mailerpress'),
                'dashboard' => __('Dashboard', 'mailerpress'),
                'campaigns' => __('Campaigns', 'mailerpress'),
                'audience' => __('Audience', 'mailerpress'),
                'templates' => __('Templates', 'mailerpress'),
                'integrations' => __('Integrations', 'mailerpress'),
                'settings' => __('Settings', 'mailerpress'),
                'licence' => __('Licence', 'mailerpress'),
        ];

        // Override if white-label active
        if (!empty($options['white_label_active'])) {
            $labels['main'] = $options['admin_menu_title'] ?? $labels['main'];
            $labels['dashboard'] = $options['dashboard_name'] ?? $labels['dashboard'];
            $labels['campaigns'] = $options['campaigns_name'] ?? $labels['campaigns'];
            $labels['audience'] = $options['audience_name'] ?? $labels['audience'];
            $labels['templates'] = $options['templates_name'] ?? $labels['templates'];
            $labels['integrations'] = $options['integrations_name'] ?? $labels['integrations'];
            $labels['settings'] = $options['settings_name'] ?? $labels['settings'];
        }

        // Fallback icon if no white label
        $menu_icon = !empty($options['white_label_active'])
                ? $options['admin_menu_icon']
                : 'data:image/svg+xml;base64,PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz4KPHN2ZyBpZD0iQ2FscXVlXzEiIHhtbG5zPSJodHRwOi8vd3d3LnczLm9yZy8yMDAwL3N2ZyIgeG1sbnM6eGxpbms9Imh0dHA6Ly93d3cudzMub3JnLzE5OTkveGxpbmsiIHhtbG5zOnNlcmlmPSJodHRwOi8vd3d3LnNlcmlmLmNvbS8iIHZlcnNpb249IjEuMSIgdmlld0JveD0iMCAwIDEwNTguMSA4NzMuOSI+CiAgPCEtLSBHZW5lcmF0b3I6IEFkb2JlIElsbHVzdHJhdG9yIDI5LjIuMCwgU1ZHIEV4cG9ydCBQbHVnLUluIC4gU1ZHIFZlcnNpb246IDIuMS4wIEJ1aWxkIDEwOCkgIC0tPgogIDxkZWZzPgogICAgPHN0eWxlPgogICAgICAuc3QwIHsKICAgICAgICBmaWxsOiAjMWUxZDFkOwogICAgICB9CiAgICA8L3N0eWxlPgogIDwvZGVmcz4KICA8cGF0aCBjbGFzcz0ic3QwIiBkPSJNMzE4LjMsMzg3LjhjMCwwLS4xLDAtLjIsMC0uNiwwLTEsLjUtMSwxaDBjMS4zLDk3LjYsMS45LDE5NS4yLDEuOSwyOTMsMCwzMiw0LjEsNTMuMiwzMC45LDY2LjcsNS45LDMsMTYuMyw0LjQsMzEuMSw0LjMsMTgwLjMtMSwzNDYuMy0uOSw0OTcuOS40LDQ1LjcuNCw2OC41LTIyLjksNjguNS02OS45LS4yLTUzLjEtLjUtMjE0LjgtLjgtNDg1LjEsMC0yMC45LTEuMy0zNC4yLTMuOS0zOS44LTUuNy0xMi4yLTEzLjYtMjItMjMuOC0yOS4xLTctNC45LTE4LjctNy40LTM1LjEtNy40LTIzOC41LDAtNDc2LjkuMS03MTUuNCwwLTIyLDAtMzguMSw4LTQ4LjIsMjQtNS45LDkuNS04LjgsMjQuNC04LjYsNDQuOSwxLjEsMTU1LjcsMi4zLDMwNi4zLDMuNCw0NTEuOCwwLDIuNC0xLjcsNC40LTQuMSw0LjgtMjYuMSw0LjItNTAuNCw0LjQtNzIuNi0xMi4yQzEyLjgsNjE2LC4xLDU4OS42LjEsNTU2LjEsMCwzMjYuMywwLDE5Ny40LDAsMTY5LjMtLjEsMTA5LjksMjQsNjMuNCw3Mi41LDI5LjcsMTA4LDUsMTM4LjIsMCwxODQuOCwwYzMwOC4yLjIsNTM4LjYuMiw2OTEuMS4yLDc5LjIsMCwxMzUuOCwzNiwxNjkuNywxMDgsNi43LDE0LjEsMTAsMzAuOSwxMC4xLDUwLjQsMS4yLDI3Mi42LDIsNDU0LjEsMi41LDU0NC42LjMsNTAuOC0xOS43LDkzLjYtNjAuMSwxMjguMi0zNy41LDMyLjEtNzEuOCw0MS45LTEyNC40LDQyLTMwLjMsMC0xODQuOS4zLTQ2My44LjUtNDYuNCwwLTc2LjYtMi40LTkwLjYtNy41LTUxLjEtMTguMi05Ni42LTY3LTEwNi4yLTEyMi4zLTEtNS45LTEuNi0yOC4zLTEuNy02Ny4xLS44LTE1OS43LS44LTMxMC4xLDAtNDUxLjEsMC0zLjEsMS42LTQuNyw0LjctNC43aDk3LjRjNC4yLDAsOC4yLDEuOCwxMSw0LjlsMjAyLjcsMjI0LjRjMCwwLC4xLjEuMi4yLDIuMywyLjMsNi4xLDIuMyw4LjQsMCwwLDAsLjEtLjEuMi0uMmwyMDAtMjIxLjZjNC4yLTQuNiwxMC4xLTcuMiwxNi4yLTcuMmg4NS45YzIuNCwwLDQuNCwyLDQuNCw0LjR2NDE1LjZjMCwxLjctLjgsMi42LTIuNSwyLjZoLTk3LjljLTEuNiwwLTMtMS4zLTMtM3YtMjQ3LjljMC01LjgtMS45LTYuNS01LjYtMi4xbC0xOTcuMywyMjcuNmMtLjgsMS0yLDEuNS0zLjMsMS41cy0yLjItLjUtMy0xLjNjLTcxLjgtNzEtMTM2LjItMTQ3LjQtMjA3LjUtMjI5LTEuMi0xLjMtMi41LTIuMS0zLjktMi40WiIvPgo8L3N2Zz4='; // (your SVG kept here)

        // Register top-level menu
        add_menu_page(
                $labels['main'],
                $labels['main'],
                'edit_posts',
                'mailerpress/campaigns.php',
                [$this, 'mailerpressRoot'],
                $menu_icon,
                20
        );

        // Define submenus
        $submenus = [
                [
                        'title' => $labels['dashboard'],
                        'menu_title' => $labels['dashboard'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome',
                        'cap' => Capabilities::MANAGE_CAMPAIGNS,
                ],
                [
                        'title' => __('New Campaign', 'mailerpress'),
                        'menu_title' => __('New Campaign', 'mailerpress'),
                        'slug' => 'mailerpress/new',
                        'cap' => Capabilities::MANAGE_CAMPAIGNS,
                ],
                [
                        'title' => $labels['campaigns'],
                        'menu_title' => $labels['campaigns'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fcampaigns',
                        'cap' => Capabilities::MANAGE_CAMPAIGNS,
                ],
                [
                        'title' => $labels['audience'],
                        'menu_title' => $labels['audience'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fcontacts',
                        'cap' => Capabilities::MANAGE_CONTACTS,
                ],
                [
                        'title' => $labels['templates'],
                        'menu_title' => $labels['templates'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Ftemplates',
                        'cap' => Capabilities::MANAGE_TEMPLATES,
                ],
                [
                        'title' => $labels['integrations'],
                        'menu_title' => $labels['integrations'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fintegrations',
                        'cap' => Capabilities::MANAGE_SETTINGS,

                ],
                [
                        'title' => $labels['settings'],
                        'menu_title' => $labels['settings'],
                        'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fsettings',
                        'cap' => Capabilities::MANAGE_SETTINGS,

                ],
        ];

        if (function_exists('is_plugin_active') && is_plugin_active('mailerpress-pro/mailerpress-pro.php')) {
            $submenus[] = [
                    'title' => $labels['licence'],
                    'menu_title' => $labels['licence'],
                    'slug' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fsettings&activeView=Licence',
                    'cap' => Capabilities::MANAGE_SETTINGS,
            ];
        }

        // Register all submenus
        foreach ($submenus as $submenu) {
            add_submenu_page(
                    'mailerpress/campaigns.php',
                    $submenu['title'],
                    $submenu['menu_title'],
                    $submenu['cap'],
                    $submenu['slug'],
                    [$this, 'mailpressCampaigns']
            );
        }
    }


    #[Action('admin_head')]
    public function hideFirstMenuItem(): void
    {
        ?>
        <style>
            #toplevel_page_mailerpress-campaigns ul.wp-submenu li:nth-child(3) {
                display: none !important;
            }
        </style>
        <?php
    }

    #[Action('parent_file', priority: 1000)]
    public function setParentFile(string $parent_file): string
    {
        global $plugin_page;

        // Normalize plugin_page for comparison
        $page = $plugin_page ? rawurldecode($plugin_page) : '';

        // Use decoded slugs here — match your actual slugs used in add_submenu_page
        $mailerpress_submenus = [
                'mailerpress/campaigns.php&path=/home',
                'mailerpress/new',
                'mailerpress/campaigns.php&path=/home/campaigns',
                'mailerpress/campaigns.php&path=/home/contacts',
                'mailerpress/campaigns.php&path=/home/templates',
                'mailerpress/campaigns.php&path=/home/integrations',
                'mailerpress/campaigns.php&path=/home/settings',
        ];

        if (in_array($page, $mailerpress_submenus, true)) {
            return 'mailerpress/campaigns.php';
        }

        return $parent_file;
    }

    #[Action('admin_body_class')]
    public function addAdminBodyClass(string $classes): string
    {
        // add a body class so you can target MailerPress admin pages easily
        if (isset($_GET['page']) && strpos(rawurldecode((string)$_GET['page']), 'mailerpress') === 0) {
            $classes .= ' mailerpress-page';
        }
        return $classes;
    }

    #[Action('parent_file', priority: 1000)]
    public function fixParentFile(string $parent_file): string
    {
        global $submenu_file;

        // Current request
        $current_page = isset($_GET['page']) ? rawurldecode($_GET['page']) : '';
        $current_path = isset($_GET['path']) ? rawurldecode($_GET['path']) : '';

        // Map decoded page/path → encoded slug exactly as used in add_submenu_page
        $map = [
                'mailerpress/campaigns.php&path=/home' => 'mailerpress%2Fcampaigns.php&path=%2Fhome',
                'mailerpress/new' => 'mailerpress/new',
                'mailerpress/campaigns.php&path=/home/campaigns' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fcampaigns',
                'mailerpress/campaigns.php&path=/home/contacts' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fcontacts',
                'mailerpress/campaigns.php&path=/home/templates' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Ftemplates',
                'mailerpress/campaigns.php&path=/home/integrations' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fintegrations',
                'mailerpress/campaigns.php&path=/home/settings' => 'mailerpress%2Fcampaigns.php&path=%2Fhome%2Fsettings',
        ];

        $decoded_key = $current_page . ($current_path ? "&path={$current_path}" : '');

        if (isset($map[$decoded_key])) {
            $parent_file = 'mailerpress/campaigns.php';
            $submenu_file = $map[$decoded_key];
        }

        return $parent_file;
    }


}
