<?php

namespace MailerPress\Actions\ActionScheduler\Processors;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\Attributes\Action;
use MailerPress\Core\DynamicPostRenderer;
use MailerPress\Core\Enums\Tables;
use MailerPress\Core\Interfaces\ContactFetcherInterface;
use MailerPress\Core\Kernel;
use MailerPress\Models\Contacts;
use MailerPress\Services\ClassicContactFetcher;
use MailerPress\Services\SegmentContactFetcher;

class MailerPressEmailBatch
{
    /**
     * @param $sendType
     * @param $post
     * @param $config
     * @param $scheduledAt
     * @param $recipientTargeting
     * @param $lists
     * @param $tags
     * @param $segment
     * @return void|\WP_REST_Response
     */
    #[Action('mailerpress_batch_email', priority: 10, acceptedArgs: 8)]
    public function process(
        $sendType,
        $post,
        $config,
        $scheduledAt,
        $recipientTargeting,
        $lists,
        $tags,
        $segment,
    ) {
        global $wpdb;

        // Select fetcher based on targeting type
        $fetcher = $this->getFetcher($recipientTargeting, $lists, $tags, $segment);

        if (!$fetcher) {
            return;
        }

        // Frequency settings
        $frequencySending = get_option('mailerpress_frequency_sending', [
            "settings" => [
                "numberEmail" => 25,
                "config" => ["value" => 5, "unit" => "minutes"],
            ],
        ]);

        if (is_string($frequencySending)) {
            $decoded = json_decode($frequencySending, true);
            $frequencySending = is_array($decoded) ? $decoded : [
                "settings" => [
                    "numberEmail" => 25,
                    "config" => ["value" => 5, "unit" => "minutes"],
                ],
            ];
        }

        $numberEmail = $frequencySending['settings']['numberEmail'] ?? 25;
        $frequencyConfig = $frequencySending['settings']['config'] ?? ['value' => 5, 'unit' => 'minutes'];

        $unit_multipliers = ['seconds' => 1, 'minutes' => MINUTE_IN_SECONDS, 'hours' => HOUR_IN_SECONDS];
        $interval_seconds = ($frequencyConfig['value'] ?? 5) * ($unit_multipliers[$frequencyConfig['unit']] ?? MINUTE_IN_SECONDS);

        $status = ('future' === $sendType) ? 'scheduled' : 'pending';

        // Get subject from config or fallback to campaign title
        $subject = $config['subject'] ?? '';
        if (empty($subject) && !empty($post)) {
            $campaign = get_post($post);
            $subject = $campaign ? $campaign->post_title : '';
        }

        // Check if a batch already exists for this campaign (created in createBatchV2)
        $existing_batch = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT id FROM " . Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES) . " 
                 WHERE campaign_id = %d AND status IN ('scheduled', 'pending') 
                 ORDER BY id DESC LIMIT 1",
                $post
            )
        );

        if ($existing_batch) {
            // Use existing batch
            $batch_id = (int)$existing_batch->id;
        } else {
            // Insert new batch record (fallback for old code paths)
            $wpdb->insert(
                Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
                [
                    'status' => $status,
                    'total_emails' => 0,
                    'sender_name' => $config['fromName'] ?? '',
                    'sender_to' => $config['fromTo'] ?? '',
                    'subject' => $subject,
                    'scheduled_at' => $scheduledAt,
                    'campaign_id' => $post,
                ]
            );

            $batch_id = $wpdb->insert_id;
            if (!$batch_id) {
                return new \WP_REST_Response(null, 400);
            }
        }

        $htmlContent = get_option('mailerpress_batch_' . $post . '_html');

        if (containsStartQueryBlock($htmlContent)) {
            $renderer = new DynamicPostRenderer($htmlContent);
            $renderer->setCampaignId($post);
            $renderedHtml = $renderer->render();

            // Check if there are any new posts (not processed yet)
            if (empty($renderedHtml)) {
                // No new posts to send, stop the process
                return new \WP_REST_Response(__('No new content to send for this automated campaign'), 400);
            }

            $htmlContent = $renderedHtml;
        }


        $dbChunk = 1000;
        $offset = 0;
        $chunk_index = 0;
        $totalEmails = 0;
        $now = time();

        // Calculate base time for scheduling chunks
        $base_time = $now;
        if ('future' === $sendType && !empty($scheduledAt)) {
            $base_time = $this->convert_scheduled_at_to_timestamp($scheduledAt);
            if ($base_time <= $now) {
                // If scheduled time is past, fallback to now
                $base_time = $now;
            }
        }

        $foundContacts = false;

        $servicesData = get_option('mailerpress_email_services', []);

        // Get API key from config or from services data
        $apiKey = $config['api_key'] ?? '';
        if (empty($apiKey) && !empty($servicesData['default_service'])) {
            $defaultService = $servicesData['default_service'];
            $apiKey = $servicesData['services'][$defaultService]['conf']['api_key'] ?? '';
        }

        if ('mailerpress' === $servicesData['default_service']) {
            error_log('MailerPress default_service is set to mailerpress');

            $offset = 0;
            $foundContacts = false;
            $totalEmails = 0;

            do {
                $contacts = $fetcher->fetch($dbChunk, $offset);
                if (empty($contacts)) {
                    break;
                }

                $foundContacts = true;
                $totalEmails += count($contacts);

                // Split contacts into smaller chunks for Laravel
                $sendingChunks = array_chunk($contacts, 1000);

                foreach ($sendingChunks as $sendingChunk) {



                    $payload = [
                        'wp_batch_id' => $batch_id,
                        'domain_id' => $servicesData['services']['mailerpress']['conf']['domain'],
                        'name' => $subject,
                        'emails' => array_map(function ($c) use ($htmlContent, $subject) {
                            $contact = Kernel::getContainer()->get(Contacts::class)->get($c);
                            return [
                                'to' => $contact->email,
                                'subject' => $subject,
                                'body' => $htmlContent,
                            ];
                        }, $sendingChunk),
                    ];

                    $payloadJson = json_encode($payload, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

                    error_log("Payload JSON for Postman:\n" . $payloadJson);

                    // Send to Laravel API
                    $response = wp_remote_post('https://mailerpress.pro/api/batch', [
                        'headers' => [
                            'Content-Type' => 'application/json',
                            'x-api-key' => $servicesData['services']['mailerpress']['conf']['api_key'],
                        ],
                        'body' => $payloadJson,
                        'timeout' => 60,
                    ]);

                    if (is_wp_error($response)) {
                        error_log('Laravel API error: ' . $response->get_error_message());
                        // Mark batch as failed if API error occurs
                        $this->markBatchAsFailed($batch_id, $post, 'Laravel API error: ' . $response->get_error_message());
                        return;
                    } else {
                        $status_code = wp_remote_retrieve_response_code($response);
                        $body = wp_remote_retrieve_body($response);
                        error_log("Laravel API response ({$status_code}): {$body}");

                        // Check if response indicates failure
                        if ($status_code >= 400) {
                            $this->markBatchAsFailed($batch_id, $post, "Laravel API returned error status {$status_code}: {$body}");
                            return;
                        }
                    }
                }

                $offset += $dbChunk;
            } while (!empty($contacts));

            // Update batch totals
            if (!$foundContacts) {
                $wpdb->update(
                    Tables::get(Tables::MAILERPRESS_CAMPAIGNS),
                    ['status' => 'error', 'updated_at' => current_time('mysql')],
                    ['campaign_id' => intval($post)],
                    ['%s', '%s'],
                    ['%d']
                );
            } else {
                $wpdb->update(
                    Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
                    ['total_emails' => $totalEmails],
                    ['id' => $batch_id],
                    ['%d'],
                    ['%d']
                );
                do_action('mailerpress_batch_event', $status, $post, $batch_id);
            }
        } else {
            do {
                $contacts = $fetcher->fetch($dbChunk, $offset);
                if (empty($contacts)) {
                    break;
                }

                /** TODO: implement sending logic with mailerpress.pro */

                $foundContacts = true;
                $totalEmails += count($contacts);
                $sendingChunks = array_chunk($contacts, $numberEmail);

                foreach ($sendingChunks as $sendingChunk) {
                    $transient_key = 'mailerpress_chunk_' . $batch_id . '_' . $chunk_index;
                    set_transient($transient_key, [
                        'html' => $htmlContent,
                        'subject' => $subject,
                        'sender_name' => $config['fromName'] ?? '',
                        'api_key' => $apiKey,
                        'sender_to' => $config['fromTo'] ?? '',
                        'contacts' => $sendingChunk,
                        'scheduled_at' => $scheduledAt,
                        'webhook_url' => get_rest_url(null, 'mailerpress/v1/webhook/notify'),
                        'sendType' => $sendType,
                    ]);

                    $scheduled_time = $base_time + ($chunk_index * $interval_seconds);

                    as_schedule_single_action(
                        $scheduled_time,
                        'mailerpress_process_contact_chunk',
                        [$batch_id, $transient_key],
                        'mailerpress'
                    );

                    $chunk_index++;
                }

                $offset += $dbChunk;
            } while (!empty($contacts));

            if (!$foundContacts) {
                $wpdb->update(
                    Tables::get(Tables::MAILERPRESS_CAMPAIGNS),
                    ['status' => 'error', 'updated_at' => current_time('mysql')],
                    ['campaign_id' => intval($post)],
                    ['%s', '%s'],
                    ['%d']
                );
            }

            // Update batch totals or mark error
            if ($totalEmails > 0) {
                $wpdb->update(
                    Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES),
                    ['total_emails' => $totalEmails],
                    ['id' => $batch_id],
                    ['%d'],
                    ['%d']
                );
                do_action('mailerpress_batch_event', $status, $post, $batch_id);
            } else {
                $wpdb->update(
                    Tables::get(Tables::MAILERPRESS_CAMPAIGNS),
                    ['status' => 'error', 'updated_at' => current_time('mysql')],
                    ['campaign_id' => intval($post)],
                    ['%s', '%s'],
                    ['%d']
                );
            }
        }
    }

    /**
     * Convert scheduled_at string (WP timezone) to Unix timestamp.
     */
    private function convert_scheduled_at_to_timestamp(string $scheduledAt): int
    {
        $tz = function_exists('wp_timezone') ? wp_timezone() : new \DateTimeZone(wp_timezone_string());
        try {
            $dt = new \DateTime($scheduledAt, $tz);
            return $dt->getTimestamp();
        } catch (\Exception $e) {
            // fallback to current time if parsing fails
            return time();
        }
    }

    /**
     * Mark batch and campaign as failed
     */
    private function markBatchAsFailed(int $batch_id, int $campaign_id, string $error_message): void
    {
        global $wpdb;

        $batchTable = Tables::get(Tables::MAILERPRESS_EMAIL_BATCHES);
        $campaignTable = Tables::get(Tables::MAILERPRESS_CAMPAIGNS);

        // Update batch status to failed
        $wpdb->update(
            $batchTable,
            [
                'status' => 'failed',
                'updated_at' => current_time('mysql'),
            ],
            ['id' => $batch_id],
            ['%s', '%s'],
            ['%d']
        );

        // Update campaign status to error
        $wpdb->update(
            $campaignTable,
            [
                'status' => 'error',
                'updated_at' => current_time('mysql'),
            ],
            ['campaign_id' => $campaign_id],
            ['%s', '%s'],
            ['%d']
        );

        // Log error
        error_log(sprintf(
            'MailerPress: Batch %d failed - %s',
            $batch_id,
            $error_message
        ));
    }

    /**
     * Returns a ContactFetcher based on recipient targeting
     */
    private function getFetcher(
        string $type,
        array $lists,
        array $tags,
        $segment
    ): ?ContactFetcherInterface {
        return match ($type) {
            'classic' => new ClassicContactFetcher($lists, $tags),
            'segment' => new SegmentContactFetcher(is_array($segment) ? $segment[0] : $segment),
            default => null
        };
    }
}
