<?php

declare(strict_types=1);

namespace MailerPress\Actions\ActionScheduler\Processors;

\defined('ABSPATH') || exit;

use MailerPress\Core\Attributes\Action;
use MailerPress\Core\Attributes\Endpoint;
use MailerPress\Core\Enums\Tables;
use ZipArchive;
use WP_REST_Request;

class ExportContact
{
    #[Action('mailerpress_export_contact_batch', priority: 10, acceptedArgs: 4)]
    public function run($export_id, $status = null, $offset = null, $contact_ids = []): void
    {
        global $wpdb;
        $table = Tables::get(Tables::MAILERPRESS_CONTACT);
        $batch_size = 200;
        $contacts = [];

        if (!empty($contact_ids) && is_array($contact_ids)) {
            $placeholders = implode(',', array_fill(0, count($contact_ids), '%d'));
            $sql = "SELECT email, first_name, last_name, updated_at FROM $table WHERE contact_id IN ($placeholders)";
            $prepared = $wpdb->prepare($sql, $contact_ids);
            $contacts = $wpdb->get_results($prepared, ARRAY_A);
        } elseif (isset($status, $offset)) {
            $contacts = $wpdb->get_results($wpdb->prepare(
                "SELECT email, first_name, last_name, updated_at FROM $table WHERE subscription_status = %s LIMIT %d OFFSET %d",
                $status,
                $batch_size,
                $offset
            ), ARRAY_A);
        }

        if (empty($contacts)) {
            return;
        }

        $upload_dir = wp_upload_dir();
        $export_dir = trailingslashit($upload_dir['basedir']) . 'mailerpress_exports/' . $export_id;
        if (!file_exists($export_dir)) {
            wp_mkdir_p($export_dir);
        }

        $status = $status ?? 'export';
        $file_path = $export_dir . "/{$status}.csv";
        $is_new_file = !file_exists($file_path);

        $file = fopen($file_path, 'a');
        if ($is_new_file && isset($contacts[0])) {
            fputcsv($file, array_keys($contacts[0]));
        }

        foreach ($contacts as $contact) {
            fputcsv($file, $contact);
        }

        fclose($file);
    }

    #[Action('mailerpress_finalize_export_contact_zip', priority: 10, acceptedArgs: 2)]
    public function handleZip($export_id, $email): void
    {
        if (!$export_id) {
            return;
        }

        $upload_dir = wp_upload_dir();
        $export_dir = trailingslashit($upload_dir['basedir']) . "mailerpress_exports/{$export_id}/";
        $zip_path = trailingslashit($upload_dir['basedir']) . "mailerpress_exports/{$export_id}.zip";

        if (!is_dir($export_dir)) {
            return;
        }

        $zip = new ZipArchive();
        if ($zip->open($zip_path, ZipArchive::CREATE | ZipArchive::OVERWRITE) !== true) {
            return;
        }

        foreach (glob($export_dir . "*.csv") as $file) {
            $filename = basename($file);
            $zip->addFile($file, $filename);
        }

        $zip->close();

        array_map('unlink', glob($export_dir . "*.csv"));
        rmdir($export_dir);

        $token = wp_generate_password(24, false);
        $expires = time() + DAY_IN_SECONDS; // 24 hours

        $export_data = [
            'zip_path' => $zip_path,
            'expires'  => $expires,
            'token'    => $token,
        ];

        update_option("mailerpress_export_{$export_id}", $export_data);

        $download_url = rest_url("mailerpress/v1/export/{$export_id}?token={$token}");

        if ($email && is_email($email)) {
            wp_mail(
                $email,
                esc_html__('Your MailerPress Export is Ready', 'mailerpress'),
                sprintf(
                    esc_html__(
                        "Hello,\n\nYour export is complete. You can download it here:\n\n%s\n\nThis link will expire in 24 hours.",
                        'mailerpress'
                    ),
                    esc_url($download_url)
                ),
                ['Content-Type: text/plain; charset=UTF-8']
            );
        }
    }
}
