<?php

namespace MailerPressPro\Services\ThirdParty;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Repeater;
use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class MailerPressElementorWidget extends Widget_Base
{
    public function get_name()
    {
        return 'mailerpress-optin';
    }

    public function get_title()
    {
        return __('MailerPress Form', 'mailerpress-pro');
    }

    public function get_icon()
    {
        return 'eicon-form-horizontal';
    }

    public function get_categories()
    {
        return ['general'];
    }

    public function get_keywords()
    {
        return ['mailerpress', 'form', 'optin', 'newsletter', 'subscribe'];
    }

    protected function register_controls()
    {
        // Section: Form Fields
        $this->start_controls_section(
            'form_fields_section',
            [
                'label' => __('Form Fields', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $repeater = new Repeater();

        $repeater->add_control(
            'field_type',
            [
                'label' => __('Field Type', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => 'email',
                'options' => [
                    'email' => __('Email', 'mailerpress-pro'),
                    'text' => __('Text', 'mailerpress-pro'),
                    'textarea' => __('Textarea', 'mailerpress-pro'),
                    'tel' => __('Phone', 'mailerpress-pro'),
                    'number' => __('Number', 'mailerpress-pro'),
                    'date' => __('Date', 'mailerpress-pro'),
                ],
            ]
        );

        $repeater->add_control(
            'field_label',
            [
                'label' => __('Label', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('Enter field label', 'mailerpress-pro'),
            ]
        );

        $repeater->add_control(
            'field_name',
            [
                'label' => __('Field Name', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('Enter field name', 'mailerpress-pro'),
            ]
        );

        $repeater->add_control(
            'field_required',
            [
                'label' => __('Required', 'mailerpress-pro'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'mailerpress-pro'),
                'label_off' => __('No', 'mailerpress-pro'),
                'return_value' => 'yes',
                'default' => 'no',
            ]
        );

        $repeater->add_control(
            'field_placeholder',
            [
                'label' => __('Placeholder', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('Enter placeholder text', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'form_fields',
            [
                'label' => __('Form Fields', 'mailerpress-pro'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $repeater->get_controls(),
                'default' => [
                    [
                        'field_type' => 'email',
                        'field_label' => __('Email', 'mailerpress-pro'),
                        'field_name' => 'email',
                        'field_required' => 'yes',
                        'field_placeholder' => __('Enter your email', 'mailerpress-pro'),
                    ],
                ],
                'title_field' => '{{{ field_label }}}',
            ]
        );

        $this->end_controls_section();

        // Section: MailerPress Settings
        $this->start_controls_section(
            'mailerpress_section',
            [
                'label' => __('MailerPress Settings', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $mailerpress_lists = $this->get_mailerpress_lists();

        $this->add_control(
            'mailerpress_list',
            [
                'label' => __('MailerPress Lists', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $mailerpress_lists,
                'default' => [],
                'description' => __('Leave empty to automatically use the default list.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'mailerpress_tags',
            [
                'label' => __('MailerPress Tags', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT2,
                'multiple' => true,
                'options' => $this->get_mailerpress_tags(),
                'default' => [],
            ]
        );

        $this->add_control(
            'double_optin',
            [
                'label' => __('Double Opt-In', 'mailerpress-pro'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'mailerpress-pro'),
                'label_off' => __('No', 'mailerpress-pro'),
                'return_value' => 'yes',
                'default' => 'yes',
                'description' => __('When enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'optin_source',
            [
                'label' => __('Opt-in Source', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., elementor, website, landing-page', 'mailerpress-pro'),
                'description' => __('Optional: Set a custom opt-in source identifier for tracking where contacts subscribed from. Leave empty to use default value.', 'mailerpress-pro'),
            ]
        );

        $this->end_controls_section();

        // Section: Field Mapping
        $this->start_controls_section(
            'field_mapping_section',
            [
                'label' => __('Field Mapping', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $mapping_repeater = new Repeater();

        $mapping_repeater->add_control(
            'form_field',
            [
                'label' => __('Form Field Name', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., email, firstname, lastname', 'mailerpress-pro'),
                'description' => __('Enter the field name from the "Form Fields" section above (the "Field Name" value, not the label).', 'mailerpress-pro'),
            ]
        );

        $mapping_repeater->add_control(
            'mailerpress_field',
            [
                'label' => __('MailerPress Field', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => $this->get_mailerpress_fields(),
            ]
        );

        $this->add_control(
            'field_mapping',
            [
                'label' => __('Field Mapping', 'mailerpress-pro'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $mapping_repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ form_field }}} → {{{ mailerpress_field }}}',
                'description' => __('Map form fields to MailerPress fields. Leave empty to use default mapping (email, contactFirstName, contactLastName).', 'mailerpress-pro'),
            ]
        );

        $this->end_controls_section();

        // Section: Custom Fields Mapping
        $this->start_controls_section(
            'custom_fields_section',
            [
                'label' => __('Custom Fields Mapping', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $custom_repeater = new Repeater();

        $custom_repeater->add_control(
            'form_field_custom',
            [
                'label' => __('Form Field Name', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => '',
                'placeholder' => __('e.g., email, firstname, lastname', 'mailerpress-pro'),
                'description' => __('Enter the field name from the "Form Fields" section above (the "Field Name" value, not the label).', 'mailerpress-pro'),
            ]
        );

        $custom_repeater->add_control(
            'custom_field_key',
            [
                'label' => __('Custom Field Key', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => '',
                'options' => $this->get_custom_fields(),
            ]
        );

        $this->add_control(
            'custom_fields_mapping',
            [
                'label' => __('Custom Fields Mapping', 'mailerpress-pro'),
                'type' => Controls_Manager::REPEATER,
                'fields' => $custom_repeater->get_controls(),
                'default' => [],
                'title_field' => '{{{ form_field_custom }}} → {{{ custom_field_key }}}',
            ]
        );

        $this->end_controls_section();

        // Section: Form Settings
        $this->start_controls_section(
            'form_settings_section',
            [
                'label' => __('Form Settings', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_CONTENT,
            ]
        );

        $this->add_control(
            'form_columns',
            [
                'label' => __('Form Columns', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'default' => '1',
                'options' => [
                    '1' => __('1 Column', 'mailerpress-pro'),
                    '2' => __('2 Columns', 'mailerpress-pro'),
                    '3' => __('3 Columns', 'mailerpress-pro'),
                ],
                'description' => __('Select the number of columns for the form fields layout.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'submit_button_text',
            [
                'label' => __('Submit Button Text', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXT,
                'default' => __('Subscribe', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'success_message',
            [
                'label' => __('Success Message', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('Thank you for subscribing!', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'error_message',
            [
                'label' => __('Error Message', 'mailerpress-pro'),
                'type' => Controls_Manager::TEXTAREA,
                'default' => __('An error occurred. Please try again.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'enable_redirect',
            [
                'label' => __('Enable Redirect After Success', 'mailerpress-pro'),
                'type' => Controls_Manager::SWITCHER,
                'label_on' => __('Yes', 'mailerpress-pro'),
                'label_off' => __('No', 'mailerpress-pro'),
                'return_value' => 'yes',
                'default' => 'no',
                'description' => __('Redirect to a page after successful form submission.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'redirect_page',
            [
                'label' => __('Redirect To Page', 'mailerpress-pro'),
                'type' => Controls_Manager::SELECT,
                'options' => $this->get_pages_list(),
                'default' => '',
                'condition' => [
                    'enable_redirect' => 'yes',
                ],
                'description' => __('Select the page to redirect to after successful submission.', 'mailerpress-pro'),
            ]
        );

        $this->add_control(
            'redirect_delay',
            [
                'label' => __('Redirect Delay (seconds)', 'mailerpress-pro'),
                'type' => Controls_Manager::NUMBER,
                'default' => 2,
                'min' => 0,
                'max' => 10,
                'step' => 0.5,
                'condition' => [
                    'enable_redirect' => 'yes',
                ],
                'description' => __('Delay before redirecting (0 = immediate).', 'mailerpress-pro'),
            ]
        );

        $this->end_controls_section();

        // Section: Style
        $this->start_controls_section(
            'style_section',
            [
                'label' => __('Style', 'mailerpress-pro'),
                'tab' => Controls_Manager::TAB_STYLE,
            ]
        );

        $this->add_control(
            'form_width',
            [
                'label' => __('Form Width', 'mailerpress-pro'),
                'type' => Controls_Manager::SLIDER,
                'size_units' => ['px', '%'],
                'range' => [
                    'px' => [
                        'min' => 0,
                        'max' => 1000,
                        'step' => 1,
                    ],
                    '%' => [
                        'min' => 0,
                        'max' => 100,
                    ],
                ],
                'default' => [
                    'unit' => '%',
                    'size' => 100,
                ],
                'selectors' => [
                    '{{WRAPPER}} .mailerpress-optin-form' => 'width: {{SIZE}}{{UNIT}};',
                ],
            ]
        );

        $this->end_controls_section();
    }

    protected function render()
    {
        $settings = $this->get_settings_for_display();
        $widget_id = $this->get_id();

        if (empty($settings['form_fields'])) {
            return;
        }

        $columns = isset($settings['form_columns']) ? intval($settings['form_columns']) : 1;
        $form_class = 'mailerpress-form-columns-' . $columns;

?>
        <div class="mailerpress-optin-form" data-widget-id="<?php echo esc_attr($widget_id); ?>">
            <form class="mailerpress-optin-form-element <?php echo esc_attr($form_class); ?>" method="post" action="">
                <?php wp_nonce_field('mailerpress_optin_submit', 'mailerpress_optin_nonce'); ?>
                <input type="hidden" name="widget_id" value="<?php echo esc_attr($widget_id); ?>">
                <input type="hidden" name="element_id" value="<?php echo esc_attr($this->get_id()); ?>">
                <input type="hidden" name="page_id" value="<?php echo esc_attr(get_the_ID()); ?>">
                <input type="hidden" name="widget_settings" value="<?php echo esc_attr(wp_json_encode($settings, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT)); ?>">
                <?php if (class_exists('\MailerPress\Services\RateLimitConfig') && \MailerPress\Services\RateLimitConfig::isHoneypotEnabled()): ?>
                    <input type="text" name="website" value="" style="position: absolute; left: -9999px; width: 1px; height: 1px; opacity: 0; pointer-events: none; tabindex: -1;" autocomplete="off" aria-hidden="true">
                <?php endif; ?>

                <div class="mailerpress-form-fields-wrapper">
                    <?php foreach ($settings['form_fields'] as $index => $field): ?>
                        <div class="mailerpress-form-field">
                            <label for="mailerpress-field-<?php echo esc_attr($widget_id . '-' . $index); ?>">
                                <?php echo esc_html($field['field_label']); ?>
                                <?php if ($field['field_required'] === 'yes'): ?>
                                    <span class="required">*</span>
                                <?php endif; ?>
                            </label>
                            <?php
                            $field_name = !empty($field['field_name']) ? $field['field_name'] : 'field_' . $index;
                            $field_id = 'mailerpress-field-' . $widget_id . '-' . $index;
                            $required = $field['field_required'] === 'yes' ? 'required' : '';
                            $placeholder = !empty($field['field_placeholder']) ? $field['field_placeholder'] : '';

                            switch ($field['field_type']) {
                                case 'textarea':
                                    echo '<textarea id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" placeholder="' . esc_attr($placeholder) . '" ' . $required . '></textarea>';
                                    break;
                                case 'email':
                                case 'text':
                                case 'tel':
                                case 'number':
                                case 'date':
                                    echo '<input type="' . esc_attr($field['field_type']) . '" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" placeholder="' . esc_attr($placeholder) . '" ' . $required . '>';
                                    break;
                            }
                            ?>
                        </div>
                    <?php endforeach; ?>
                </div>

                <div class="mailerpress-form-submit">
                    <button type="submit" class="mailerpress-submit-button">
                        <?php echo esc_html($settings['submit_button_text']); ?>
                    </button>
                </div>

                <div class="mailerpress-form-message" style="display: none;"></div>
            </form>
        </div>

        <script>
            (function() {
                // Update available fields list in Elementor editor (keep jQuery for Elementor API)
                if (typeof elementor !== 'undefined' && typeof jQuery !== 'undefined') {
                    var updateFieldsList = function() {
                        var widgetId = '<?php echo esc_js($widget_id); ?>';
                        var formFields = [];

                        // Try to get form fields from Elementor panel
                        if (elementor.preview && elementor.preview.getCurrentElement) {
                            var element = elementor.preview.getCurrentElement();
                            if (element && element.model) {
                                formFields = element.model.getSetting('form_fields') || [];
                            }
                        }

                        // Fallback: get from settings if available
                        if (formFields.length === 0 && typeof elementor !== 'undefined' && elementor.config && elementor.config.settings) {
                            var settings = elementor.config.settings;
                            if (settings.form_fields) {
                                formFields = settings.form_fields;
                            }
                        }

                        var fieldsList = [];
                        if (formFields && formFields.length > 0) {
                            formFields.forEach(function(field, index) {
                                if (field && typeof field === 'object') {
                                    var fieldName = field.field_name || ('field_' + index);
                                    var fieldLabel = field.field_label || fieldName;
                                    fieldsList.push('<strong>' + fieldLabel + ':</strong> <code style="background: #fff; padding: 2px 6px; border-radius: 3px; color: #2271b1;">' + fieldName + '</code>');
                                }
                            });
                        }

                        var fieldsHtml = fieldsList.length > 0 ?
                            fieldsList.join('<br>') :
                            '<em><?php echo esc_js(__('No form fields added yet. Add fields in the "Form Fields" section above.', 'mailerpress-pro')); ?></em>';

                        // Use jQuery only for Elementor integration
                        jQuery('.mailerpress-available-fields-list').html(fieldsHtml);
                    };

                    // Update when panel opens
                    elementor.hooks.addAction('panel/open_editor/widget/mailerpress-optin', function(panel, model, view) {
                        setTimeout(updateFieldsList, 100);

                        // Listen for changes to form_fields
                        if (model) {
                            model.on('change:form_fields', function() {
                                setTimeout(updateFieldsList, 100);
                            });
                        }
                    });

                    // Also update on panel refresh
                    if (typeof jQuery !== 'undefined') {
                        jQuery(document).on('elementor:init', function() {
                            setTimeout(updateFieldsList, 500);
                        });
                    }
                }

                // Form submission with vanilla JS and fetch
                document.addEventListener('DOMContentLoaded', function() {
                    var formElement = document.querySelector('.mailerpress-optin-form[data-widget-id="<?php echo esc_js($widget_id); ?>"] form');

                    if (!formElement) {
                        return;
                    }

                    formElement.addEventListener('submit', function(e) {
                        e.preventDefault();

                        var form = this;
                        var message = form.querySelector('.mailerpress-form-message');
                        var button = form.querySelector('.mailerpress-submit-button');
                        var originalButtonText = button ? button.textContent : '<?php echo esc_js($settings['submit_button_text']); ?>';

                        // Hide message and reset classes
                        if (message) {
                            message.style.display = 'none';
                            message.className = 'mailerpress-form-message';
                            message.textContent = '';
                        }

                        // Disable button and show loading state
                        if (button) {
                            button.disabled = true;
                            button.textContent = '<?php echo esc_js(__('Submitting...', 'mailerpress-pro')); ?>';
                        }

                        // Serialize form data
                        var formData = new FormData(form);
                        formData.append('action', 'mailerpress_elementor_submit');

                        // Make fetch request
                        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
                                method: 'POST',
                                body: formData
                            })
                            .then(function(response) {
                                if (!response.ok) {
                                    throw new Error('Network response was not ok');
                                }
                                return response.json();
                            })
                            .then(function(data) {
                                if (button) {
                                    button.disabled = false;
                                    button.textContent = originalButtonText;
                                }

                                if (data.success) {
                                    if (message) {
                                        message.className = 'mailerpress-form-message success';
                                        message.textContent = data.data.message;
                                        message.style.display = 'block';
                                    }
                                    form.reset();

                                    // Check if redirect is enabled
                                    var enableRedirect = <?php echo ($settings['enable_redirect'] === 'yes') ? 'true' : 'false'; ?>;
                                    var redirectPage = '<?php echo esc_js($settings['redirect_page'] ?? ''); ?>';
                                    var redirectDelay = <?php echo floatval($settings['redirect_delay'] ?? 2); ?>;

                                    if (enableRedirect && redirectPage) {
                                        setTimeout(function() {
                                            window.location.href = redirectPage;
                                        }, redirectDelay * 1000);
                                    }
                                } else {
                                    if (message) {
                                        message.className = 'mailerpress-form-message error';
                                        message.textContent = data.data.message || '<?php echo esc_js($settings['error_message']); ?>';
                                        message.style.display = 'block';
                                    }
                                }
                            })
                            .catch(function(error) {
                                if (button) {
                                    button.disabled = false;
                                    button.textContent = originalButtonText;
                                }

                                if (message) {
                                    message.className = 'mailerpress-form-message error';
                                    message.textContent = '<?php echo esc_js($settings['error_message']); ?>';
                                    message.style.display = 'block';
                                }
                            });
                    });
                });
            })();
        </script>

        <style>
            .mailerpress-optin-form {
                max-width: 100%;
            }

            .mailerpress-form-fields-wrapper {
                display: flex;
                flex-wrap: wrap;
                gap: 15px;
            }

            .mailerpress-form-field {
                margin-bottom: 15px;
                flex: 1 1 100%;
            }

            .mailerpress-form-columns-2 .mailerpress-form-field {
                flex: 1 1 calc(50% - 7.5px);
            }

            .mailerpress-form-columns-3 .mailerpress-form-field {
                flex: 1 1 calc(33.333% - 10px);
            }

            .mailerpress-form-field label {
                display: block;
                margin-bottom: 5px;
                font-weight: 500;
            }

            .mailerpress-form-field .required {
                color: #d63638;
            }

            .mailerpress-form-field input,
            .mailerpress-form-field textarea {
                width: 100%;
                padding: 10px;
                border: 1px solid #ddd;
                border-radius: 4px;
                box-sizing: border-box;
            }

            .mailerpress-form-field textarea {
                min-height: 100px;
            }

            .mailerpress-form-submit {
                margin-top: 20px;
                clear: both;
            }

            .mailerpress-submit-button {
                padding: 12px 24px;
                background-color: #2271b1;
                color: #fff;
                border: none;
                border-radius: 4px;
                cursor: pointer;
                font-size: 16px;
            }

            .mailerpress-submit-button:hover {
                background-color: #135e96;
            }

            .mailerpress-submit-button:disabled {
                opacity: 0.6;
                cursor: not-allowed;
            }

            .mailerpress-form-message {
                margin-top: 15px;
                padding: 10px;
                border-radius: 4px;
            }

            .mailerpress-form-message.success {
                background-color: #00a32a;
                color: #fff;
            }

            .mailerpress-form-message.error {
                background-color: #d63638;
                color: #fff;
            }

            @media (max-width: 768px) {

                .mailerpress-form-columns-2 .mailerpress-form-field,
                .mailerpress-form-columns-3 .mailerpress-form-field {
                    flex: 1 1 100%;
                }
            }
        </style>
<?php
    }

    protected function get_mailerpress_lists(): array
    {
        $lists = [];

        if (!function_exists('mailerpress_get_lists')) {
            return $lists;
        }

        $mailerpress_lists = mailerpress_get_lists();

        if (is_array($mailerpress_lists)) {
            foreach ($mailerpress_lists as $list) {
                if (is_object($list) && isset($list->list_id)) {
                    $lists[$list->list_id] = $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id;
                } elseif (is_array($list) && isset($list['list_id'])) {
                    $lists[$list['list_id']] = $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'];
                }
            }
        }

        return $lists;
    }

    protected function get_mailerpress_tags(): array
    {
        $tags = [];

        if (!function_exists('mailerpress_get_tags')) {
            return $tags;
        }

        $mailerpress_tags = mailerpress_get_tags();

        if (is_array($mailerpress_tags)) {
            foreach ($mailerpress_tags as $tag) {
                if (is_object($tag) && isset($tag->tag_id)) {
                    $tags[$tag->tag_id] = $tag->name ?? __('Tag', 'mailerpress-pro') . ' ' . $tag->tag_id;
                } elseif (is_array($tag) && isset($tag['tag_id'])) {
                    $tags[$tag['tag_id']] = $tag['name'] ?? __('Tag', 'mailerpress-pro') . ' ' . $tag['tag_id'];
                }
            }
        }

        return $tags;
    }

    protected function get_mailerpress_fields(): array
    {
        return [
            '' => __('-- Select --', 'mailerpress-pro'),
            'email' => __('Email', 'mailerpress-pro'),
            'contactFirstName' => __('First Name', 'mailerpress-pro'),
            'contactLastName' => __('Last Name', 'mailerpress-pro'),
        ];
    }

    protected function get_form_fields_for_mapping(): array
    {
        $fields = ['' => __('-- Select Form Field --', 'mailerpress-pro')];

        // Don't try to get settings during control registration
        // Elementor will handle dynamic options through JavaScript
        // Return empty list - options will be populated dynamically by Elementor

        return $fields;
    }

    protected function get_custom_fields(): array
    {
        $fields = ['' => __('-- Select --', 'mailerpress-pro')];

        try {
            $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
            $custom_fields = $customFieldsModel->all();

            foreach ($custom_fields as $field) {
                $fields[$field->field_key] = $field->label . ' (' . $field->type . ')';
            }
        } catch (\Exception $e) {
            // Silently fail if container is not available
        }

        return $fields;
    }

    protected function get_pages_list(): array
    {
        $pages = ['' => __('-- Select a page --', 'mailerpress-pro')];

        $wp_pages = get_pages([
            'sort_column' => 'post_title',
            'sort_order' => 'ASC',
        ]);

        foreach ($wp_pages as $page) {
            $pages[get_permalink($page->ID)] = $page->post_title;
        }

        return $pages;
    }
}
