<?php

namespace MailerPressPro\Services\ThirdParty;

use ET_Builder_Module;
use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

/**
 * Wrapper for MailerPress Divi Module.
 *
 * @since   1.0.0
 *
 * @package MailerPressPro\Services\ThirdParty
 */
class MailerPressDiviModule extends ET_Builder_Module
{
    /**
     * @var string Module slug
     */
    public $slug = 'et_pb_mailerpress_optin';

    /**
     * @var string Visual Builder support
     */
    public $vb_support = 'on';

    /**
     * @var string Module folder name
     */
    public $folder_name = 'mailerpress';

    /**
     * @var array Module credits
     */
    public $module_credits = [
        'module_uri' => 'https://mailerpress.com',
        'author' => 'MailerPress',
        'author_uri' => 'https://mailerpress.com',
    ];

    /**
     * @var string Plugin required message
     */
    public static $PLUGIN_REQUIRED;

    /**
     * @var string Module credits owner
     */
    protected $module_credits_owner = 'MailerPress';

    /**
     * Constructor.
     */
    public function __construct()
    {
        parent::__construct();

        if (null === self::$PLUGIN_REQUIRED) {
            self::$PLUGIN_REQUIRED = esc_html__('MailerPress plugin is either not installed or not activated.', 'mailerpress-pro');
        }
    }

    /**
     * Initialize module.
     *
     * @return void
     */
    public function init()
    {
        $this->name = esc_html__('MailerPress Optin Form', 'mailerpress-pro');
        $this->icon_path = $this->get_icon_path();

        $this->settings_modal_toggles = [
            'general' => [
                'toggles' => [
                    'form_fields' => esc_html__('Form Fields', 'mailerpress-pro'),
                    'mailerpress_settings' => esc_html__('MailerPress Settings', 'mailerpress-pro'),
                    'field_mapping' => esc_html__('Field Mapping', 'mailerpress-pro'),
                    'custom_fields' => esc_html__('Custom Fields Mapping', 'mailerpress-pro'),
                    'form_settings' => esc_html__('Form Settings', 'mailerpress-pro'),
                ],
            ],
            'advanced' => [
                'toggles' => [
                    'form_styling' => esc_html__('Form Styling', 'mailerpress-pro'),
                ],
            ],
        ];
    }

    /**
     * Get module icon path.
     *
     * @return string
     */
    public function get_icon_path()
    {
        return plugin_dir_path(__FILE__) . 'icons/mailerpress.svg';
    }

    /**
     * Get module fields configuration.
     *
     * @return array
     */
    public function get_fields()
    {
        $lists_data = $this->fetch_subscriber_lists();
        $tags_data = $this->fetch_subscriber_tags();
        $fields_data = $this->get_default_fields();
        $custom_fields_data = $this->fetch_custom_fields();
        $pages_data = $this->fetch_pages_list();

        return [
            // Form Fields
            'form_fields' => [
                'label' => esc_html__('Form Fields', 'mailerpress-pro'),
                'type' => 'multiple_fields',
                'toggle_slug' => 'form_fields',
                'sub_fields' => [
                    'field_type' => [
                        'label' => esc_html__('Field Type', 'mailerpress-pro'),
                        'type' => 'select',
                        'options' => [
                            'email' => esc_html__('Email', 'mailerpress-pro'),
                            'text' => esc_html__('Text', 'mailerpress-pro'),
                            'textarea' => esc_html__('Textarea', 'mailerpress-pro'),
                            'tel' => esc_html__('Phone', 'mailerpress-pro'),
                            'number' => esc_html__('Number', 'mailerpress-pro'),
                            'date' => esc_html__('Date', 'mailerpress-pro'),
                        ],
                        'default' => 'email',
                    ],
                    'field_label' => [
                        'label' => esc_html__('Field Label', 'mailerpress-pro'),
                        'type' => 'text',
                        'default' => '',
                    ],
                    'field_name' => [
                        'label' => esc_html__('Field Name', 'mailerpress-pro'),
                        'type' => 'text',
                        'default' => '',
                    ],
                    'field_required' => [
                        'label' => esc_html__('Required', 'mailerpress-pro'),
                        'type' => 'yes_no_button',
                        'options' => [
                            'off' => esc_html__('No', 'mailerpress-pro'),
                            'on' => esc_html__('Yes', 'mailerpress-pro'),
                        ],
                        'default' => 'off',
                    ],
                    'field_placeholder' => [
                        'label' => esc_html__('Placeholder', 'mailerpress-pro'),
                        'type' => 'text',
                        'default' => '',
                    ],
                ],
                'default' => json_encode([
                    [
                        'field_type' => 'email',
                        'field_label' => esc_html__('Email', 'mailerpress-pro'),
                        'field_name' => 'email',
                        'field_required' => 'on',
                        'field_placeholder' => esc_html__('Enter your email', 'mailerpress-pro'),
                    ],
                ]),
            ],

            // MailerPress Settings
            'mailerpress_list' => [
                'label' => esc_html__('MailerPress Lists', 'mailerpress-pro'),
                'type' => 'multiple_checkboxes',
                'options' => $lists_data,
                'toggle_slug' => 'mailerpress_settings',
                'description' => esc_html__('Leave empty to automatically use the default list.', 'mailerpress-pro'),
            ],

            'mailerpress_tags' => [
                'label' => esc_html__('MailerPress Tags', 'mailerpress-pro'),
                'type' => 'multiple_checkboxes',
                'options' => $tags_data,
                'toggle_slug' => 'mailerpress_settings',
            ],

            'double_optin' => [
                'label' => esc_html__('Double Opt-In', 'mailerpress-pro'),
                'type' => 'yes_no_button',
                'options' => [
                    'on' => esc_html__('Yes', 'mailerpress-pro'),
                    'off' => esc_html__('No', 'mailerpress-pro'),
                ],
                'default' => 'on',
                'toggle_slug' => 'mailerpress_settings',
                'description' => esc_html__('When enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.', 'mailerpress-pro'),
            ],

            'optin_source' => [
                'label' => esc_html__('Opt-in Source', 'mailerpress-pro'),
                'type' => 'text',
                'default' => '',
                'toggle_slug' => 'mailerpress_settings',
                'description' => esc_html__('Optional: Set a custom opt-in source identifier for tracking where contacts subscribed from. Leave empty to use default value.', 'mailerpress-pro'),
            ],

            // Field Mapping
            'field_mapping' => [
                'label' => esc_html__('Field Mapping', 'mailerpress-pro'),
                'type' => 'multiple_fields',
                'toggle_slug' => 'field_mapping',
                'sub_fields' => [
                    'form_field' => [
                        'label' => esc_html__('Form Field Name', 'mailerpress-pro'),
                        'type' => 'text',
                        'default' => '',
                    ],
                    'mailerpress_field' => [
                        'label' => esc_html__('MailerPress Field', 'mailerpress-pro'),
                        'type' => 'select',
                        'options' => $fields_data,
                        'default' => '',
                    ],
                ],
                'default' => json_encode([]),
                'description' => esc_html__('Map form fields to MailerPress fields. Leave empty to use default mapping (email, contactFirstName, contactLastName).', 'mailerpress-pro'),
            ],

            // Custom Fields Mapping
            'custom_fields_mapping' => [
                'label' => esc_html__('Custom Fields Mapping', 'mailerpress-pro'),
                'type' => 'multiple_fields',
                'toggle_slug' => 'custom_fields',
                'sub_fields' => [
                    'form_field_custom' => [
                        'label' => esc_html__('Form Field Name', 'mailerpress-pro'),
                        'type' => 'text',
                        'default' => '',
                    ],
                    'custom_field_key' => [
                        'label' => esc_html__('Custom Field Key', 'mailerpress-pro'),
                        'type' => 'select',
                        'options' => $custom_fields_data,
                        'default' => '',
                    ],
                ],
                'default' => json_encode([]),
            ],

            // Form Settings
            'form_columns' => [
                'label' => esc_html__('Form Columns', 'mailerpress-pro'),
                'type' => 'select',
                'options' => [
                    '1' => esc_html__('1 Column', 'mailerpress-pro'),
                    '2' => esc_html__('2 Columns', 'mailerpress-pro'),
                    '3' => esc_html__('3 Columns', 'mailerpress-pro'),
                ],
                'default' => '1',
                'toggle_slug' => 'form_settings',
                'description' => esc_html__('Select the number of columns for the form fields layout.', 'mailerpress-pro'),
            ],

            'submit_button_text' => [
                'label' => esc_html__('Submit Button Text', 'mailerpress-pro'),
                'type' => 'text',
                'default' => esc_html__('Subscribe', 'mailerpress-pro'),
                'toggle_slug' => 'form_settings',
            ],

            'success_message' => [
                'label' => esc_html__('Success Message', 'mailerpress-pro'),
                'type' => 'textarea',
                'default' => esc_html__('Thank you for subscribing!', 'mailerpress-pro'),
                'toggle_slug' => 'form_settings',
            ],

            'error_message' => [
                'label' => esc_html__('Error Message', 'mailerpress-pro'),
                'type' => 'textarea',
                'default' => esc_html__('An error occurred. Please try again.', 'mailerpress-pro'),
                'toggle_slug' => 'form_settings',
            ],

            'enable_redirect' => [
                'label' => esc_html__('Enable Redirect After Success', 'mailerpress-pro'),
                'type' => 'yes_no_button',
                'options' => [
                    'off' => esc_html__('No', 'mailerpress-pro'),
                    'on' => esc_html__('Yes', 'mailerpress-pro'),
                ],
                'default' => 'off',
                'toggle_slug' => 'form_settings',
                'description' => esc_html__('Redirect to a page after successful form submission.', 'mailerpress-pro'),
            ],

            'redirect_page' => [
                'label' => esc_html__('Redirect To Page', 'mailerpress-pro'),
                'type' => 'select',
                'options' => $pages_data,
                'default' => '',
                'toggle_slug' => 'form_settings',
                'show_if' => [
                    'enable_redirect' => 'on',
                ],
            ],

            'redirect_delay' => [
                'label' => esc_html__('Redirect Delay (seconds)', 'mailerpress-pro'),
                'type' => 'range',
                'default' => '2',
                'range_settings' => [
                    'min' => 0,
                    'max' => 10,
                    'step' => 0.5,
                ],
                'toggle_slug' => 'form_settings',
                'show_if' => [
                    'enable_redirect' => 'on',
                ],
            ],
        ];
    }

    /**
     * Render module output.
     *
     * @param array  $attrs       Module attributes.
     * @param string $content     Module content.
     * @param string $render_slug Render slug.
     *
     * @return string
     */
    public function render($attrs, $content = null, $render_slug)
    {
        // Check if MailerPress is available
        if (!$this->is_mailerpress_available()) {
            return $this->render_error_message(self::$PLUGIN_REQUIRED);
        }

        $form_fields = $this->get_form_fields();

        if (empty($form_fields)) {
            return '';
        }

        $element_id = $this->generate_element_id();
        $form_class = $this->get_form_class();

        return $this->render_form($element_id, $form_class, $form_fields);
    }

    /**
     * Check if MailerPress is available.
     *
     * @return bool
     */
    protected function is_mailerpress_available()
    {
        return function_exists('mailerpress_get_lists')
            || (defined('MAILERPRESS_VERSION') && class_exists('\MailerPress\Core\Kernel'));
    }

    /**
     * Get form fields from props.
     *
     * @return array
     */
    protected function get_form_fields()
    {
        return !empty($this->props['form_fields'])
            ? json_decode($this->props['form_fields'], true)
            : [];
    }

    /**
     * Generate unique element ID.
     *
     * @return string
     */
    protected function generate_element_id()
    {
        return uniqid('mailerpress-form-');
    }

    /**
     * Get form CSS class based on column setting.
     *
     * @return string
     */
    protected function get_form_class()
    {
        $columns = $this->props['form_columns'] ?? '1';
        return 'mailerpress-form-columns-' . $columns;
    }

    /**
     * Render error message.
     *
     * @param string $message Error message.
     *
     * @return string
     */
    protected function render_error_message($message)
    {
        return sprintf(
            '<div class="mailerpress-error">%s</div>',
            esc_html($message)
        );
    }

    /**
     * Render the complete form.
     *
     * @param string $element_id  Unique element ID.
     * @param string $form_class  Form CSS class.
     * @param array  $form_fields Form fields configuration.
     *
     * @return string
     */
    protected function render_form($element_id, $form_class, $form_fields)
    {
        ob_start();
        ?>
        <div class="mailerpress-optin-form" data-element-id="<?php echo esc_attr($element_id); ?>">
            <?php echo $this->render_form_element($element_id, $form_class, $form_fields); ?>
            <?php echo $this->render_form_script($element_id); ?>
            <?php echo $this->render_form_styles(); ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render form element.
     *
     * @param string $element_id  Unique element ID.
     * @param string $form_class  Form CSS class.
     * @param array  $form_fields Form fields configuration.
     *
     * @return string
     */
    protected function render_form_element($element_id, $form_class, $form_fields)
    {
        $submit_button_text = $this->props['submit_button_text'] ?? esc_html__('Subscribe', 'mailerpress-pro');

        ob_start();
        ?>
        <form class="mailerpress-optin-form-element <?php echo esc_attr($form_class); ?>" method="post" action="">
            <?php wp_nonce_field('mailerpress_optin_submit', 'mailerpress_optin_nonce'); ?>
            <?php echo $this->render_hidden_fields($element_id); ?>
            <?php echo $this->render_form_fields($element_id, $form_fields); ?>
            <?php echo $this->render_submit_button($submit_button_text); ?>
            <div class="mailerpress-form-message" style="display: none;"></div>
        </form>
        <?php
        return ob_get_clean();
    }

    /**
     * Render hidden form fields.
     *
     * @param string $element_id Unique element ID.
     *
     * @return string
     */
    protected function render_hidden_fields($element_id)
    {
        $settings = wp_json_encode($this->props, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);

        ob_start();
        ?>
        <input type="hidden" name="element_id" value="<?php echo esc_attr($element_id); ?>">
        <input type="hidden" name="page_id" value="<?php echo esc_attr(get_the_ID()); ?>">
        <input type="hidden" name="element_settings" value="<?php echo esc_attr($settings); ?>">
        <?php
        return ob_get_clean();
    }

    /**
     * Render form fields.
     *
     * @param string $element_id  Unique element ID.
     * @param array  $form_fields Form fields configuration.
     *
     * @return string
     */
    protected function render_form_fields($element_id, $form_fields)
    {
        ob_start();
        ?>
        <div class="mailerpress-form-fields-wrapper">
            <?php foreach ($form_fields as $index => $field): ?>
                <?php echo $this->render_single_field($element_id, $index, $field); ?>
            <?php endforeach; ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render a single form field.
     *
     * @param string $element_id Unique element ID.
     * @param int    $index      Field index.
     * @param array  $field      Field configuration.
     *
     * @return string
     */
    protected function render_single_field($element_id, $index, $field)
    {
        $field_id = $element_id . '-field-' . $index;
        $field_name = !empty($field['field_name']) ? $field['field_name'] : 'field_' . $index;
        $field_label = $field['field_label'] ?? '';
        $field_type = $field['field_type'] ?? 'text';
        $required = ($field['field_required'] ?? 'off') === 'on';
        $placeholder = $field['field_placeholder'] ?? '';

        ob_start();
        ?>
        <div class="mailerpress-form-field">
            <label for="<?php echo esc_attr($field_id); ?>">
                <?php echo esc_html($field_label); ?>
                <?php if ($required): ?>
                    <span class="required">*</span>
                <?php endif; ?>
            </label>
            <?php echo $this->render_field_input($field_id, $field_name, $field_type, $placeholder, $required); ?>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render field input element.
     *
     * @param string $field_id    Field ID.
     * @param string $field_name  Field name.
     * @param string $field_type  Field type.
     * @param string $placeholder Placeholder text.
     * @param bool   $required    Whether field is required.
     *
     * @return string
     */
    protected function render_field_input($field_id, $field_name, $field_type, $placeholder, $required)
    {
        $required_attr = $required ? 'required' : '';

        switch ($field_type) {
            case 'textarea':
                return sprintf(
                    '<textarea id="%s" name="%s" placeholder="%s" %s></textarea>',
                    esc_attr($field_id),
                    esc_attr($field_name),
                    esc_attr($placeholder),
                    $required_attr
                );

            case 'email':
            case 'text':
            case 'tel':
            case 'number':
            case 'date':
                return sprintf(
                    '<input type="%s" id="%s" name="%s" placeholder="%s" %s>',
                    esc_attr($field_type),
                    esc_attr($field_id),
                    esc_attr($field_name),
                    esc_attr($placeholder),
                    $required_attr
                );

            default:
                return '';
        }
    }

    /**
     * Render submit button.
     *
     * @param string $button_text Button text.
     *
     * @return string
     */
    protected function render_submit_button($button_text)
    {
        ob_start();
        ?>
        <div class="mailerpress-form-submit">
            <button type="submit" class="mailerpress-submit-button et_pb_button">
                <?php echo esc_html($button_text); ?>
            </button>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * Render form JavaScript.
     *
     * @param string $element_id Unique element ID.
     *
     * @return string
     */
    protected function render_form_script($element_id)
    {
        $enable_redirect = ($this->props['enable_redirect'] ?? 'off') === 'on';
        $redirect_page = $this->props['redirect_page'] ?? '';
        $redirect_delay = floatval($this->props['redirect_delay'] ?? '2');
        $error_message = $this->props['error_message'] ?? esc_html__('An error occurred. Please try again.', 'mailerpress-pro');

        ob_start();
        ?>
        <script>
            (function($) {
                $(document).ready(function() {
                    $('.mailerpress-optin-form[data-element-id="<?php echo esc_js($element_id); ?>"] form').on('submit', function(e) {
                        e.preventDefault();
                        var $form = $(this);
                        var $message = $form.find('.mailerpress-form-message');
                        var $button = $form.find('.mailerpress-submit-button');
                        var originalButtonText = $button.text();

                        $message.hide().removeClass('success error').html('');
                        $button.prop('disabled', true).text('<?php echo esc_js(__('Submitting...', 'mailerpress-pro')); ?>');

                        $.ajax({
                            url: '<?php echo esc_url(admin_url('admin-ajax.php')); ?>',
                            type: 'POST',
                            data: $form.serialize() + '&action=mailerpress_divi_submit',
                            success: function(response) {
                                if (response.success) {
                                    $message.addClass('success').html(response.data.message).show();
                                    $form[0].reset();

                                    <?php if ($enable_redirect && $redirect_page): ?>
                                    setTimeout(function() {
                                        window.location.href = '<?php echo esc_js($redirect_page); ?>';
                                    }, <?php echo $redirect_delay * 1000; ?>);
                                    <?php endif; ?>
                                } else {
                                    $message.addClass('error').html(response.data.message || '<?php echo esc_js($error_message); ?>').show();
                                }
                                $button.prop('disabled', false).text(originalButtonText);
                            },
                            error: function() {
                                $message.addClass('error').html('<?php echo esc_js($error_message); ?>').show();
                                $button.prop('disabled', false).text(originalButtonText);
                            }
                        });
                    });
                });
            })(jQuery);
        </script>
        <?php
        return ob_get_clean();
    }

    /**
     * Render form styles.
     *
     * @return string
     */
    protected function render_form_styles()
    {
        ob_start();
        ?>
        <style>
            .mailerpress-optin-form {
                max-width: 100%;
                width: 100%;
            }

            .mailerpress-form-fields-wrapper {
                display: flex;
                flex-wrap: wrap;
                gap: 15px;
                margin-bottom: 15px;
            }

            .mailerpress-form-field {
                flex: 1 1 100%;
            }

            .mailerpress-form-columns-2 .mailerpress-form-field {
                flex: 1 1 calc(50% - 7.5px);
            }

            .mailerpress-form-columns-3 .mailerpress-form-field {
                flex: 1 1 calc(33.333% - 10px);
            }

            .mailerpress-form-field label {
                display: block;
                margin-bottom: 5px;
                font-weight: 500;
            }

            .mailerpress-form-field .required {
                color: #d63638;
            }

            .mailerpress-form-field input,
            .mailerpress-form-field textarea {
                width: 100%;
                padding: 10px;
                border: 1px solid #ddd;
                border-radius: 4px;
                font-size: 14px;
            }

            .mailerpress-form-field textarea {
                min-height: 100px;
                resize: vertical;
            }

            .mailerpress-submit-button {
                margin-top: 10px;
            }

            .mailerpress-form-message {
                margin-top: 15px;
                padding: 12px;
                border-radius: 4px;
            }

            .mailerpress-form-message.success {
                background-color: #00a32a;
                color: #fff;
            }

            .mailerpress-form-message.error {
                background-color: #d63638;
                color: #fff;
            }

            .mailerpress-error {
                padding: 12px;
                background-color: #fff8e5;
                border-left: 4px solid #ffb900;
                color: #333;
            }

            @media (max-width: 768px) {
                .mailerpress-form-columns-2 .mailerpress-form-field,
                .mailerpress-form-columns-3 .mailerpress-form-field {
                    flex: 1 1 100%;
                }
            }
        </style>
        <?php
        return ob_get_clean();
    }

    /**
     * Fetch subscriber lists from MailerPress.
     *
     * @return array|string Array of lists on success, error message on failure.
     */
    protected function fetch_subscriber_lists()
    {
        $lists = ['' => esc_html__('-- Select --', 'mailerpress-pro')];

        if (!function_exists('mailerpress_get_lists')) {
            return $lists;
        }

        $mailerpress_lists = mailerpress_get_lists();

        if (is_array($mailerpress_lists)) {
            foreach ($mailerpress_lists as $list) {
                if (is_object($list) && isset($list->list_id)) {
                    $lists[$list->list_id] = $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id;
                } elseif (is_array($list) && isset($list['list_id'])) {
                    $lists[$list['list_id']] = $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'];
                }
            }
        }

        return $lists;
    }

    /**
     * Fetch subscriber tags from MailerPress.
     *
     * @return array Array of tags.
     */
    protected function fetch_subscriber_tags()
    {
        $tags = ['' => esc_html__('-- Select --', 'mailerpress-pro')];

        if (!function_exists('mailerpress_get_tags')) {
            return $tags;
        }

        $mailerpress_tags = mailerpress_get_tags();

        if (is_array($mailerpress_tags)) {
            foreach ($mailerpress_tags as $tag) {
                if (is_object($tag) && isset($tag->tag_id)) {
                    $tags[$tag->tag_id] = $tag->name ?? __('Tag', 'mailerpress-pro') . ' ' . $tag->tag_id;
                } elseif (is_array($tag) && isset($tag['tag_id'])) {
                    $tags[$tag['tag_id']] = $tag['name'] ?? __('Tag', 'mailerpress-pro') . ' ' . $tag['tag_id'];
                }
            }
        }

        return $tags;
    }

    /**
     * Get default MailerPress fields.
     *
     * @return array
     */
    protected function get_default_fields()
    {
        return [
            '' => esc_html__('-- Select --', 'mailerpress-pro'),
            'email' => esc_html__('Email', 'mailerpress-pro'),
            'contactFirstName' => esc_html__('First Name', 'mailerpress-pro'),
            'contactLastName' => esc_html__('Last Name', 'mailerpress-pro'),
        ];
    }

    /**
     * Fetch custom fields from MailerPress.
     *
     * @return array Array of custom fields.
     */
    protected function fetch_custom_fields()
    {
        $fields = ['' => esc_html__('-- Select --', 'mailerpress-pro')];

        try {
            if (class_exists('\MailerPress\Core\Kernel')) {
                $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
                $custom_fields = $customFieldsModel->all();

                foreach ($custom_fields as $field) {
                    $fields[$field->field_key] = $field->label . ' (' . $field->type . ')';
                }
            }
        } catch (\Exception $e) {
            // Silently fail if container is not available
        }

        return $fields;
    }

    /**
     * Fetch WordPress pages list.
     *
     * @return array Array of pages.
     */
    protected function fetch_pages_list()
    {
        $pages = ['' => esc_html__('-- Select a page --', 'mailerpress-pro')];

        $wp_pages = get_pages([
            'sort_column' => 'post_title',
            'sort_order' => 'ASC',
        ]);

        foreach ($wp_pages as $page) {
            $pages[get_permalink($page->ID)] = $page->post_title;
        }

        return $pages;
    }
}
