<?php

namespace MailerPressPro\Services\ThirdParty;

use Bricks\Element;
use MailerPress\Models\CustomFields;

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly.
}

class MailerPressBricksElement extends Element
{
    public $category = 'general';
    public $name = 'mailerpress-optin';
    public $icon = 'ti-email';
    public $tag = 'div';

    public function get_label()
    {
        return __('MailerPress Optin Block', 'mailerpress-pro');
    }

    public function set_control_groups()
    {
        $this->control_groups['form_fields'] = [
            'title' => __('Form Fields', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['field_style'] = [
            'title' => __('Field Style', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['label_style'] = [
            'title' => __('Label Style', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['button'] = [
            'title' => __('Submit Button', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['mailerpress_settings'] = [
            'title' => __('MailerPress Settings', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['field_mapping'] = [
            'title' => __('Field Mapping', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['custom_fields'] = [
            'title' => __('Custom Fields Mapping', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['form_settings'] = [
            'title' => __('Form Settings', 'mailerpress-pro'),
            'tab' => 'content',
        ];

        $this->control_groups['spam_protection'] = [
            'title' => __('Spam Protection', 'mailerpress-pro'),
            'tab' => 'content',
        ];

    }

    public function set_controls()
    {
        // Form Fields Repeater
        $this->controls['form_fields'] = [
            'tab' => 'content',
            'group' => 'form_fields',
            'label' => __('Form Fields', 'mailerpress-pro'),
            'type' => 'repeater',
            'fields' => [
                'field_type' => [
                    'label' => __('Field Type', 'mailerpress-pro'),
                    'type' => 'select',
                    'options' => [
                        'email' => __('Email', 'mailerpress-pro'),
                        'text' => __('Text', 'mailerpress-pro'),
                        'textarea' => __('Textarea', 'mailerpress-pro'),
                        'tel' => __('Phone', 'mailerpress-pro'),
                        'number' => __('Number', 'mailerpress-pro'),
                        'date' => __('Date', 'mailerpress-pro'),
                    ],
                    'default' => 'email',
                ],
                'field_label' => [
                    'label' => __('Label', 'mailerpress-pro'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Enter field label', 'mailerpress-pro'),
                ],
                'field_name' => [
                    'label' => __('Field Name', 'mailerpress-pro'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Enter field name', 'mailerpress-pro'),
                ],
                'field_required' => [
                    'label' => __('Required', 'mailerpress-pro'),
                    'type' => 'checkbox',
                    'default' => false,
                ],
                'field_placeholder' => [
                    'label' => __('Placeholder', 'mailerpress-pro'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('Enter placeholder text', 'mailerpress-pro'),
                ],
            ],
            'default' => [
                [
                    'field_type' => 'email',
                    'field_label' => __('Email', 'mailerpress-pro'),
                    'field_name' => 'email',
                    'field_required' => true,
                    'field_placeholder' => __('Enter your email', 'mailerpress-pro'),
                ],
            ],
        ];

        // MailerPress Settings
        $mailerpress_lists = $this->get_mailerpress_lists();

        $this->controls['mailerpress_list'] = [
            'tab' => 'content',
            'group' => 'mailerpress_settings',
            'label' => __('MailerPress Lists', 'mailerpress-pro'),
            'type' => 'select',
            'multiple' => true,
            'options' => $mailerpress_lists,
            'default' => [],
            'description' => __('Leave empty to automatically use the default list.', 'mailerpress-pro'),
        ];

        $this->controls['mailerpress_tags'] = [
            'tab' => 'content',
            'group' => 'mailerpress_settings',
            'label' => __('MailerPress Tags', 'mailerpress-pro'),
            'type' => 'select',
            'multiple' => true,
            'options' => $this->get_mailerpress_tags(),
            'default' => [],
        ];

        $this->controls['double_optin'] = [
            'tab' => 'content',
            'group' => 'mailerpress_settings',
            'label' => __('Double Opt-In', 'mailerpress-pro'),
            'type' => 'select',
            'options' => [
                'yes' => __('Yes', 'mailerpress-pro'),
                'no' => __('No', 'mailerpress-pro'),
            ],
            'default' => 'yes',
            'inline' => true,
            'clearable' => false,
            'description' => __('When enabled, MailerPress will send a confirmation email to the user and will only add them to your MailerPress list upon confirmation.', 'mailerpress-pro'),
        ];

        $this->controls['optin_source'] = [
            'tab' => 'content',
            'group' => 'mailerpress_settings',
            'label' => __('Opt-in Source', 'mailerpress-pro'),
            'type' => 'text',
            'default' => '',
            'placeholder' => __('e.g., bricks, website, landing-page', 'mailerpress-pro'),
            'description' => __('Optional: Set a custom opt-in source identifier for tracking where contacts subscribed from. Leave empty to use default value.', 'mailerpress-pro'),
        ];

        // Field Mapping Repeater
        $this->controls['field_mapping'] = [
            'tab' => 'content',
            'group' => 'field_mapping',
            'label' => __('Field Mapping', 'mailerpress-pro'),
            'type' => 'repeater',
            'fields' => [
                'form_field' => [
                    'label' => __('Form Field Name', 'mailerpress-pro'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('e.g., email, firstname, lastname', 'mailerpress-pro'),
                    'description' => __('Enter the field name from the "Form Fields" section above (the "Field Name" value, not the label).', 'mailerpress-pro'),
                ],
                'mailerpress_field' => [
                    'label' => __('MailerPress Field', 'mailerpress-pro'),
                    'type' => 'select',
                    'options' => $this->get_mailerpress_fields(),
                    'default' => '',
                ],
            ],
            'default' => [],
            'description' => __('Map form fields to MailerPress fields. Leave empty to use default mapping (email, contactFirstName, contactLastName).', 'mailerpress-pro'),
        ];

        // Custom Fields Mapping Repeater
        $this->controls['custom_fields_mapping'] = [
            'tab' => 'content',
            'group' => 'custom_fields',
            'label' => __('Custom Fields Mapping', 'mailerpress-pro'),
            'type' => 'repeater',
            'fields' => [
                'form_field_custom' => [
                    'label' => __('Form Field Name', 'mailerpress-pro'),
                    'type' => 'text',
                    'default' => '',
                    'placeholder' => __('e.g., email, firstname, lastname', 'mailerpress-pro'),
                    'description' => __('Enter the field name from the "Form Fields" section above (the "Field Name" value, not the label).', 'mailerpress-pro'),
                ],
                'custom_field_key' => [
                    'label' => __('Custom Field Key', 'mailerpress-pro'),
                    'type' => 'select',
                    'options' => $this->get_custom_fields(),
                    'default' => '',
                ],
            ],
            'default' => [],
        ];

        // Form Settings
        $this->controls['form_columns'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Form Columns', 'mailerpress-pro'),
            'type' => 'select',
            'options' => [
                '1' => __('1 Column', 'mailerpress-pro'),
                '2' => __('2 Columns', 'mailerpress-pro'),
                '3' => __('3 Columns', 'mailerpress-pro'),
            ],
            'default' => '1',
            'description' => __('Select the number of columns for the form fields layout.', 'mailerpress-pro'),
        ];

        $this->controls['success_message'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Success Message', 'mailerpress-pro'),
            'type' => 'textarea',
            'default' => __('Thank you for subscribing!', 'mailerpress-pro'),
        ];

        $this->controls['error_message'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Error Message', 'mailerpress-pro'),
            'type' => 'textarea',
            'default' => __('An error occurred. Please try again.', 'mailerpress-pro'),
        ];

        $this->controls['enable_redirect'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Enable Redirect After Success', 'mailerpress-pro'),
            'type' => 'checkbox',
            'default' => false,
            'description' => __('Redirect to a page after successful form submission.', 'mailerpress-pro'),
        ];

        $this->controls['redirect_page'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Redirect To Page', 'mailerpress-pro'),
            'type' => 'select',
            'options' => $this->get_pages_list(),
            'default' => '',
            'required' => [
                ['enable_redirect', '=', true],
            ],
            'description' => __('Select the page to redirect to after successful submission.', 'mailerpress-pro'),
        ];

        $this->controls['redirect_delay'] = [
            'tab' => 'content',
            'group' => 'form_settings',
            'label' => __('Redirect Delay (milliseconds)', 'mailerpress-pro'),
            'type' => 'number',
            'default' => 2000,
            'min' => 0,
            'max' => 10000,
            'step' => 100,
            'required' => [
                ['enable_redirect', '=', true],
            ],
            'description' => __('Delay before redirecting in milliseconds (0 = immediate).', 'mailerpress-pro'),
        ];

        // Spam Protection Controls
        $recaptcha_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyGoogleRecaptcha', '') : '';
        $recaptcha_secret_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiSecretKeyGoogleRecaptcha', '') : '';
        $hcaptcha_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyHCaptcha', '') : '';
        $hcaptcha_secret_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiSecretKeyHCaptcha', '') : '';
        $turnstile_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyTurnstile', '') : '';
        $turnstile_secret_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiSecretKeyTurnstile', '') : '';

        $this->controls['enableRecaptcha'] = [
            'tab' => 'content',
            'group' => 'spam_protection',
            'label' => __('Google reCAPTCHA v3', 'mailerpress-pro'),
            'type' => 'checkbox',
            'default' => false,
            'description' => empty($recaptcha_site_key) || empty($recaptcha_secret_key)
                ? sprintf(
                    '<span style="color: #d63638;">%s</span> <a href="%s" target="_blank">%s</a>',
                    __('Google reCAPTCHA API key required! Add key in dashboard under:', 'mailerpress-pro'),
                    admin_url('admin.php?page=bricks-settings#tab-api-keys'),
                    __('Bricks > Settings > API keys', 'mailerpress-pro')
                )
                : __('Enable Google reCAPTCHA v3 (invisible) for spam protection.', 'mailerpress-pro'),
        ];

        $this->controls['enableTurnstile'] = [
            'tab' => 'content',
            'group' => 'spam_protection',
            'label' => __('Cloudflare Turnstile', 'mailerpress-pro'),
            'type' => 'checkbox',
            'default' => false,
            'description' => empty($turnstile_site_key) || empty($turnstile_secret_key)
                ? sprintf(
                    '<span style="color: #d63638;">%s</span> <a href="%s" target="_blank">%s</a>',
                    __('Cloudflare Turnstile API key required! Add key in dashboard under:', 'mailerpress-pro'),
                    admin_url('admin.php?page=bricks-settings'),
                    __('Bricks > Settings', 'mailerpress-pro')
                )
                : __('Enable Cloudflare Turnstile for spam protection.', 'mailerpress-pro'),
        ];

        $this->controls['enableHCaptcha'] = [
            'tab' => 'content',
            'group' => 'spam_protection',
            'label' => __('hCaptcha', 'mailerpress-pro'),
            'type' => 'checkbox',
            'default' => false,
            'description' => empty($hcaptcha_site_key) || empty($hcaptcha_secret_key)
                ? sprintf(
                    '<span style="color: #d63638;">%s</span> <a href="%s" target="_blank">%s</a>',
                    __('hCaptcha key required! Add key in dashboard under:', 'mailerpress-pro'),
                    admin_url('admin.php?page=bricks-settings#tab-api-keys'),
                    __('Bricks > Settings > API keys', 'mailerpress-pro')
                )
                : __('Enable hCaptcha for spam protection.', 'mailerpress-pro'),
        ];

        // Field Style Controls
        $this->controls['fieldPadding'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Padding', 'mailerpress-pro'),
            'type' => 'spacing',
            'css' => [
                [
                    'property' => 'padding',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'padding',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldMargin'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Margin', 'mailerpress-pro'),
            'type' => 'spacing',
            'css' => [
                [
                    'property' => 'margin',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'margin',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldHeight'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Height', 'mailerpress-pro'),
            'type' => 'number',
            'units' => true,
            'css' => [
                [
                    'property' => 'height',
                    'selector' => '.mailerpress-form-field input:not([type="checkbox"]):not([type="radio"])',
                ],
            ],
        ];

        $this->controls['fieldTextAlign'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Text Align', 'mailerpress-pro'),
            'type' => 'text-align',
            'css' => [
                [
                    'property' => 'text-align',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'text-align',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldBackground'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Background', 'mailerpress-pro'),
            'type' => 'color',
            'css' => [
                [
                    'property' => 'background-color',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'background-color',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldColor'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Text Color', 'mailerpress-pro'),
            'type' => 'color',
            'css' => [
                [
                    'property' => 'color',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'color',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldBorder'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Border', 'mailerpress-pro'),
            'type' => 'border',
            'css' => [
                [
                    'property' => 'border',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'border',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldBoxShadow'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Box Shadow', 'mailerpress-pro'),
            'type' => 'box-shadow',
            'css' => [
                [
                    'property' => 'box-shadow',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'box-shadow',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        $this->controls['fieldTypography'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Typography', 'mailerpress-pro'),
            'type' => 'typography',
            'css' => [
                [
                    'property' => 'font',
                    'selector' => '.mailerpress-form-field input',
                ],
                [
                    'property' => 'font',
                    'selector' => '.mailerpress-form-field textarea',
                ],
            ],
        ];

        // Focus State
        $this->controls['fieldFocusBorderColor'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Focus Border Color', 'mailerpress-pro'),
            'type' => 'color',
            'css' => [
                [
                    'property' => 'border-color',
                    'selector' => '.mailerpress-form-field input:focus',
                ],
                [
                    'property' => 'border-color',
                    'selector' => '.mailerpress-form-field textarea:focus',
                ],
            ],
        ];

        $this->controls['fieldFocusBoxShadow'] = [
            'tab' => 'content',
            'group' => 'field_style',
            'label' => __('Focus Box Shadow', 'mailerpress-pro'),
            'type' => 'box-shadow',
            'css' => [
                [
                    'property' => 'box-shadow',
                    'selector' => '.mailerpress-form-field input:focus',
                ],
                [
                    'property' => 'box-shadow',
                    'selector' => '.mailerpress-form-field textarea:focus',
                ],
            ],
        ];

        // Label Style Controls
        $this->controls['labelMargin'] = [
            'tab' => 'content',
            'group' => 'label_style',
            'label' => __('Margin', 'mailerpress-pro'),
            'type' => 'spacing',
            'css' => [
                [
                    'property' => 'margin',
                    'selector' => '.mailerpress-form-field label',
                ],
            ],
        ];

        $this->controls['labelColor'] = [
            'tab' => 'content',
            'group' => 'label_style',
            'label' => __('Color', 'mailerpress-pro'),
            'type' => 'color',
            'css' => [
                [
                    'property' => 'color',
                    'selector' => '.mailerpress-form-field label',
                ],
            ],
        ];

        $this->controls['labelTypography'] = [
            'tab' => 'content',
            'group' => 'label_style',
            'label' => __('Typography', 'mailerpress-pro'),
            'type' => 'typography',
            'css' => [
                [
                    'property' => 'font',
                    'selector' => '.mailerpress-form-field label',
                ],
            ],
        ];

        $this->controls['labelRequiredColor'] = [
            'tab' => 'content',
            'group' => 'label_style',
            'label' => __('Required Star Color', 'mailerpress-pro'),
            'type' => 'color',
            'css' => [
                [
                    'property' => 'color',
                    'selector' => '.mailerpress-form-field .required',
                ],
            ],
        ];

        // Button Controls
        $this->controls['buttonText'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Text', 'mailerpress-pro'),
            'type' => 'text',
            'default' => __('Subscribe', 'mailerpress-pro'),
        ];

        $this->controls['buttonSize'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Size', 'mailerpress-pro'),
            'type' => 'select',
            'options' => [
                'sm' => __('Small', 'mailerpress-pro'),
                'md' => __('Medium', 'mailerpress-pro'),
                'lg' => __('Large', 'mailerpress-pro'),
                'xl' => __('Extra Large', 'mailerpress-pro'),
            ],
            'default' => 'md',
            'inline' => true,
            'rerender' => true,
        ];

        $this->controls['buttonStyle'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Style', 'mailerpress-pro'),
            'type' => 'select',
            'options' => [
                'primary' => __('Primary', 'mailerpress-pro'),
                'secondary' => __('Secondary', 'mailerpress-pro'),
                'light' => __('Light', 'mailerpress-pro'),
                'dark' => __('Dark', 'mailerpress-pro'),
                'muted' => __('Muted', 'mailerpress-pro'),
                'info' => __('Info', 'mailerpress-pro'),
                'success' => __('Success', 'mailerpress-pro'),
                'warning' => __('Warning', 'mailerpress-pro'),
                'danger' => __('Danger', 'mailerpress-pro'),
                'outline-primary' => __('Outline Primary', 'mailerpress-pro'),
                'outline-secondary' => __('Outline Secondary', 'mailerpress-pro'),
                'custom' => __('Custom', 'mailerpress-pro'),
            ],
            'default' => 'primary',
            'inline' => true,
            'rerender' => true,
        ];

        $this->controls['buttonAlign'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Alignment', 'mailerpress-pro'),
            'type' => 'text-align',
            'css' => [
                [
                    'property' => 'text-align',
                    'selector' => '.mailerpress-form-submit',
                ],
            ],
        ];

        $this->controls['buttonWidth'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Width', 'mailerpress-pro'),
            'type' => 'number',
            'units' => true,
            'css' => [
                [
                    'property' => 'width',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
        ];

        $this->controls['buttonMargin'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Margin', 'mailerpress-pro'),
            'type' => 'spacing',
            'css' => [
                [
                    'property' => 'margin',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
            'default' => [
                'top' => '10px',
            ],
        ];

        $this->controls['buttonTypography'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Typography', 'mailerpress-pro'),
            'type' => 'typography',
            'css' => [
                [
                    'property' => 'font',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
        ];

        $this->controls['buttonBackground'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Background', 'mailerpress-pro'),
            'type' => 'background',
            'css' => [
                [
                    'property' => 'background',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
            'required' => [
                ['buttonStyle', '=', 'custom'],
            ],
        ];

        $this->controls['buttonBorder'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Border', 'mailerpress-pro'),
            'type' => 'border',
            'css' => [
                [
                    'property' => 'border',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
        ];

        $this->controls['buttonBoxShadow'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Box Shadow', 'mailerpress-pro'),
            'type' => 'box-shadow',
            'css' => [
                [
                    'property' => 'box-shadow',
                    'selector' => '.mailerpress-submit-button',
                ],
            ],
        ];

        $this->controls['buttonIcon'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Icon', 'mailerpress-pro'),
            'type' => 'icon',
            'rerender' => true,
        ];

        $this->controls['buttonIconPosition'] = [
            'tab' => 'content',
            'group' => 'button',
            'label' => __('Icon Position', 'mailerpress-pro'),
            'type' => 'select',
            'options' => [
                'left' => __('Left', 'mailerpress-pro'),
                'right' => __('Right', 'mailerpress-pro'),
            ],
            'default' => 'left',
            'inline' => true,
            'required' => [
                ['buttonIcon', '!=', ''],
            ],
        ];
    }

    public function render()
    {
        $settings = $this->settings;
        $element_id = $this->id;

        if (empty($settings['form_fields'])) {
            return;
        }

        $columns = isset($settings['form_columns']) ? intval($settings['form_columns']) : 1;
        $form_class = 'mailerpress-form-columns-' . $columns;

        // Set root attributes for Bricks wrapper
        $root_classes = ['mailerpress-optin-form'];
        $this->set_attribute('_root', 'class', $root_classes);
        $this->set_attribute('_root', 'data-element-id', $element_id);

        // Set button attributes
        $button_classes = ['mailerpress-submit-button', 'bricks-button'];

        // Button style
        $button_style = isset($settings['buttonStyle']) ? $settings['buttonStyle'] : 'primary';
        if ($button_style !== 'custom') {
            $button_classes[] = 'bricks-background-' . $button_style;
        }

        // Button size (Bricks uses direct class names: sm, lg, xl - no class for md as it's default)
        $button_size = isset($settings['buttonSize']) ? $settings['buttonSize'] : 'md';
        if ($button_size && $button_size !== 'md') {
            $button_classes[] = $button_size;
        }

        $this->set_attribute('button', 'type', 'submit');
        $this->set_attribute('button', 'class', $button_classes);

        // Get button text
        $button_text = isset($settings['buttonText']) ? $settings['buttonText'] : (isset($settings['submit_button_text']) ? $settings['submit_button_text'] : __('Subscribe', 'mailerpress-pro'));

        // Get Bricks API keys for spam protection
        $recaptcha_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyGoogleRecaptcha', '') : '';
        $hcaptcha_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyHCaptcha', '') : '';
        $turnstile_site_key = class_exists('\Bricks\Database') ? \Bricks\Database::get_setting('apiKeyTurnstile', '') : '';

        // Check which spam protection methods are enabled
        $enable_recaptcha = !empty($settings['enableRecaptcha']) && !empty($recaptcha_site_key);
        $enable_hcaptcha = !empty($settings['enableHCaptcha']) && !empty($hcaptcha_site_key);
        $enable_turnstile = !empty($settings['enableTurnstile']) && !empty($turnstile_site_key);

        // Render root wrapper with Bricks attributes
        echo "<div {$this->render_attributes('_root')}>";
?>
        <form class="mailerpress-optin-form-element <?php echo esc_attr($form_class); ?>" method="post" action="">
            <?php wp_nonce_field('mailerpress_optin_submit', 'mailerpress_optin_nonce'); ?>
            <input type="hidden" name="element_id" value="<?php echo esc_attr($element_id); ?>">
            <input type="hidden" name="page_id" value="<?php echo esc_attr(get_the_ID()); ?>">
            <input type="hidden" name="element_settings" value="<?php echo esc_attr(wp_json_encode($settings, JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT)); ?>">

            <?php // Honeypot field (from global MailerPress settings) ?>
            <?php if (class_exists('\MailerPress\Services\RateLimitConfig') && \MailerPress\Services\RateLimitConfig::isHoneypotEnabled()): ?>
                <input type="text" name="website" value="" style="position: absolute; left: -9999px; width: 1px; height: 1px; opacity: 0; pointer-events: none; tabindex: -1;" autocomplete="off" aria-hidden="true">
            <?php endif; ?>

            <?php // reCAPTCHA v3 hidden input for token ?>
            <?php if ($enable_recaptcha): ?>
                <input type="hidden" name="recaptchaToken" value="">
            <?php endif; ?>

            <div class="mailerpress-form-fields-wrapper">
                <?php foreach ($settings['form_fields'] as $index => $field): ?>
                    <div class="mailerpress-form-field">
                        <label for="mailerpress-field-<?php echo esc_attr($element_id . '-' . $index); ?>">
                            <?php echo esc_html($field['field_label'] ?? ''); ?>
                            <?php if (!empty($field['field_required'])): ?>
                                <span class="required">*</span>
                            <?php endif; ?>
                        </label>
                        <?php
                        $field_name = !empty($field['field_name']) ? $field['field_name'] : 'field_' . $index;
                        $field_id = 'mailerpress-field-' . $element_id . '-' . $index;
                        $required = !empty($field['field_required']) ? 'required' : '';
                        $placeholder = !empty($field['field_placeholder']) ? $field['field_placeholder'] : '';

                        switch ($field['field_type'] ?? 'text') {
                            case 'textarea':
                                echo '<textarea id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" placeholder="' . esc_attr($placeholder) . '" ' . $required . '></textarea>';
                                break;
                            case 'email':
                            case 'text':
                            case 'tel':
                            case 'number':
                            case 'date':
                                echo '<input type="' . esc_attr($field['field_type']) . '" id="' . esc_attr($field_id) . '" name="' . esc_attr($field_name) . '" placeholder="' . esc_attr($placeholder) . '" ' . $required . '>';
                                break;
                        }
                        ?>
                    </div>
                <?php endforeach; ?>
            </div>

            <?php // hCaptcha widget ?>
            <?php if ($enable_hcaptcha): ?>
                <div class="mailerpress-hcaptcha-wrapper" style="margin: 10px 0;">
                    <div class="h-captcha" data-sitekey="<?php echo esc_attr($hcaptcha_site_key); ?>"></div>
                </div>
            <?php endif; ?>

            <?php // Cloudflare Turnstile widget ?>
            <?php if ($enable_turnstile): ?>
                <div class="mailerpress-turnstile-wrapper" style="margin: 10px 0;">
                    <div class="cf-turnstile" data-sitekey="<?php echo esc_attr($turnstile_site_key); ?>"></div>
                </div>
            <?php endif; ?>

            <div class="mailerpress-form-submit">
                <button <?php echo $this->render_attributes('button'); ?>>
                    <?php
                    // DEBUG: Display icon data (remove after testing)
                    if (current_user_can('manage_options')) {
                        echo '<!-- Button Icon Data: ' . esc_html(print_r($settings['buttonIcon'] ?? 'empty', true)) . ' -->';
                    }

                    // Check if icon exists (Bricks stores icons as arrays)
                    $has_icon = !empty($settings['buttonIcon']) && is_array($settings['buttonIcon']);
                    $icon_position = !empty($settings['buttonIconPosition']) ? $settings['buttonIconPosition'] : 'left';

                    // Button icon (left)
                    if ($has_icon && $icon_position === 'left') {
                        $icon_html = self::render_icon($settings['buttonIcon'], ['icon']);
                        if ($icon_html) {
                            echo '<span class="bricks-button-icon">' . $icon_html . '</span>';
                        }
                    }

                    // Button text
                    echo '<span class="bricks-button-text">' . esc_html($button_text) . '</span>';

                    // Button icon (right)
                    if ($has_icon && $icon_position === 'right') {
                        $icon_html = self::render_icon($settings['buttonIcon'], ['icon']);
                        if ($icon_html) {
                            echo '<span class="bricks-button-icon">' . $icon_html . '</span>';
                        }
                    }
                    ?>
                </button>
            </div>

            <div class="mailerpress-form-message" style="display: none;"></div>
        </form>

        <?php // Load captcha scripts ?>
        <?php if ($enable_recaptcha): ?>
            <script src="https://www.google.com/recaptcha/api.js?render=<?php echo esc_attr($recaptcha_site_key); ?>"></script>
        <?php endif; ?>
        <?php if ($enable_hcaptcha): ?>
            <script src="https://js.hcaptcha.com/1/api.js" async defer></script>
        <?php endif; ?>
        <?php if ($enable_turnstile): ?>
            <script src="https://challenges.cloudflare.com/turnstile/v0/api.js" async defer></script>
        <?php endif; ?>

        <script>
            (function() {
                document.addEventListener('DOMContentLoaded', function() {
                    var formElement = document.querySelector('.mailerpress-optin-form[data-element-id="<?php echo esc_js($element_id); ?>"] form');

                    if (!formElement) {
                        return;
                    }

                    <?php if ($enable_recaptcha): ?>
                    // reCAPTCHA v3 configuration
                    var recaptchaSiteKey = '<?php echo esc_js($recaptcha_site_key); ?>';
                    <?php endif; ?>

                    formElement.addEventListener('submit', function(e) {
                        e.preventDefault();

                        var form = this;
                        var message = form.querySelector('.mailerpress-form-message');
                        var button = form.querySelector('.mailerpress-submit-button');
                        var originalButtonText = button ? button.textContent : '<?php echo esc_js($button_text); ?>';

                        <?php if ($enable_recaptcha): ?>
                        // Get reCAPTCHA token before submitting
                        if (typeof grecaptcha !== 'undefined') {
                            grecaptcha.ready(function() {
                                grecaptcha.execute(recaptchaSiteKey, {action: 'mailerpress_form_submit'}).then(function(token) {
                                    var recaptchaInput = form.querySelector('input[name="recaptchaToken"]');
                                    if (recaptchaInput) {
                                        recaptchaInput.value = token;
                                    }
                                    submitForm(form, message, button, originalButtonText);
                                });
                            });
                            return;
                        }
                        <?php endif; ?>

                        submitForm(form, message, button, originalButtonText);
                    });

                    function submitForm(form, message, button, originalButtonText) {

                        // Hide message and reset classes
                        if (message) {
                            message.style.display = 'none';
                            message.className = 'mailerpress-form-message';
                            message.textContent = '';
                        }

                        // Disable button and show loading state
                        if (button) {
                            button.disabled = true;
                            button.textContent = '<?php echo esc_js(__('Submitting...', 'mailerpress-pro')); ?>';
                        }

                        // Serialize form data
                        var formData = new FormData(form);
                        formData.append('action', 'mailerpress_bricks_submit');

                        // Make fetch request
                        fetch('<?php echo esc_url(admin_url('admin-ajax.php')); ?>', {
                                method: 'POST',
                                body: formData
                            })
                            .then(function(response) {
                                if (!response.ok) {
                                    throw new Error('Network response was not ok');
                                }
                                return response.json();
                            })
                            .then(function(data) {
                                if (button) {
                                    button.disabled = false;
                                    button.textContent = originalButtonText;
                                }

                                if (data.success) {
                                    if (message) {
                                        message.className = 'mailerpress-form-message success';
                                        message.textContent = data.data.message;
                                        message.style.display = 'block';
                                    }
                                    form.reset();

                                    // Check if redirect is enabled
                                    var enableRedirect = <?php echo (!empty($settings['enable_redirect'])) ? 'true' : 'false'; ?>;
                                    var redirectPage = '<?php echo esc_js($settings['redirect_page'] ?? ''); ?>';
                                    var redirectDelay = <?php echo intval($settings['redirect_delay'] ?? 2000); ?>;

                                    if (enableRedirect && redirectPage) {
                                        setTimeout(function() {
                                            window.location.href = redirectPage;
                                        }, redirectDelay);
                                    }
                                } else {
                                    if (message) {
                                        message.className = 'mailerpress-form-message error';
                                        message.textContent = data.data.message || '<?php echo esc_js($settings['error_message'] ?? __('An error occurred. Please try again.', 'mailerpress-pro')); ?>';
                                        message.style.display = 'block';
                                    }
                                }
                            })
                            .catch(function(error) {
                                if (button) {
                                    button.disabled = false;
                                    button.textContent = originalButtonText;
                                }

                                if (message) {
                                    message.className = 'mailerpress-form-message error';
                                    message.textContent = '<?php echo esc_js($settings['error_message'] ?? __('An error occurred. Please try again.', 'mailerpress-pro')); ?>';
                                    message.style.display = 'block';
                                }
                            });
                    }
                });
            })();
        </script>

        <style>
            .mailerpress-optin-form {
                max-width: 100%;
                width: 100%;
            }

            .mailerpress-form-fields-wrapper {
                display: flex;
                flex-wrap: wrap;
                gap: 8px;
            }

            .mailerpress-form-field {
                flex: 1 1 100%;
            }

            .mailerpress-form-columns-2 .mailerpress-form-field {
                flex: 1 1 calc(50% - 7.5px);
            }

            .mailerpress-form-columns-3 .mailerpress-form-field {
                flex: 1 1 calc(33.333% - 10px);
            }

            .mailerpress-form-field label {
                display: block;
            }

            .mailerpress-form-field input,
            .mailerpress-form-field textarea {
                width: 100%;
                box-sizing: border-box;
            }

            .mailerpress-form-field textarea {
                min-height: 100px;
                resize: vertical;
            }

            .mailerpress-submit-button {
                display: inline-flex;
                align-items: center;
                gap: 8px;
                margin-top: 10px;
            }

            .mailerpress-submit-button .bricks-button-icon {
                display: inline-flex;
                align-items: center;
            }

            .mailerpress-form-message {
                margin-top: 15px;
                padding: 10px;
                border-radius: 4px;
            }

            .mailerpress-form-message.success {
                background-color: #00a32a;
                color: #fff;
            }

            .mailerpress-form-message.error {
                background-color: #d63638;
                color: #fff;
            }

            @media (max-width: 768px) {

                .mailerpress-form-columns-2 .mailerpress-form-field,
                .mailerpress-form-columns-3 .mailerpress-form-field {
                    flex: 1 1 100%;
                }
            }
        </style>
        </div>
<?php
    }

    protected function get_mailerpress_lists(): array
    {
        $lists = [];

        if (!function_exists('mailerpress_get_lists')) {
            return $lists;
        }

        $mailerpress_lists = mailerpress_get_lists();

        if (is_array($mailerpress_lists)) {
            foreach ($mailerpress_lists as $list) {
                if (is_object($list) && isset($list->list_id)) {
                    $lists[$list->list_id] = $list->name ?? __('List', 'mailerpress-pro') . ' ' . $list->list_id;
                } elseif (is_array($list) && isset($list['list_id'])) {
                    $lists[$list['list_id']] = $list['name'] ?? __('List', 'mailerpress-pro') . ' ' . $list['list_id'];
                }
            }
        }

        return $lists;
    }

    protected function get_mailerpress_tags(): array
    {
        $tags = [];

        if (!function_exists('mailerpress_get_tags')) {
            return $tags;
        }

        $mailerpress_tags = mailerpress_get_tags();

        if (is_array($mailerpress_tags)) {
            foreach ($mailerpress_tags as $tag) {
                if (is_object($tag) && isset($tag->tag_id)) {
                    $tags[$tag->tag_id] = $tag->name ?? __('Tag', 'mailerpress-pro') . ' ' . $tag->tag_id;
                } elseif (is_array($tag) && isset($tag['tag_id'])) {
                    $tags[$tag['tag_id']] = $tag['name'] ?? __('Tag', 'mailerpress-pro') . ' ' . $tag['tag_id'];
                }
            }
        }

        return $tags;
    }

    protected function get_mailerpress_fields(): array
    {
        return [
            '' => __('-- Select --', 'mailerpress-pro'),
            'email' => __('Email', 'mailerpress-pro'),
            'contactFirstName' => __('First Name', 'mailerpress-pro'),
            'contactLastName' => __('Last Name', 'mailerpress-pro'),
        ];
    }

    protected function get_custom_fields(): array
    {
        $fields = ['' => __('-- Select --', 'mailerpress-pro')];

        try {
            $customFieldsModel = \MailerPress\Core\Kernel::getContainer()->get(CustomFields::class);
            $custom_fields = $customFieldsModel->all();

            foreach ($custom_fields as $field) {
                $fields[$field->field_key] = $field->label . ' (' . $field->type . ')';
            }
        } catch (\Exception $e) {
            // Silently fail if container is not available
        }

        return $fields;
    }

    protected function get_pages_list(): array
    {
        $pages = ['' => __('-- Select a page --', 'mailerpress-pro')];

        $wp_pages = get_pages([
            'sort_column' => 'post_title',
            'sort_order' => 'ASC',
        ]);

        foreach ($wp_pages as $page) {
            $pages[get_permalink($page->ID)] = $page->post_title;
        }

        return $pages;
    }
}
