<?php

declare(strict_types=1);

namespace MailerPressPro\Services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;

class Activation
{
    private const FREE_PLUGIN_SLUG = 'mailerpress/mailerpress.php';
    private const FREE_PLUGIN_NAME = 'MailerPress';
    private const FREE_PLUGIN_URL = 'https://wordpress.org/plugins/mailerpress/';

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function activate(): void
    {
        // Check if free plugin is active
        if (!$this->isFreePluginActive()) {
            // Deactivate pro plugin
            deactivate_plugins(plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE));

            // Store notice in transient to show after redirect
            set_transient('mailerpress_pro_activation_notice', 'missing_free_plugin', 60);

            // Prevent "Plugin activated" message
            if (isset($_GET['activate'])) {
                unset($_GET['activate']);
            }

            return;
        }

        // Free plugin is active, proceed with activation
        update_option('mailerpress_pro_activated', 'yes');
    }

    /**
     * Check if the free plugin is installed and active
     */
    private function isFreePluginActive(): bool
    {
        include_once ABSPATH . 'wp-admin/includes/plugin.php';
        return is_plugin_active(self::FREE_PLUGIN_SLUG);
    }

    /**
     * Check on every admin page if pro is active without free plugin
     * If so, deactivate it
     */
    public static function checkDependency(): void
    {
        include_once ABSPATH . 'wp-admin/includes/plugin.php';

        if (!is_plugin_active(self::FREE_PLUGIN_SLUG)) {
            if (is_plugin_active(plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE))) {
                deactivate_plugins(plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE));

                // Prevent "Plugin activated" message
                if (isset($_GET['activate'])) {
                    unset($_GET['activate']);
                }
            }
        }
    }

    /**
     * Enqueue thickbox for plugin install modal
     */
    public static function enqueueScripts(): void
    {
        $screen = get_current_screen();
        if ($screen && $screen->id === 'plugins') {
            add_thickbox();
        }
    }

    /**
     * Inject inline notice in plugin description using JavaScript
     */
    public static function injectInlineNotice(): void
    {
        if (!is_plugin_active(self::FREE_PLUGIN_SLUG)) {
            $screen = get_current_screen();
            if ($screen && $screen->id === 'plugins') {
                // URL to open plugin install modal
                $plugin_slug = 'mailerpress';
                $install_url = add_query_arg(
                    [
                        'tab' => 'plugin-information',
                        'plugin' => $plugin_slug,
                        'TB_iframe' => 'true',
                        'width' => '600',
                        'height' => '550',
                    ],
                    admin_url('plugin-install.php')
                );

                $plugin_basename = plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE);
                $notice_html = sprintf(
                    '<div style="margin-top: 1em;">
                        <p style="margin: 0 0 0.5em 0;">
                            <strong>%s</strong> <a href="%s" class="thickbox open-plugin-details-modal">%s</a>
                        </p>
                        <div class="notice notice-error inline notice-alt" style="margin: 0; padding: 12px;">
                            <p style="margin: 0.5em 0;">%s</p>
                        </div>
                    </div>',
                    esc_html__('Requires:', 'mailerpress-pro'),
                    esc_url($install_url),
                    esc_html(self::FREE_PLUGIN_NAME),
                    esc_html__('This plugin cannot be activated because required plugins are missing or inactive.', 'mailerpress-pro')
                );

                ?>
                <script>
                jQuery(document).ready(function($) {
                    var pluginRow = $('tr[data-slug="<?php echo esc_js(dirname($plugin_basename)); ?>"]');
                    if (pluginRow.length) {
                        var descColumn = pluginRow.find('.plugin-description');
                        if (descColumn.length) {
                            descColumn.append(<?php echo wp_json_encode($notice_html); ?>);
                        }
                    }
                });
                </script>
                <?php
            }
        }
    }

    /**
     * Add a "Requires" row in the plugin list table
     */
    public static function addPluginRowMeta(array $plugin_meta, string $plugin_file): array
    {
        if (plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE) === $plugin_file) {
            if (!is_plugin_active(self::FREE_PLUGIN_SLUG)) {
                $required_plugin_link = sprintf(
                    '<a href="%s" target="_blank">%s</a>',
                    esc_url(self::FREE_PLUGIN_URL),
                    esc_html(self::FREE_PLUGIN_NAME)
                );

                $plugin_meta[] = sprintf(
                    /* translators: %s: Link to required plugin */
                    esc_html__('Requires: %s', 'mailerpress-pro'),
                    $required_plugin_link
                );
            }
        }

        return $plugin_meta;
    }

    /**
     * Remove activate link if free plugin is missing
     */
    public static function addPluginActionLinks(array $actions, string $plugin_file): array
    {
        if (plugin_basename(MAILERPRESS_PRO_PLUGIN_FILE) === $plugin_file) {
            if (!is_plugin_active(self::FREE_PLUGIN_SLUG)) {
                // Remove activate link
                unset($actions['activate']);
            }
        }

        return $actions;
    }
}
