<?php

namespace MailerPressPro\Services\AI;

use Exception;
use MailerPressPro\Abstract\AbstractAIService;
use MailerPressPro\Interfaces\AiModelInterface;

class DeepSeekService extends AbstractAIService implements AiModelInterface
{
    public function generate(string $prompt, array $options = []): string
    {
        $body = $this->createRequestBody($prompt, $options);

        $response = wp_remote_post('https://api.deepseek.com/v1/chat/completions', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            throw new Exception('DeepSeek Request Error: ' . $response->get_error_message());
        }

        $status_code = wp_remote_retrieve_response_code($response);
        $response_body = wp_remote_retrieve_body($response);

        $data = json_decode($response_body, true);

        // Debugging: log unexpected responses
        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('DeepSeek Response JSON decode error: ' . json_last_error_msg());
        }

        // Validate HTTP status
        if ($status_code < 200 || $status_code >= 300) {
            throw new Exception('DeepSeek API returned HTTP ' . $status_code . ': ' . $response_body);
        }

        // Validate expected data structure
        if (!isset($data['choices'][0]['message']['content'])) {
            throw new Exception('DeepSeek API returned unexpected structure: ' . $response_body);
        }

        return trim($data['choices'][0]['message']['content']);
    }

    public function suggest(string $prompt, array $options = [], int $limit = 3): array
    {
        if (empty($prompt)) {
            return [];
        }

        // Customize prompt for short suggestions
        $body = [
            'model' => $options['text_ai']['model'] ?? 'default',
            'messages' => [
                [
                    'role'    => 'user',
                    'content' => $prompt . "\nSuggest $limit possible next words as a JSON array."
                ]
            ],
            'max_tokens' => 20, // only a few tokens
        ];

        $response = wp_remote_post('https://api.deepseek.com/v1/chat/completions', [
            'headers' => [
                'Authorization' => 'Bearer ' . $this->api_key,
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            throw new Exception('DeepSeek Request Error: ' . $response->get_error_message());
        }

        $response_body = wp_remote_retrieve_body($response);
        $data = json_decode($response_body, true);

        if (json_last_error() !== JSON_ERROR_NONE) {
            throw new Exception('DeepSeek Response JSON decode error: ' . json_last_error_msg());
        }

        if (!isset($data['choices'][0]['message']['content'])) {
            throw new Exception('DeepSeek API returned unexpected structure: ' . $response_body);
        }

        $suggestionsText = trim($data['choices'][0]['message']['content']);
        $suggestions = json_decode($suggestionsText, true);

        if (!is_array($suggestions)) {
            // fallback: split by whitespace
            $suggestions = preg_split('/\s+/', $suggestionsText);
        }

        return array_slice($suggestions, 0, $limit);
    }

}
