<?php

namespace MailerPressPro\Core;

use Exception;
use MailerPressPro\Core\Attributes\Action;
use MailerPressPro\Core\Attributes\Filter;
use DI\Container;
use DI\ContainerBuilder;
use DI\DependencyException;
use DI\NotFoundException;
use ReflectionAttribute;
use ReflectionClass;
use ReflectionException;


class Kernel
{

    public static array $config;
    protected static ?Container $container = null;
    private static Routing\Router $router;

    /**
     * @throws Exception
     */
    public static function getContainer(): ?Container
    {
        if (null === self::$container) {
            self::$container = self::buildContainer();
        }

        return self::$container;
    }

    /**
     * @throws Exception
     */
    private static function buildContainer(): Container
    {

        $containerBuilder = new ContainerBuilder;
        if (file_exists(self::$config['root'] . '/src/container-config.php')) {
            $containerBuilder->addDefinitions(self::$config['root'] . '/src/container-config.php');
            self::fillContainer($containerBuilder);
        }

        return $containerBuilder->build();
    }


    private static function fillContainer(ContainerBuilder $containerBuilder): void
    {
        $containerBuilder->addDefinitions([
            'actions' => self::buildClasses(
                self::$config['root'] . '/src/Actions',
                'actions',
                'MailerPressPro\\Actions\\'
            ),
            'endpoints' => self::buildClasses(self::$config['root'] . '/src/Api', 'actions', 'MailerPressPro\\Api\\'),
        ]);
    }

    private static function buildClasses($path, $type, $namespace): array
    {
        $services = [];

        if (!is_dir($path)) {
            return [];
        }

        $files = array_diff(scandir($path), ['..', '.', 'Services', 'partials', 'templates', 'Interfaces']);

        foreach ($files as $filename) {
            $fullPath = $path . DIRECTORY_SEPARATOR . $filename;

            if (is_dir($fullPath)) {
                $services = array_merge(
                    $services,
                    self::buildClasses($fullPath, $type, $namespace . $filename . '\\')
                );
                continue;
            }

            $pathinfo = pathinfo($filename);

            if (
                !isset($pathinfo['extension']) ||
                $pathinfo['extension'] !== 'php' ||
                str_contains($pathinfo['filename'], '.blade')
            ) {
                continue;
            }

            $class = $namespace . $pathinfo['filename'];
            $services[] = $class;
        }

        return $services;
    }

    private static function array_flatten($array): bool|array
    {
        if (!is_array($array)) {
            return false;
        }
        $result = array();
        foreach ($array as $key => $value) {
            if (is_array($value)) {
                $result = array_merge($result, self::array_flatten($value));
            } else {
                $result[$key] = $value;
            }
        }

        return $result;
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws ReflectionException
     * @throws Exception
     */
    public static function setup(): void
    {
        self::$container = self::buildContainer();
        self::includes();
        self::executeActions();
        self::executeEndpoints();
    }

    public static function includes(): void
    {
        if (file_exists(self::$config['root'] . '/src/functions.php')) {
            require_once self::$config['root'] . '/src/functions.php';
        }
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     * @throws ReflectionException
     */
    private static function executeActions(): void
    {
        foreach (self::$container->get('actions') as $action) {
            $actionClass = self::$container->get($action);
            $class = new ReflectionClass($actionClass);
            foreach ($class->getMethods() as $method) {
                self::launch(
                    [
                        $method->getAttributes(Action::class, ReflectionAttribute::IS_INSTANCEOF),
                        $method->getAttributes(Filter::class, ReflectionAttribute::IS_INSTANCEOF)
                    ],
                    $actionClass,
                    $method
                );
            }
        }
    }

    private static function launch(array $array, $actionClass, $method): void
    {
        foreach ($array as $item) {
            if (empty($item)) {
                continue;
            }

            foreach ($item as $methodAttr) {
                $action = $methodAttr->newInstance();
                $action->execute([$actionClass, $method->getName()]);
            }
        }
    }

    /**
     * @throws Exception
     */
    public static function execute(array $config): void
    {
        self::$config = $config;
        add_action('plugins_loaded', [__CLASS__, 'setup'], 1);
    }

    /**
     * @throws ReflectionException
     * @throws DependencyException
     * @throws NotFoundException
     */
    private static function executeEndpoints()
    {
        foreach (self::$container->get('endpoints') as $action) {
            $actionClass = self::$container->get($action);
            $class = new ReflectionClass($actionClass);
            foreach ($class->getMethods() as $method) {
                self::launch(
                    [
                        $method->getAttributes(Attributes\Endpoint::class, ReflectionAttribute::IS_INSTANCEOF),
                    ],
                    $actionClass,
                    $method
                );
            }
        }
    }
}
