<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use MailerPressPro\Core\EmailManager\BounceDetectionInterface;
use MailerPressPro\Core\EmailManager\BounceDetectionTrait;
use WP_Error;

class SendGridService extends AbstractEmailService implements BounceDetectionInterface
{
    use BounceDetectionTrait;
    private Contacts $contactModel;
    private Lists $listModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        $apiKey = $emailData['apiKey'] ?? '';

        if (empty($apiKey)) {
            return false;
        }

        $url = 'https://api.sendgrid.com/v3/mail/send';

        // Build the request body according to SendGrid API
        $body = [
            'personalizations' => [
                [
                    'to' => array_map(fn($to) => ['email' => $to], (array)($emailData['to'] ?? [])),
                    'subject' => $emailData['subject'] ?? '',
                ],
            ],
            'from' => [
                'email' => $emailData['sender_to'] ?? get_bloginfo('admin_email'),
                'name' => $emailData['sender_name'] ?? get_bloginfo('name'),
            ],
            'content' => [
                [
                    'type' => $emailData['content_type'] ?? 'text/html',
                    'value' => $emailData['body'] ?? '',
                ],
            ],
        ];

        // Handle CC/BCC if present
        if (!empty($emailData['cc'])) {
            $body['personalizations'][0]['cc'] = array_map(fn($cc) => ['email' => $cc], (array)$emailData['cc']);
        }
        if (!empty($emailData['bcc'])) {
            $body['personalizations'][0]['bcc'] = array_map(fn($bcc) => ['email' => $bcc], (array)$emailData['bcc']);
        }

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            $wpError = new WP_Error(
                sprintf(
                    __('SendGrid sendEmail error: %s', 'mailerpress-pro'),
                    $response->get_error_message()
                )
            );
            $this->logEmail($emailData, 'sendgrid', $wpError);
            return $wpError;
        }

        $status_code = wp_remote_retrieve_response_code($response);

        // SendGrid returns 202 Accepted on success
        if ($status_code !== 202) {
            $body = wp_remote_retrieve_body($response);
            $errorMessage = $this->getErrorMessageFromResponse($status_code, $body);

            $wpError = new WP_Error('sendgrid_send_error', esc_html($errorMessage));
            $this->logEmail($emailData, 'sendgrid', $wpError, $errorMessage);
            return $wpError;
        }

        // Log success
        $this->logEmail($emailData, 'sendgrid', true);

        return true;
    }

    /**
     * Extracts and formats an explicit error message from the SendGrid API response
     * 
     * @param int $code HTTP response code
     * @param string $resBody Response body
     * @return string Formatted and explicit error message
     */
    private function getErrorMessageFromResponse(int $code, string $resBody): string
    {
        // Explicit error messages for common HTTP codes
        $httpErrorMessages = [
            400 => __('SendGrid request error (400): The email parameters are invalid. Please check the email address format, subject, and content.', 'mailerpress-pro'),
            401 => __('SendGrid authentication error (401): The API key is invalid or missing. Please check your API key in the SendGrid settings.', 'mailerpress-pro'),
            403 => __('SendGrid access denied (403): Your API key does not have the necessary permissions or your account has been suspended.', 'mailerpress-pro'),
            404 => __('SendGrid resource not found (404): The domain or requested resource is not found. Please check your domain configuration.', 'mailerpress-pro'),
            413 => __('SendGrid file too large (413): The email or attachments exceed the maximum allowed size.', 'mailerpress-pro'),
            429 => __('SendGrid rate limit exceeded (429): Too many requests have been sent. Please wait before trying again.', 'mailerpress-pro'),
            500 => __('SendGrid server error (500): An internal error occurred on SendGrid\'s side. Please try again later.', 'mailerpress-pro'),
            502 => __('SendGrid gateway error (502): The SendGrid service is temporarily unavailable. Please try again later.', 'mailerpress-pro'),
            503 => __('SendGrid service unavailable (503): The SendGrid service is under maintenance. Please try again later.', 'mailerpress-pro'),
        ];

        // Base message with HTTP code
        $baseMessage = $httpErrorMessages[$code] ?? sprintf(
            __('SendGrid API error (HTTP %d)', 'mailerpress-pro'),
            $code
        );

        // Try to extract additional details from the response body
        if (!empty($resBody)) {
            $errorData = json_decode($resBody, true);

            if (json_last_error() === JSON_ERROR_NONE && is_array($errorData)) {
                $details = [];

                // SendGrid returns errors in an 'errors' array
                if (isset($errorData['errors']) && is_array($errorData['errors']) && !empty($errorData['errors'])) {
                    foreach ($errorData['errors'] as $err) {
                        if (isset($err['message'])) {
                            $details[] = $err['message'];
                        } elseif (isset($err['field'])) {
                            $details[] = sprintf(__('Error on field %s', 'mailerpress-pro'), $err['field']);
                        }
                    }
                }

                if (isset($errorData['message'])) {
                    $details[] = $errorData['message'];
                }

                // Build the final message with details
                if (!empty($details)) {
                    $detailsText = implode('; ', array_unique($details));
                    return sprintf(
                        __('%s Details: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $detailsText
                    );
                }
            } else {
                // If the body is not valid JSON, add it as is
                $resBodyTrimmed = trim($resBody);
                if (!empty($resBodyTrimmed) && strlen($resBodyTrimmed) < 500) {
                    return sprintf(
                        __('%s API response: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $resBodyTrimmed
                    );
                }
            }
        }

        return $baseMessage;
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'sendgrid',
            'name' => 'SendGrid',
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="44" height="44" viewBox="4 3.043 92 92" fill="none"><g filter="url(#filter0_d_2944_85289)"><rect width="92" height="92" x="4" y="3.043" fill="#294661" rx="46"></rect><path fill="#A3E1F2" d="M25 40.04v17.003h16.998V74.04h17.003v-34H25Z"></path><path fill="#1A82E2" d="M41.998 57.044H25v16.998h16.998V57.044Z"></path><path fill="#00B3E3" d="M58.998 40.04V23.042H41.996V57.043h34V40.04H58.998Z"></path><path fill="#1A82E2" d="M76 23.043H59.002v16.998H76V23.043ZM42.131 40.173h17.131v17.131H42.131z"></path></g><defs><filter id="filter0_d_2944_85289" width="100" height="100" x="0" y=".043" color-interpolation-filters="sRGB" filterUnits="userSpaceOnUse"><feFlood flood-opacity="0" result="BackgroundImageFix"></feFlood><feColorMatrix in="SourceAlpha" result="hardAlpha" values="0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 0 127 0"></feColorMatrix><feOffset dy="1"></feOffset><feGaussianBlur stdDeviation="2"></feGaussianBlur><feColorMatrix values="0 0 0 0 0.0687866 0 0 0 0 0.097585 0 0 0 0 0.37981 0 0 0 0.0779552 0"></feColorMatrix><feBlend in2="BackgroundImageFix" result="effect1_dropShadow_2944_85289"></feBlend><feBlend in="SourceGraphic" in2="effect1_dropShadow_2944_85289" result="shape"></feBlend></filter></defs></svg>',
            'link' => 'https://www.brevo.com/fr/pricing/',
            'createAccountLink' => 'https://onboarding.brevo.com/account/register',
            'linkApiKey' => 'https://app.brevo.com/settings/keys/api',
            'description' => __(
                'Send at scale with Twilio SendGrid, which boasts an industry-leading 99% deliverability rate. SendGrid offers both a free 100 emails per day plan and, if you need to exceed that limit, a selection of preset pricing plans, starting at $19.95 per month for up to 50,000 emails. For more information on getting started with SendGrid, check out our documentation.',
                'mailerpress-pro'
            ),
            'recommended' => true,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ],
            'bounce_doc_url' => 'https://mailerpress.com/docs/bounce-tracking/sendgrid',
        ];
    }

    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_token', $this->config()['key']);
        $pageToken = get_option($optionName, null);
        $limit = 100;

        $url = add_query_arg([
            'page_size' => $limit,
            'page_token' => $pageToken ?: null,
        ], 'https://api.sendgrid.com/v3/marketing/contacts');

        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['result'])) {
            foreach ($data['result'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'] ?? null,
                    'email' => $contact['email'],
                    'remote_lists' => $contact['list_ids'] ?? [],
                    'attributes' => [
                        'first_name' => $contact['first_name'] ?? null,
                        'last_name' => $contact['last_name'] ?? null,
                    ],
                ]);
            }

            // Store the next page token if it exists
            if (!empty($data['next_page_token'])) {
                update_option($optionName, $data['next_page_token']);
                // Re-enqueue next batch
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            } else {
                // No more pages, clear stored token
                delete_option($optionName);
            }
        }
    }


    public function syncLists(string $apiKey): void
    {
        $limit = 20;
        $offset = 0;

        do {
            $url = add_query_arg([
                'page_size' => $limit,
                'page_token' => $offset > 0 ? $offset : null, // SendGrid uses tokens for pagination
            ], 'https://api.sendgrid.com/v3/marketing/lists');

            $response = wp_remote_get($url, [
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Accept' => 'application/json',
                ],
            ]);

            if (is_wp_error($response)) {
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            $lists = $data['result'] ?? [];

            if (!empty($lists)) {
                foreach ($lists as $list) {
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }
            }

            $offset = $data['next_page_token'] ?? null;
        } while ($offset);
    }

    public function syncTags(string $apiKey): void {}

    public function pushContacts(string $apiKey, array $data): mixed
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];

        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                     FROM $list_sync_table
                     WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)
                    )
                );
            }

            $url = 'https://api.sendgrid.com/v3/marketing/contacts';

            $body = json_encode([
                'list_ids' => $remoteLists,
                'contacts' => [
                    [
                        'email' => $contact->email,
                        'firstName' => $contact->first_name,
                        'lastName' => $contact->last_name,
                    ]
                ]
            ]);

            $response = wp_remote_post($url, [
                'method' => 'PUT',
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['errors'])) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'SENDGRID_ERROR',
                    'error_message' => json_encode($data['errors']),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data)) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $contact->email, // SendGrid does not return ID, use email
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }

            return $data;
        }

        return null;
    }


    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            // Create list in SendGrid
            $response = wp_remote_post('https://api.sendgrid.com/v3/marketing/lists', [
                'headers' => [
                    'Authorization' => 'Bearer ' . $apiKey,
                    'Content-Type' => 'application/json',
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error(
                    'sendgrid_request_failed',
                    'Request to SendGrid failed.',
                    $response->get_error_message()
                );
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('sendgrid_api_error', 'SendGrid API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }


    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        // TODO: Implement deleteLists() method.
    }

    public function deleteAllLists(string $apiKey): void
    {
        // TODO: Implement deleteAllLists() method.
    }

    public function deleteContact(string $apiKey, array $data)
    {
        // TODO: Implement deleteContact() method.
    }

    // ============================================
    // Bounce Detection Implementation
    // ============================================

    /**
     * SendGrid supporte les webhooks pour les événements de bounce
     * 
     * Documentation: https://docs.sendgrid.com/for-developers/tracking-events/event
     */
    public function supportsWebhookBounceDetection(): bool
    {
        return true;
    }

    /**
     * SendGrid ne supporte plus l'API pour récupérer les bounces
     * Seuls les webhooks sont utilisés pour la détection de bounces
     */
    public function supportsApiBounceDetection(): bool
    {
        return false;
    }

    /**
     * Configure un webhook pour recevoir les notifications de bounce
     * 
     * Note: SendGrid ne permet pas de configurer les webhooks via API.
     * L'utilisateur doit configurer l'URL manuellement dans son compte SendGrid.
     * Cette méthode retourne simplement l'URL à configurer.
     * 
     * @param string $apiKey La clé API de SendGrid (non utilisée pour la configuration)
     * @param string $webhookUrl L'URL du webhook à configurer
     * @return array|false Retourne la configuration du webhook ou false en cas d'échec
     */
    public function configureBounceWebhook(string $apiKey, string $webhookUrl): array|false
    {
        // SendGrid ne permet pas de configurer les webhooks via API
        // L'utilisateur doit le faire manuellement dans son compte
        // On sauvegarde juste l'URL pour référence
        update_option('mailerpress_sendgrid_bounce_webhook_url', $webhookUrl);

        return [
            'url' => $webhookUrl,
            'description' => 'MailerPress Bounce Detection - Configure this URL manually in your SendGrid account',
        ];
    }

    /**
     * Supprime le webhook de bounce configuré
     * 
     * Note: SendGrid ne permet pas de supprimer les webhooks via API.
     * L'utilisateur doit le faire manuellement dans son compte SendGrid.
     * 
     * @param string $apiKey La clé API de SendGrid (non utilisée)
     * @param string|null $webhookId L'ID du webhook (non utilisé)
     * @return bool True si la suppression a réussi (on supprime juste la référence locale)
     */
    public function removeBounceWebhook(string $apiKey, ?string $webhookId = null): bool
    {
        delete_option('mailerpress_sendgrid_bounce_webhook_url');
        return true;
    }

    /**
     * Récupère les bounces depuis l'API SendGrid
     * 
     * Note: Cette méthode n'est plus utilisée, SendGrid ne supporte que les webhooks
     */
    public function fetchBouncesFromApi(string $apiKey, array $options = []): array
    {
        // SendGrid ne supporte plus la récupération de bounces via API
        return [];
    }

    /**
     * Traite un webhook de bounce reçu depuis SendGrid
     * 
     * Format du webhook SendGrid:
     * SendGrid envoie un tableau d'événements:
     * [
     *   {
     *     "email": "example@example.com",
     *     "timestamp": 1513299569,
     *     "smtp-id": "<14c5d75ce93.dfd.64b469@ismtpd-555>",
     *     "event": "bounce",
     *     "category": ["cat facts"],
     *     "sg_event_id": "sg_event_id",
     *     "sg_message_id": "sg_message_id",
     *     "reason": "550 5.1.1 The user you are trying to contact is not receiving mail.",
     *     "status": "5.1.1",
     *     "type": "bounce"
     *   }
     * ]
     */
    public function processBounceWebhook(array $payload): bool
    {
        // SendGrid envoie un tableau d'événements
        // Si le payload est un tableau, traiter chaque événement
        if (isset($payload[0])) {
            $processed = false;
            $processedCount = 0;
            foreach ($payload as $index => $event) {
                if ($this->processBounceEvent($event)) {
                    $processed = true;
                    $processedCount++;
                }
            }
            return $processed;
        }

        // Si c'est un seul événement, le traiter directement
        return $this->processBounceEvent($payload);
    }

    /**
     * Traite un événement de bounce individuel
     * 
     * @param array $event L'événement SendGrid
     * @return bool True si le traitement a réussi
     */
    private function processBounceEvent(array $event): bool
    {
        if (!isset($event['event']) || !isset($event['email'])) {
            return false;
        }

        $eventType = $event['event'];
        $email = $event['email'];
        $reason = $event['reason'] ?? $event['status'] ?? $eventType;

        // Traiter uniquement les événements de bounce
        $bounceEvents = ['bounce', 'blocked', 'dropped'];

        if (!in_array($eventType, $bounceEvents, true)) {
            return false;
        }

        $metadata = [
            'event' => $eventType,
            'type' => $event['type'] ?? null,
            'status' => $event['status'] ?? null,
            'sg_event_id' => $event['sg_event_id'] ?? null,
            'sg_message_id' => $event['sg_message_id'] ?? null,
            'smtp_id' => $event['smtp-id'] ?? null,
            'timestamp' => $event['timestamp'] ?? null,
            'category' => $event['category'] ?? null,
        ];

        return $this->markContactAsBounced($email, $reason, $metadata);
    }

    /**
     * Obtient l'URL du webhook pour SendGrid
     */
    public function getBounceWebhookUrl(): ?string
    {
        $webhookUrl = get_option('mailerpress_sendgrid_bounce_webhook_url');

        if (empty($webhookUrl)) {
            // Construire l'URL du webhook MailerPress
            $webhookUrl = rest_url('mailerpress/v1/esp/bounce/sendgrid');
        }

        return $webhookUrl;
    }

    /**
     * Valide la signature d'un webhook SendGrid
     * 
     * SendGrid utilise une signature ECDSA avec une clé publique
     * Documentation: https://docs.sendgrid.com/for-developers/tracking-events/getting-started-event-webhook-security-features
     * 
     * Note: La vérification complète nécessite la clé publique SendGrid.
     * Pour l'instant, on retourne true si la signature est présente.
     */
    public function verifyWebhookSignature(string $payload, string|array $signature, ?string $secret = null): bool
    {
        if (!is_string($signature) || empty($signature)) {
            return false;
        }

        $servicesConfig = get_option('mailerpress_email_services', []);
        $serviceConfig = $servicesConfig['services']['sendgrid']['conf'] ?? [];
        $verificationKey = $serviceConfig['webhook_verification_key'] ?? '';

        if (empty($verificationKey)) {
            return true;
        }

        try {
            $decodedSignature = base64_decode($signature);

            if ($decodedSignature === false) {
                return false;
            }

            // Construire la clé publique au format PEM si ce n'est pas déjà le cas
            if (strpos($verificationKey, 'BEGIN PUBLIC KEY') === false) {
                $publicKeyPEM = "-----BEGIN PUBLIC KEY-----\n" .
                    chunk_split($verificationKey, 64, "\n") .
                    "-----END PUBLIC KEY-----";
            } else {
                $publicKeyPEM = $verificationKey;
            }

            // Vérifier la signature avec OpenSSL
            $publicKey = openssl_pkey_get_public($publicKeyPEM);

            if ($publicKey === false) {
                return false;
            }

            $verified = openssl_verify($payload, $decodedSignature, $publicKey, OPENSSL_ALGO_SHA256);

            openssl_free_key($publicKey);

            if ($verified === 1) {
                return true;
            } elseif ($verified === 0) {
                return false;
            } else {
                return false;
            }
        } catch (\Exception $e) {
            return false;
        }
    }
}
