<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use MailerPressPro\Core\EmailManager\BounceDetectionInterface;
use MailerPressPro\Core\EmailManager\BounceDetectionTrait;
use WP_Error;

class Mailjet extends AbstractEmailService implements BounceDetectionInterface
{
    use BounceDetectionTrait;
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        // Load Mailjet config from options
        $services   = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];

        $apiKey    = $clientConf['api_key'] ?? '';
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiKey) || empty($apiSecret)) {
            return new WP_Error('mailjet_config_missing', __('Mailjet API key/secret is missing.', 'mailerpress-pro'));
        }

        if (empty($emailData['to'])) {
            return new WP_Error('mailjet_no_recipient', __('No recipient specified.', 'mailerpress-pro'));
        }

        $url = 'https://api.mailjet.com/v3.1/send';

        // Use sender_to and sender_name from emailData, fallback to Mailjet config defaults, then WordPress defaults
        $senderEmail = $emailData['sender_to'] 
            ?? $clientConf['default_email'] 
            ?? get_bloginfo('admin_email');
        $senderName = $emailData['sender_name'] 
            ?? $clientConf['default_name'] 
            ?? get_bloginfo('name');

        $message = [
            'From' => [
                'Email' => $senderEmail,
                'Name'  => $senderName,
            ],
            'To'       => array_map(fn($to) => ['Email' => $to], (array)$emailData['to']),
            'Subject'  => $emailData['subject'] ?? '',
            'HTMLPart' => $emailData['body'] ?? '',
        ];

        if (!empty($emailData['cc'])) {
            $message['Cc'] = array_map(fn($cc) => ['Email' => $cc], (array)$emailData['cc']);
        }

        if (!empty($emailData['bcc'])) {
            $message['Bcc'] = array_map(fn($bcc) => ['Email' => $bcc], (array)$emailData['bcc']);
        }

        $body = ['Messages' => [$message]];

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type'  => 'application/json',
            ],
            'body'    => wp_json_encode($body),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            $wpError = new WP_Error('mailjet_request_failed', $response->get_error_message());
            $this->logEmail($emailData, 'mailjet', $wpError);
            return $wpError;
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $resBody    = wp_remote_retrieve_body($response);

        // Vérifier d'abord le code HTTP
        if ($statusCode !== 200) {
            $errorMessage = $this->getErrorMessageFromResponse($statusCode, $resBody);
            $wpError = new WP_Error('mailjet_send_error', esc_html($errorMessage));
            $this->logEmail($emailData, 'mailjet', $wpError, $errorMessage);
            return $wpError;
        }

        // Log success
        $this->logEmail($emailData, 'mailjet', true);

        return true;
    }

    /**
     * Extracts and formats an explicit error message from the Mailjet API response
     * 
     * @param int $code HTTP response code
     * @param string $resBody Response body
     * @return string Formatted and explicit error message
     */
    private function getErrorMessageFromResponse(int $code, string $resBody): string
    {
        // Explicit error messages for common HTTP codes
        $httpErrorMessages = [
            400 => __('Mailjet request error (400): The email parameters are invalid. Please check the email address format, subject, and content.', 'mailerpress-pro'),
            401 => __('Mailjet authentication error (401): The API key or API secret is invalid or missing. Please check your credentials in the Mailjet settings.', 'mailerpress-pro'),
            403 => __('Mailjet access denied (403): Your API key does not have the necessary permissions or your account has been suspended.', 'mailerpress-pro'),
            404 => __('Mailjet resource not found (404): The domain or requested resource is not found. Please check your domain configuration.', 'mailerpress-pro'),
            413 => __('Mailjet file too large (413): The email or attachments exceed the maximum allowed size.', 'mailerpress-pro'),
            429 => __('Mailjet rate limit exceeded (429): Too many requests have been sent. Please wait before trying again.', 'mailerpress-pro'),
            500 => __('Mailjet server error (500): An internal error occurred on Mailjet\'s side. Please try again later.', 'mailerpress-pro'),
        ];

        // Base message with HTTP code
        $baseMessage = $httpErrorMessages[$code] ?? sprintf(
            __('Mailjet API error (HTTP %d)', 'mailerpress-pro'),
            $code
        );

        // Try to extract additional details from the response body
        if (!empty($resBody)) {
            $errorData = json_decode($resBody, true);
            
            if (json_last_error() === JSON_ERROR_NONE && is_array($errorData)) {
                $details = [];
                
                // Mailjet returns errors in Messages[0].Errors
                if (isset($errorData['Messages'][0]['Errors']) && is_array($errorData['Messages'][0]['Errors'])) {
                    foreach ($errorData['Messages'][0]['Errors'] as $err) {
                        if (isset($err['ErrorMessage'])) {
                            $details[] = $err['ErrorMessage'];
                        } elseif (isset($err['ErrorCode'])) {
                            $details[] = sprintf(__('Error code: %s', 'mailerpress-pro'), $err['ErrorCode']);
                        }
                    }
                }
                
                if (isset($errorData['ErrorMessage'])) {
                    $details[] = $errorData['ErrorMessage'];
                }
                
                if (isset($errorData['error'])) {
                    $errorText = is_string($errorData['error']) 
                        ? $errorData['error'] 
                        : json_encode($errorData['error']);
                    $details[] = $errorText;
                }
                
                // Build the final message with details
                if (!empty($details)) {
                    $detailsText = implode('; ', array_unique($details));
                    return sprintf(
                        __('%s Details: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $detailsText
                    );
                }
            } else {
                // If the body is not valid JSON, add it as is
                $resBodyTrimmed = trim($resBody);
                if (!empty($resBodyTrimmed) && strlen($resBodyTrimmed) < 500) {
                    return sprintf(
                        __('%s API response: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $resBodyTrimmed
                    );
                }
            }
        }

        return $baseMessage;
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'mailjet',
            'name' => __('Mailjet', 'mailerpress-pro'),
            'icon' => '<svg width="32" height="32" fill="rgb(111, 103, 217)" role="img" viewBox="0 0 104 104" color="mj50" class="StyledIcon__StyledSVG-eKBrmB kCMxxC StyledIcon-epMSEY cdLgWb MailjetLogoMono-crPHje jrTbNw c-n StyledIcon-epMSEY cdLgWb MailjetLogoMono-crPHje jrTbNw"><g><path d="m93.5,10.5c-2.45-2.46-6.12-3.21-9.35-1.92L12.17,37.37c-3.21,1.29-5.3,4.23-5.44,7.69-.14,3.46,1.69,6.56,4.78,8.11l21.31,10.65,7.75-7.75-20.96-10.48,64.65-25.86-25.86,64.65-10.48-20.96-7.75,7.75,10.65,21.31c1.49,2.98,4.43,4.79,7.73,4.79.12,0,.25,0,.38,0,3.46-.14,6.4-2.23,7.69-5.44l28.79-71.99c1.29-3.22.54-6.89-1.92-9.35Z"></path></g></svg>',
            'link' => 'https://mailjet.com/',
            'createAccountLink' => 'https://app.mailjet.com/signup',
            'linkApiKey' => 'https://app.mailjet.com/account/apikeys',
            'description' => __('Mailjet is an all-in-one email delivery service designed for both marketing and transactional emails. It offers powerful APIs, real-time analytics, and advanced personalization features. With a free plan that allows sending up to 6,000 emails per month (200 emails per day), Mailjet makes it easy to scale as your needs grow. For more information on how to get started with Mailjet, read our documentation.', 'mailerpress-pro'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ],
            'bounce_doc_url' => 'https://mailerpress.com/docs/bounce-tracking/mailjet',
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }
        } while ($offset < $total);

        delete_option($optionName);
    }

    public function syncTags(string $apiKey): void {}

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];


        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }
    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
        //        $option_key = 'mailerpress_brevo_folder_id';
        //        $folder_id = get_option($option_key);
        //
        //        if ($folder_id) {
        //            return $folder_id;
        //        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
                //                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

        //        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        } else {
            $body = wp_remote_retrieve_body($response);
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }

    // ============================================
    // Bounce Detection Implementation
    // ============================================

    /**
     * Mailjet supporte les webhooks pour les événements d'email
     */
    public function supportsWebhookBounceDetection(): bool
    {
        return true;
    }

    /**
     * Mailjet ne supporte plus l'API pour récupérer les événements
     * Seuls les webhooks sont utilisés pour la détection de bounces
     */
    public function supportsApiBounceDetection(): bool
    {
        return false;
    }

    /**
     * Configure un webhook Mailjet pour recevoir les notifications de bounce
     * 
     * Documentation Mailjet: https://documentation.mailjet.com/hc/en-us/articles/360043578154-Event-Tracking-API
     */
    public function configureBounceWebhook(string $apiKey, string $webhookUrl): array|false
    {
        // Mailjet nécessite api_key et api_secret
        $services = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiSecret)) {
            return false;
        }

        // Vérifier si un webhook existe déjà
        $existingWebhooks = $this->getExistingWebhooks($apiKey, $apiSecret);

        // Chercher un webhook existant avec la même URL
        foreach ($existingWebhooks as $webhook) {
            if (isset($webhook['Url']) && $webhook['Url'] === $webhookUrl) {
                update_option('mailerpress_mailjet_bounce_webhook_id', $webhook['ID']);
                return [
                    'id' => $webhook['ID'],
                    'url' => $webhook['Url'],
                    'description' => $webhook['Description'] ?? 'MailerPress Bounce Detection',
                ];
            }
        }

        // Créer un nouveau webhook
        $webhookData = [
            'Url' => $webhookUrl,
            'Description' => 'MailerPress Bounce Detection',
            'EventType' => [
                'bounce',
                'blocked',
                'spam',
                'unsub',
            ],
        ];

        $response = wp_remote_post('https://api.mailjet.com/v3.1/eventcallbackurl', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'body' => wp_json_encode($webhookData),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200 && $code !== 201) {
            return false;
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);

        if ($result && isset($result['Data']) && !empty($result['Data'])) {
            $webhook = $result['Data'][0];
            update_option('mailerpress_mailjet_bounce_webhook_id', $webhook['ID']);
            return [
                'id' => $webhook['ID'],
                'url' => $webhook['Url'],
                'description' => $webhook['Description'] ?? 'MailerPress Bounce Detection',
            ];
        }

        return false;
    }

    /**
     * Supprime le webhook de bounce configuré
     */
    public function removeBounceWebhook(string $apiKey, ?string $webhookId = null): bool
    {
        $services = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiSecret)) {
            return false;
        }

        $idToDelete = $webhookId ?? get_option('mailerpress_mailjet_bounce_webhook_id');

        if (empty($idToDelete)) {
            return false;
        }

        $response = wp_remote_request("https://api.mailjet.com/v3.1/eventcallbackurl/{$idToDelete}", [
            'method' => 'DELETE',
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return false;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204 || $code === 200) {
            delete_option('mailerpress_mailjet_bounce_webhook_id');
            return true;
        }

        return false;
    }

    /**
     * Récupère les bounces depuis l'API Mailjet
     * 
     * Documentation: https://documentation.mailjet.com/hc/en-us/articles/360043578154-Event-Tracking-API
     */
    public function fetchBouncesFromApi(string $apiKey, array $options = []): array
    {
        $services = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiSecret)) {
            return [];
        }

        $startDate = $options['start_date'] ?? date('Y-m-d', strtotime('-7 days'));
        $endDate = $options['end_date'] ?? date('Y-m-d');
        $limit = $options['limit'] ?? 50;
        $offset = $options['offset'] ?? 0;

        $params = [
            'FromTS' => strtotime($startDate),
            'ToTS' => strtotime($endDate),
            'Limit' => $limit,
            'Offset' => $offset,
            'EventType' => 'bounce,blocked,spam',
        ];

        $url = 'https://api.mailjet.com/v3.1/eventstatistics?' . http_build_query($params);

        $response = wp_remote_get($url, [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return [];
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);

        if (!$result || !isset($result['Data'])) {
            return [];
        }

        $bounces = [];
        foreach ($result['Data'] as $event) {
            if (in_array($event['EventType'], ['bounce', 'blocked', 'spam'], true)) {
                $bounces[] = [
                    'email' => $event['Email'] ?? null,
                    'event' => $event['EventType'],
                    'reason' => $event['Comment'] ?? $event['EventType'],
                    'date' => isset($event['Time']) ? date('Y-m-d H:i:s', $event['Time']) : null,
                    'message_id' => $event['MessageID'] ?? null,
                ];
            }
        }

        return $bounces;
    }

    /**
     * Traite un webhook de bounce reçu depuis Mailjet
     * 
     * Format officiel du webhook Mailjet:
     * {
     *   "event": "bounce",
     *   "time": 1430812195,
     *   "MessageID": 13792286917004336,
     *   "Message_GUID": "1ab23cd4-e567-8901-2345-6789f0gh1i2j",
     *   "email": "bounce@mailjet.com",
     *   "mj_campaign_id": 0,
     *   "mj_contact_id": 0,
     *   "customcampaign": "",
     *   "CustomID": "helloworld",
     *   "Payload": "",
     *   "blocked": false,
     *   "hard_bounce": true,
     *   "error_related_to": "recipient",
     *   "error": "user unknown",
     *   "comment": "Host or domain name not found. Name service error for name=lbjsnrftlsiuvbsren.com type=A: Host not found"
     * }
     * 
     * Documentation: https://documentation.mailjet.com/hc/en-us/articles/360043578154-Event-Tracking-API
     */
    public function processBounceWebhook(array $payload): bool
    {
        // Mailjet peut envoyer un tableau d'événements (même pour un seul événement)
        // Vérifier d'abord si c'est un tableau d'événements
        // On vérifie si le premier élément existe et est un tableau, OU si le payload n'a pas de clé 'event'
        $isArrayOfEvents = false;
        if (!empty($payload) && is_array($payload)) {
            // Vérifier si c'est un tableau indexé numériquement (commence à 0)
            $keys = array_keys($payload);
            if (!empty($keys) && is_numeric($keys[0]) && $keys[0] === 0) {
                // C'est probablement un tableau d'événements
                $isArrayOfEvents = true;
            } elseif (!isset($payload['event']) && isset($payload[0])) {
                // Pas de clé 'event' mais il y a un index 0, c'est probablement un tableau
                $isArrayOfEvents = true;
            }
        }

        if ($isArrayOfEvents) {
            // Si c'est un tableau, traiter chaque événement
            $processed = false;
            foreach ($payload as $index => $eventPayload) {
                if (is_array($eventPayload)) {
                    if ($this->processBounceWebhook($eventPayload)) {
                        $processed = true;
                    }
                }
            }
            return $processed;
        }

        // Vérifier que c'est un événement unique valide
        if (!isset($payload['event'])) {
            return false;
        }

        $event = $payload['event'];

        // Format officiel Mailjet : l'email est dans le champ "email"
        $email = $payload['email'] ?? $payload['Email'] ?? '';

        // Si l'email est vide, essayer de le récupérer via mj_contact_id si disponible
        if (empty($email) && !empty($payload['mj_contact_id']) && $payload['mj_contact_id'] > 0) {
            $email = $this->getEmailByMailjetContactId((int)$payload['mj_contact_id']);
        }

        // Si l'email est toujours vide, essayer de le récupérer via MessageID
        if (empty($email) && !empty($payload['MessageID']) && $payload['MessageID'] > 0) {
            $email = $this->getEmailByMailjetMessageId((int)$payload['MessageID']);
        }

        if (empty($email)) {
            // Vérifier si c'est un webhook de test (email vide, MessageID à 0, mj_contact_id à 0)
            $isTestWebhook = (empty($payload['MessageID']) || $payload['MessageID'] == 0) &&
                (empty($payload['mj_contact_id']) || $payload['mj_contact_id'] == 0) &&
                empty($payload['Message_GUID']);

            if ($isTestWebhook) {
                return true;
            }

            return false;
        }

        // Format officiel Mailjet : utiliser "comment" si disponible, sinon "error", sinon "error_related_to"
        $reason = $payload['comment'] ?? $payload['error'] ?? $payload['error_related_to'] ?? $event;

        // Traiter uniquement les événements de bounce, blocked et spam
        // "blocked" est traité comme un bounce car l'email est bloqué et ne peut pas être délivré
        // "spam" est traité comme un bounce car l'email a été marqué comme spam
        $bounceEvents = ['bounce', 'blocked', 'spam'];

        if (!in_array($event, $bounceEvents, true)) {
            return false;
        }

        $metadata = [
            'event' => $event,
            'message_id' => $payload['MessageID'] ?? null,
            'message_guid' => $payload['Message_GUID'] ?? null,
            'time' => $payload['time'] ?? null,
            'mj_campaign_id' => $payload['mj_campaign_id'] ?? null,
            'mj_contact_id' => $payload['mj_contact_id'] ?? null,
            'customcampaign' => $payload['customcampaign'] ?? null,
            'custom_id' => $payload['CustomID'] ?? null,
            'payload' => $payload['Payload'] ?? null,
            'hard_bounce' => $payload['hard_bounce'] ?? false,
            'blocked' => $payload['blocked'] ?? false,
            'error_related_to' => $payload['error_related_to'] ?? null,
            'error' => $payload['error'] ?? null,
            'comment' => $payload['comment'] ?? null,
        ];

        return $this->markContactAsBounced($email, $reason, $metadata);
    }

    /**
     * Obtient l'URL du webhook pour Mailjet
     */
    public function getBounceWebhookUrl(): ?string
    {
        $webhookId = get_option('mailerpress_mailjet_bounce_webhook_id');

        if (empty($webhookId)) {
            return null;
        }

        // Construire l'URL du webhook MailerPress
        $webhookUrl = rest_url('mailerpress/v1/esp/bounce/mailjet');

        return $webhookUrl;
    }

    /**
     * Valide la signature d'un webhook Mailjet
     * 
     * Mailjet utilise une signature dans le paramètre "token" de la requête
     * ou dans l'en-tête X-Mailjet-Signature
     */
    public function verifyWebhookSignature(string $payload, string|array $signature, ?string $secret = null): bool
    {
        // Mailjet envoie une signature de type string (pas array)
        if (!is_string($signature) || empty($signature)) {
            return false;
        }

        // Récupérer la clé secrète du webhook depuis les options
        $webhookSecret = $secret ?? get_option('mailerpress_mailjet_webhook_secret');

        if (empty($webhookSecret)) {
            // Si aucun secret n'est configuré, on accepte la requête
            return true;
        }

        // Mailjet utilise généralement le token dans les paramètres de la requête
        // Pour les webhooks, on peut vérifier le token
        // Calculer la signature attendue (HMAC-SHA256)
        $expectedSignature = hash_hmac('sha256', $payload, $webhookSecret);

        // Comparer les signatures de manière sécurisée
        return hash_equals($expectedSignature, $signature);
    }

    /**
     * Récupère l'email d'un contact Mailjet via son ID
     * 
     * @param int $contactId L'ID du contact Mailjet
     * @return string L'email du contact ou une chaîne vide si non trouvé
     */
    private function getEmailByMailjetContactId(int $contactId): string
    {
        $services = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];
        $apiKey = $clientConf['api_key'] ?? '';
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiKey) || empty($apiSecret)) {
            return '';
        }

        $response = wp_remote_get("https://api.mailjet.com/v3/REST/contact/{$contactId}", [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return '';
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return '';
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        return $data['Data'][0]['Email'] ?? '';
    }

    /**
     * Récupère l'email d'un message Mailjet via son MessageID
     * 
     * @param int $messageId L'ID du message Mailjet
     * @return string L'email du destinataire ou une chaîne vide si non trouvé
     */
    private function getEmailByMailjetMessageId(int $messageId): string
    {
        $services = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailjet']['conf'] ?? [];
        $apiKey = $clientConf['api_key'] ?? '';
        $apiSecret = $clientConf['api_secret'] ?? '';

        if (empty($apiKey) || empty($apiSecret)) {
            return '';
        }

        // Mailjet API pour récupérer les informations d'un message
        $response = wp_remote_get("https://api.mailjet.com/v3.1/message/{$messageId}", [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            return '';
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return '';
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);
        // Le format de réponse peut varier, essayer différents champs
        return $data['Data'][0]['Recipient'] ?? $data['Recipient'] ?? $data['To'] ?? '';
    }

    /**
     * Récupère les webhooks existants depuis Mailjet
     * 
     * @param string $apiKey
     * @param string $apiSecret
     * @return array
     */
    private function getExistingWebhooks(string $apiKey, string $apiSecret): array
    {
        $response = wp_remote_get('https://api.mailjet.com/v3.1/eventcallbackurl', [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $apiSecret),
                'Content-Type' => 'application/json',
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return [];
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code !== 200) {
            return [];
        }

        $result = json_decode(wp_remote_retrieve_body($response), true);

        if ($result && isset($result['Data'])) {
            return $result['Data'];
        }

        return [];
    }
}
