<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager;

\defined('ABSPATH') || exit;

use MailerPress\Core\Enums\Tables;

/**
 * Trait pour gérer la détection de bounces de manière centralisée
 * 
 * Ce trait fournit des méthodes communes pour marquer les contacts comme bounced
 * et gérer les logs de bounces.
 */
trait BounceDetectionTrait
{
    /**
     * Marque un contact comme bounced
     * 
     * @param string $email L'email du contact
     * @param string|null $reason La raison du bounce (optionnel)
     * @param array $metadata Métadonnées supplémentaires (optionnel)
     * @return bool True si le contact a été marqué comme bounced
     */
    protected function markContactAsBounced(string $email, ?string $reason = null, array $metadata = []): bool
    {
        global $wpdb;
        $table = Tables::get(Tables::MAILERPRESS_CONTACT);
        $espName = $this->getEspName();

        $contact = $wpdb->get_row(
            $wpdb->prepare("SELECT contact_id FROM $table WHERE email = %s", $email)
        );

        if (!$contact) {
            return false;
        }

        $updateData = [
            'subscription_status' => 'bounced',
        ];

        // Mettre à jour le contact
        $result = $wpdb->update(
            $table,
            $updateData,
            ['contact_id' => $contact->contact_id],
            ['%s'],
            ['%d']
        );

        if ($result === false) {
            return false;
        }

        // Logger le bounce si nécessaire
        if (!empty($reason)) {
            $this->logBounce((int)$contact->contact_id, $email, $reason, $metadata);
        }

        return true;
    }

    /**
     * Log un bounce pour audit
     * 
     * @param int $contactId L'ID du contact
     * @param string $email L'email du contact
     * @param string $reason La raison du bounce
     * @param array $metadata Métadonnées supplémentaires
     * @return void
     */
    protected function logBounce(int $contactId, string $email, string $reason, array $metadata = []): void
    {
        $logData = [
            'contact_id' => $contactId,
            'email' => $email,
            'reason' => $reason,
            'metadata' => $metadata,
            'timestamp' => current_time('mysql'),
            'esp' => $this->getEspName(),
        ];

        // Stocker dans une option pour les logs récents
        $logs = get_option('mailerpress_bounce_logs', []);
        $logs[] = $logData;

        // Garder seulement les 100 derniers logs
        if (count($logs) > 100) {
            $logs = array_slice($logs, -100);
        }

        update_option('mailerpress_bounce_logs', $logs);
    }

    /**
     * Obtient le nom de l'ESP (à surcharger dans les classes enfants)
     * 
     * @return string
     */
    protected function getEspName(): string
    {
        if (method_exists($this, 'config')) {
            $config = $this->config();
            // Utiliser la clé (key) plutôt que le nom (name) car la clé est toujours en anglais
            // et ne dépend pas de la traduction, ce qui évite les confusions
            $espKey = $config['key'] ?? 'unknown';

            // Mapper les clés vers des noms lisibles pour les logs
            $espNameMap = [
                'mailjet' => 'Mailjet',
                'brevo' => 'Brevo',
                'sendgrid' => 'SendGrid',
                'postmark' => 'PostMark',
                'mailgun' => 'Mailgun',
                'amazonses' => 'Amazon SES',
                'gmail' => 'Gmail',
                'resend' => 'Resend',
                'mailersend' => 'MailerSend',
            ];

            return $espNameMap[strtolower($espKey)] ?? ucfirst($espKey);
        }
        return 'Unknown';
    }
}
