<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager;

\defined('ABSPATH') || exit;

/**
 * Interface pour la détection de bounces via API ou webhooks
 * 
 * Les services ESP qui implémentent cette interface peuvent gérer automatiquement
 * la détection des emails rebondis via leur API ou système de webhooks.
 */
interface BounceDetectionInterface
{
    /**
     * Vérifie si l'ESP supporte la détection de bounces via webhooks
     * 
     * @return bool True si les webhooks sont supportés
     */
    public function supportsWebhookBounceDetection(): bool;

    /**
     * Vérifie si l'ESP supporte la détection de bounces via API
     * 
     * @return bool True si l'API est supportée
     */
    public function supportsApiBounceDetection(): bool;

    /**
     * Configure un webhook pour recevoir les notifications de bounce
     * 
     * @param string $apiKey La clé API de l'ESP
     * @param string $webhookUrl L'URL du webhook à configurer
     * @return array|false Retourne la configuration du webhook ou false en cas d'échec
     */
    public function configureBounceWebhook(string $apiKey, string $webhookUrl): array|false;

    /**
     * Supprime le webhook de bounce configuré
     * 
     * @param string $apiKey La clé API de l'ESP
     * @param string|null $webhookId L'ID du webhook à supprimer (optionnel)
     * @return bool True si la suppression a réussi
     */
    public function removeBounceWebhook(string $apiKey, ?string $webhookId = null): bool;

    /**
     * Récupère les bounces depuis l'API de l'ESP
     * 
     * @param string $apiKey La clé API de l'ESP
     * @param array $options Options pour la récupération (date, limit, etc.)
     * @return array Liste des bounces avec les informations nécessaires
     */
    public function fetchBouncesFromApi(string $apiKey, array $options = []): array;

    /**
     * Traite un webhook de bounce reçu depuis l'ESP
     * 
     * @param array $payload Les données du webhook
     * @return bool True si le traitement a réussi
     */
    public function processBounceWebhook(array $payload): bool;

    /**
     * Obtient l'URL du webhook pour cet ESP
     * 
     * @return string|null L'URL du webhook ou null si non configuré
     */
    public function getBounceWebhookUrl(): ?string;

    /**
     * Valide la signature d'un webhook de bounce
     * 
     * @param string $payload Le corps de la requête
     * @param string|array $signature La signature fournie (peut être un array pour certains ESP comme Mailgun)
     * @param string|null $secret Le secret pour valider la signature (optionnel)
     * @return bool True si la signature est valide
     */
    public function verifyWebhookSignature(string $payload, string|array $signature, ?string $secret = null): bool;
}
