<?php

namespace MailerPressPro\Services\AI\Images;

use MailerPressPro\Core\ImageGeneratorInterface;
use WP_Error;

class OpenAIImageGenerator implements ImageGeneratorInterface
{
    private string $apiKey;
    private string $modelId;

    /**
     * @param string $apiKey
     * @param string $modelId
     */
    public function __construct(string $apiKey, string $modelId = 'dall-e-2')
    {
        $this->apiKey = $apiKey;
        $this->modelId = $modelId;
    }

    public function generate(string $prompt, int $number = 1, string $size = '256x256')
    {
        // Check if we should simulate images
        if (defined('MP_FAKE_AI_IMAGES') && MP_FAKE_AI_IMAGES) {
            return $this->generateFakeImages($number, $size);
        }

        $endpoint = "https://api.openai.com/v1/images/generations";

        $data = [
            'prompt' => $prompt,
            'n' => $number,
            'size' => $size,
            'model' => $this->modelId, // <-- specify the model here
        ];

        $response = wp_remote_post($endpoint, [
            'headers' => [
                'Content-Type' => 'application/json',
                'Authorization' => "Bearer {$this->apiKey}",
            ],
            'body' => wp_json_encode($data),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('http_error', $response->get_error_message());
        }

        $body = wp_remote_retrieve_body($response);
        $responseData = json_decode($body, true);

        if ($responseData === null) {
            return new WP_Error('invalid_json', 'Failed to parse API response.', ['body' => $body]);
        }

        if (isset($responseData['error'])) {
            return new WP_Error('api_error', $responseData['error']['message'] ?? 'Unknown error');
        }

        $images = [];
        foreach ($responseData['data'] ?? [] as $item) {
            if (!empty($item['b64_json'])) {
                $images[] = 'data:image/png;base64,' . $item['b64_json'];
            } elseif (!empty($item['url'])) {
                $images[] = $item['url'];
            }
        }

        return $images ?: new WP_Error('no_images', 'No images generated');
    }

    private function generateFakeImages(int $number, string $size): array
    {
        $images = [];
        $width = explode('x', $size)[0] ?? 256;
        $height = explode('x', $size)[1] ?? 256;

        for ($i = 0; $i < $number; $i++) {
            // Create a blank image
            $img = imagecreatetruecolor($width, $height);

            // Fill with random color
            $bgColor = imagecolorallocate($img, rand(100, 255), rand(100, 255), rand(100, 255));
            imagefill($img, 0, 0, $bgColor);

            // Add some text to indicate it's fake
            $textColor = imagecolorallocate($img, 0, 0, 0);
            $text = "Fake AI Img " . ($i + 1);
            imagestring($img, 5, 10, $height / 2 - 7, $text, $textColor);

            // Capture the image as base64
            ob_start();
            imagepng($img);
            $pngData = ob_get_clean();
            imagedestroy($img);

            $images[] = 'data:image/png;base64,' . base64_encode($pngData);
        }

        return $images;
    }

}
