<?php

namespace MailerPressPro\Core\Segmentation\Conditions;

use MailerPressPro\Core\Segmentation\Operators;
use MailerPressPro\Core\Segmentation\SegmentCondition;
use MailerPressPro\Core\Tables;

class PurchaseAmountCondition implements SegmentCondition
{
    private float $amount;
    private string $operator;

    private const ALLOWED_OPERATORS = [
        Operators::GREATER_THAN,
        Operators::LESS_THAN,
        Operators::GREATER_THAN_OR_EQUALS,
        Operators::LESS_THAN_OR_EQUALS,
    ];

    public function __construct(float $amount, string $operator = Operators::GREATER_THAN)
    {
        if (!in_array($operator, self::ALLOWED_OPERATORS, true)) {
            throw new \InvalidArgumentException("Operator {$operator} not supported for PurchaseAmountCondition");
        }

        $this->amount = $amount;
        $this->operator = $operator;
    }

    public function toSqlCondition(): string
    {
        global $wpdb;
        $table = $wpdb->prefix . 'mailerpress_contact_stats';

        $subQuery = "SELECT contact_id, SUM(COALESCE(revenue, 0)) AS total_revenue 
                     FROM {$table} 
                     GROUP BY contact_id";

        switch ($this->operator) {
            case Operators::GREATER_THAN:
                return $wpdb->prepare(
                    "c.contact_id IN (SELECT contact_id FROM ({$subQuery}) AS t WHERE total_revenue > %f)",
                    $this->amount
                );

            case Operators::LESS_THAN:
                return $wpdb->prepare(
                    "c.contact_id IN (SELECT contact_id FROM ({$subQuery}) AS t WHERE total_revenue < %f)",
                    $this->amount
                );

            case Operators::GREATER_THAN_OR_EQUALS:
                return $wpdb->prepare(
                    "c.contact_id IN (SELECT contact_id FROM ({$subQuery}) AS t WHERE total_revenue >= %f)",
                    $this->amount
                );

            case Operators::LESS_THAN_OR_EQUALS:
                return $wpdb->prepare(
                    "c.contact_id IN (SELECT contact_id FROM ({$subQuery}) AS t WHERE total_revenue <= %f)",
                    $this->amount
                );

            default:
                throw new \RuntimeException("Unsupported operator: {$this->operator}");
        }
    }

    public static function getConditionMetadata(): array
    {
        return [
            'field' => 'total_purchases',
            'label' => __('Total Purchases', 'mailerpress-pro'),
            'operators' => [
                Operators::GREATER_THAN => __('is greater than', 'mailerpress-pro'),
                Operators::LESS_THAN => __('is less than', 'mailerpress-pro'),
                Operators::GREATER_THAN_OR_EQUALS => __('is greater than or equals', 'mailerpress-pro'),
                Operators::LESS_THAN_OR_EQUALS => __('is less than or equals', 'mailerpress-pro'),
            ],
            'input_type' => 'number',
            'choices' => [],
        ];
    }

    public static function canRun(): bool
    {
        return true;
    }
}
