<?php

declare(strict_types=1);

namespace MailerPressPro\Core\EmailManager\services;

\defined('ABSPATH') || exit;

use DI\DependencyException;
use DI\NotFoundException;
use MailerPress\Core\EmailManager\AbstractEmailService;
use MailerPress\Core\Enums\Tables;
use MailerPress\Models\Contacts;
use MailerPress\Models\Lists;
use WP_Error;

class MailerSend extends AbstractEmailService
{
    private Lists $listModel;
    private Contacts $contactModel;

    public function __construct(
        Lists $listModel,
        Contacts $contactModel,
    ) {
        $this->listModel = $listModel;
        $this->contactModel = $contactModel;
    }

    public function sendEmail(array $emailData): WP_Error|bool
    {
        // Load MailerSend API key from options or $emailData
        $services   = get_option('mailerpress_email_services', []);
        $clientConf = $services['services']['mailersend']['conf'] ?? [];
        $apiKey     = $emailData['apiKey'] ?? $clientConf['api_key'] ?? '';

        if (empty($apiKey)) {
            return new WP_Error('mailersend_missing_key', __('MailerSend API key is missing', 'mailerpress-pro'));
        }

        $url = 'https://api.mailersend.com/v1/email';

        // Build recipients
        $to = [];
        foreach ((array)($emailData['to'] ?? []) as $recipient) {
            $to[] = ['email' => $recipient];
        }

        $cc = [];
        foreach ((array)($emailData['cc'] ?? []) as $recipient) {
            $cc[] = ['email' => $recipient];
        }

        $bcc = [];
        foreach ((array)($emailData['bcc'] ?? []) as $recipient) {
            $bcc[] = ['email' => $recipient];
        }

        // Prepare request body
        $body = [
            'from' => [
                'email' => $emailData['sender_to'] ?? get_bloginfo('admin_email'),
                'name'  => $emailData['sender_name'] ?? get_bloginfo('name'),
            ],
            'to'      => $to,
            'cc'      => $cc ?: null,
            'bcc'     => $bcc ?: null,
            'subject' => $emailData['subject'] ?? '',
            'html'    => $emailData['body'] ?? '',
        ];

        // Clean null values
        $body = array_filter($body, fn($v) => !is_null($v));

        $response = wp_remote_post($url, [
            'headers' => [
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type'  => 'application/json',
            ],
            'body' => wp_json_encode($body),
            'timeout' => 30,
        ]);

        if (is_wp_error($response)) {
            $wpError = new WP_Error(
                'mailersend_request_failed',
                sprintf(__('MailerSend request failed: %s', 'mailerpress-pro'), $response->get_error_message())
            );
            $this->logEmail($emailData, 'mailersend', $wpError);
            return $wpError;
        }

        $statusCode = wp_remote_retrieve_response_code($response);
        $resBody    = wp_remote_retrieve_body($response);
        $decoded    = json_decode($resBody, true);

        if ($statusCode !== 202) {
            $errorMsg = $this->getErrorMessageFromResponse($statusCode, $resBody);

            $wpError = new WP_Error('mailersend_send_error', esc_html($errorMsg));
            $this->logEmail($emailData, 'mailersend', $wpError, $errorMsg);
            return $wpError;
        }

        // Log success
        $this->logEmail($emailData, 'mailersend', true);

        return true;
    }

    /**
     * Extracts and formats an explicit error message from the MailerSend API response
     * 
     * @param int $code HTTP response code
     * @param string $resBody Response body
     * @return string Formatted and explicit error message
     */
    private function getErrorMessageFromResponse(int $code, string $resBody): string
    {
        // Explicit error messages for common HTTP codes
        $httpErrorMessages = [
            400 => __('MailerSend request error (400): The email parameters are invalid. Please check the email address format, subject, and content.', 'mailerpress-pro'),
            401 => __('MailerSend authentication error (401): The API key is invalid or missing. Please check your API key in the MailerSend settings.', 'mailerpress-pro'),
            403 => __('MailerSend access denied (403): Your API key does not have the necessary permissions or your account has been suspended.', 'mailerpress-pro'),
            404 => __('MailerSend resource not found (404): The domain or requested resource is not found. Please check your domain configuration.', 'mailerpress-pro'),
            422 => __('MailerSend validation error (422): The email data is not valid. Please check the email addresses, content format, and attachments.', 'mailerpress-pro'),
            429 => __('MailerSend rate limit exceeded (429): Too many requests have been sent. Please wait before trying again.', 'mailerpress-pro'),
            500 => __('MailerSend server error (500): An internal error occurred on MailerSend\'s side. Please try again later.', 'mailerpress-pro'),
        ];

        // Base message with HTTP code
        $baseMessage = $httpErrorMessages[$code] ?? sprintf(
            __('MailerSend API error (HTTP %d)', 'mailerpress-pro'),
            $code
        );

        // Try to extract additional details from the response body
        if (!empty($resBody)) {
            $errorData = json_decode($resBody, true);
            
            if (json_last_error() === JSON_ERROR_NONE && is_array($errorData)) {
                $details = [];
                
                if (isset($errorData['message'])) {
                    $details[] = $errorData['message'];
                }
                
                // MailerSend can return multiple errors in an array
                if (isset($errorData['errors']) && is_array($errorData['errors'])) {
                    foreach ($errorData['errors'] as $err) {
                        if (isset($err['message'])) {
                            $details[] = $err['message'];
                        } elseif (is_string($err)) {
                            $details[] = $err;
                        }
                    }
                }
                
                if (isset($errorData['error'])) {
                    $errorText = is_string($errorData['error']) 
                        ? $errorData['error'] 
                        : json_encode($errorData['error']);
                    $details[] = $errorText;
                }
                
                // Build the final message with details
                if (!empty($details)) {
                    $detailsText = implode('; ', array_unique($details));
                    return sprintf(
                        __('%s Details: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $detailsText
                    );
                }
            } else {
                // If the body is not valid JSON, add it as is
                $resBodyTrimmed = trim($resBody);
                if (!empty($resBodyTrimmed) && strlen($resBodyTrimmed) < 500) {
                    return sprintf(
                        __('%s API response: %s', 'mailerpress-pro'),
                        $baseMessage,
                        $resBodyTrimmed
                    );
                }
            }
        }

        return $baseMessage;
    }

    public function testConnection(): bool
    {
        return true;
    }

    public function config(): array
    {
        return [
            'key' => 'mailersend',
            'name' => __('MailerSend', 'mailerpress-pro'),
            'icon' => '<svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 92 92" fill="none"><rect width="92" height="92" fill="#F1F1F1" rx="46"></rect><g clip-path="url(#mailer-send-logo-clip-path)"><path fill="#4E48E0" d="M68.34 37.927a.953.953 0 0 0-.79-.401H50.124l8.689-13.888a1.124 1.124 0 0 0-.035-1.237.964.964 0 0 0-.799-.401H42.976c-1.506 0-2.914.808-3.758 2.156L22.166 51.414c-.223.357-.232.845.037 1.236a.952.952 0 0 0 .792.401H40.41l-8.69 13.892c-.156.25-.204.566-.133.866.11.466.522.772.967.772h14.995c1.518 0 2.936-.815 3.786-2.173l8.353-13.357 8.689-13.888a1.13 1.13 0 0 0-.036-1.236Z"></path><path fill="#4E48E0" d="m59.69 53.048-8.355 13.36c-.85 1.358-2.268 2.172-3.786 2.172H32.554a.997.997 0 0 1-.967-.77c-.071-.301-.023-.617.133-.867l8.691-13.895h19.28Z"></path><path fill="#1C81C1" d="M50.114 37.526h17.435c.306 0 .61.136.792.401.269.391.259.88.036 1.236l-8.688 13.889H40.403l9.71-15.526Z"></path><path fill="#5CC4CD" d="M22.994 53.051a.952.952 0 0 1-.791-.4 1.13 1.13 0 0 1-.036-1.237l17.051-27.258C40.063 22.808 41.47 22 42.977 22H57.98c.308 0 .614.136.799.401.27.392.258.88.035 1.237L40.411 53.05H22.994Z"></path></g><defs><clipPath id="mailer-send-logo-clip-path"><path fill="#fff" d="M22 22h47v47H22z"></path></clipPath></defs></svg>',
            'link' => 'https://mailersend.com/',
            'createAccountLink' => 'https://www.mailersend.com/signup',
            'linkApiKey' => 'https://app.mailersend.com/tokens/create',
            'description' => __('MailerSend is a modern email delivery service built for transactional and marketing emails. It offers a robust API, high deliverability, real-time analytics, and advanced personalization options. With a developer-friendly interface and scalable plans, MailerSend allows you to send emails reliably and efficiently. For more information on getting started with MailerSend, check their official documentation.', 'mailerpress-pro'),
            'recommended' => false,
            'sending_frequency' => [
                "numberEmail" => 100,
                "frequency" => [
                    'value' => 2,
                    'unit' => 'minutes',
                ]
            ]
        ];
    }

    /**
     * @throws DependencyException
     * @throws NotFoundException
     */
    public function syncContacts(string $apiKey): void
    {
        $optionName = sprintf('%s_contacts_last_offset', $this->config()['key']);
        $limit = 100;
        $offset = 0;

        $url = sprintf(
            "https://api.brevo.com/v3/contacts?limit=%d&offset=%d",
            $limit,
            $offset
        );

        $response = wp_remote_get($url, [
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
            ]
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $status_code = wp_remote_retrieve_response_code($response);
        if ($status_code !== 200) {
            return;
        }

        $data = json_decode(wp_remote_retrieve_body($response), true);

        if (!empty($data['contacts'])) {
            foreach ($data['contacts'] as $contact) {
                $this->contactModel->createOrUpdate([
                    'remote_id' => $contact['id'],
                    'email' => $contact['email'],
                    'remote_lists' => $contact['listIds'],
                    'attributes' => [
                        'first_name' => $contact['attributes']['PRENOM'],
                        'last_name' => $contact['attributes']['NOM'],
                    ],
                ]);
            }

            // Update the offset
            $newOffset = $offset + count($data['contacts']);
            update_option($optionName, $newOffset);

            if (count($data['contacts']) === $limit) {
                as_enqueue_async_action('mailerpress_esp_sync_contacts');
            }
        }
    }

    public function syncLists(string $apiKey): void
    {
        $optionName = sprintf('%s_lists_last_offset', $this->config()['key']);
        $limit = 20;
        $offset = (int)get_option($optionName, 0);
        $total = null;

        do {
            $url = sprintf(
                "https://api.brevo.com/v3/contacts/lists?limit=%d&offset=%d",
                $limit,
                $offset
            );

            $response = wp_remote_get($url, [
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                ]
            ]);

            if (is_wp_error($response)) {
                break;
            }

            $status_code = wp_remote_retrieve_response_code($response);
            if ($status_code !== 200) {
                break;
            }

            $body = wp_remote_retrieve_body($response);
            $data = json_decode($body, true);

            if ($total === null && isset($data['count'])) {
                $total = (int)$data['count'];
            }

            $lists = $data['lists'] ?? [];
            if (!empty($lists)) {
                foreach ($lists as $list) {
                    // Do something with the list
                    $this->listModel->createOrUpdate([
                        'name' => $list['name'],
                        'esp_list_id' => $list['id'],
                    ]);
                }

                // Update the stored offset after each batch
                $offset += $limit;
                update_option($optionName, $offset);
            }
        } while ($offset < $total);

        delete_option($optionName);
    }

    public function syncTags(string $apiKey): void {}

    public function pushContacts(string $apiKey, array $data): mixed
    {

        global $wpdb;

        if (empty($data['contact'])) {
            return null;
        }

        $contact = $data['contact'];


        if ('subscribed' !== $contact->subscription_status) {
            return null;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $list_sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d LIMIT 1",
            $contact->contact_id,
            $active_provider_id
        ));

        if ($existing_sync) {
            /** TODO update the contact */
        } else {
            // Insert the contact
            $remoteLists = [];
            if (!empty($contact->local_list_ids)) {
                $contact->local_list_ids = array_map('intval', $contact->local_list_ids);
                $placeholders = implode(',', array_fill(0, count($contact->local_list_ids), '%d'));

                $remoteLists = $wpdb->get_col(
                    $wpdb->prepare(
                        "SELECT remote_list_id
                 FROM $list_sync_table
                 WHERE provider_account_id = %d AND list_id IN ($placeholders)",
                        array_merge([(int)$active_provider_id], $contact->local_list_ids)

                    )
                );
            }

            $url = 'https://api.brevo.com/v3/contacts';

            $body = json_encode([
                'email' => $contact->email,
                'attributes' => [
                    'FIRSTNAME' => $contact->first_name,
                    'LASTNAME' => $contact->last_name,
                ],
                'listIds' => array_map('intval', $remoteLists), // Convert list IDs to integers
                'updateEnabled' => true, // To update the contact if already exists
            ]);

            $response = wp_remote_post($url, [
                'method' => 'POST',
                'headers' => [
                    'api-key' => $apiKey,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                ],
                'body' => $body,
                'timeout' => 60, // Set a timeout for the request
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR', // You can set this based on your error
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            // Successfully added/updated the contact in Brevo
            $response_body = wp_remote_retrieve_body($response);
            $data = json_decode($response_body, true);

            if (isset($data['code']) && $data['code'] != 200) {
                // Handle API error response if any
                mailerpress_log_sync_error([
                    'provider_account_id' => $contact->provider_account_id,
                    'entity_type' => 'contact', // Specify the entity type as contact
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $data['code'], // You can set this based on your error
                    'error_message' => $data['message'],
                    'context' => [
                        'contact_email' => $contact->email,
                        'remote_lists' => $remoteLists,
                    ],
                ]);
                return null;
            }

            if (!empty($data['id'])) {
                $wpdb->insert(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => (int)$active_provider_id,
                        'remote_contact_id' => $data['id'],
                        'last_synced_at' => current_time('mysql'),
                    ],
                    [
                        '%d',
                        '%d',
                        '%s',
                        '%s'
                    ]
                );
            }


            return $data;
        }

        return null;
    }

    public function pushLists(string $apiKey, array $data): mixed
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS); // Adjust to your actual constant
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");

        $folderId = $this->getFolder($apiKey, 'MailerPress');

        $existing_sync = $wpdb->get_row($wpdb->prepare(
            "SELECT * FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d LIMIT 1",
            $data['localListId'],
            $active_provider_id
        ));

        if ($existing_sync) {
            return '';
        } else {
            $response = wp_remote_post('https://api.brevo.com/v3/contacts/lists', [
                'headers' => [
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json',
                    'api-key' => $apiKey,
                ],
                'body' => wp_json_encode([
                    'name' => $data['name'],
                    'folderId' => (int)$folderId,
                ]),
                'method' => 'POST',
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                return new WP_Error('brevo_request_failed', 'Request to Brevo failed.', $response->get_error_message());
            }

            $code = wp_remote_retrieve_response_code($response);
            $body = json_decode(wp_remote_retrieve_body($response), true);

            if ($code !== 201) {
                return new WP_Error('brevo_api_error', 'Brevo API returned an error.', $body);
            }

            $wpdb->insert(
                $sync_table,
                [
                    'list_id' => $data['localListId'],
                    'provider_account_id' => (int)$active_provider_id,
                    'remote_list_id' => $body['id'],
                    'last_synced_at' => current_time('mysql'),
                ],
                [
                    '%d',
                    '%d',
                    '%s',
                    '%s'
                ]
            );

            return $body;
        }
    }

    public function deleteLists(string $apiKey, int $listId): void
    {
        global $wpdb;

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_LISTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);
        $active_provider_id = $wpdb->get_var("SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1");


        $remoteListId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_list_id FROM {$sync_table} WHERE list_id = %d AND provider_account_id = %d",
                $listId,
                $active_provider_id
            )
        );

        if ($remoteListId) {
            wp_remote_request("https://api.brevo.com/v3/contacts/lists/{$remoteListId}", [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);
        }
    }

    public function pushTags(string $apiKey): void
    {
        // TODO: Implement pushTags() method.
    }

    private function getFolder(string $apiKey, string $folder_name)
    {
        //        $option_key = 'mailerpress_brevo_folder_id';
        //        $folder_id = get_option($option_key);
        //
        //        if ($folder_id) {
        //            return $folder_id;
        //        }

        // Try to find the folder via Brevo API
        $response = wp_remote_get('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Failed to fetch folders.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 200 || empty($body['folders'])) {
            return new WP_Error('brevo_api_error', 'Error fetching folders list.', $body);
        }


        // Check if folder already exists
        foreach ($body['folders'] as $folder) {
            if (strcasecmp($folder['name'], $folder_name) === 0) {
                //                update_option($option_key, $folder['id']);
                return $folder['id'];
            }
        }

        // Folder doesn't exist, so create it
        $response = wp_remote_post('https://api.brevo.com/v3/contacts/folders', [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
                'api-key' => $apiKey,
            ],
            'body' => wp_json_encode([
                'name' => $folder_name,
            ]),
            'timeout' => 60,
        ]);

        if (is_wp_error($response)) {
            return new WP_Error('brevo_request_failed', 'Folder creation failed.', $response->get_error_message());
        }

        $code = wp_remote_retrieve_response_code($response);
        $body = json_decode(wp_remote_retrieve_body($response), true);

        if ($code !== 201) {
            return new WP_Error('brevo_api_error', 'Brevo API error.', $body);
        }

        //        update_option($option_key, $body['id']);

        return $body['id'];
    }


    public function deleteAllLists(string $apiKey): void
    {
        $folder_id = get_option('mailerpress_brevo_folder_id');

        if (!$folder_id) {
            return;
        }

        $response = wp_remote_request("https://api.brevo.com/v3/contacts/folders/{$folder_id}", [
            'method' => 'DELETE',
            'headers' => [
                'api-key' => $apiKey,
                'Accept' => 'application/json',
                'Content-Type' => 'application/json',
            ],
            'timeout' => 15,
        ]);

        if (is_wp_error($response)) {
            return;
        }

        $code = wp_remote_retrieve_response_code($response);
        if ($code === 204) {
            // Folder deleted successfully
            delete_option('mailerpress_brevo_folder_id');
        }
    }

    public function deleteContact(string $apiKey, array $data): void
    {
        global $wpdb;

        if (empty($data['contact'])) {
            return;
        }

        $contact = $data['contact'];

        if (empty($contact->email)) {
            return;
        }

        $sync_table = Tables::get(Tables::MAILERPRESS_PROVIDER_CONTACTS);
        $providers_table = Tables::get(Tables::MAILERPRESS_PROVIDER_ACCOUNTS);

        $active_provider_id = $wpdb->get_var(
            "SELECT provider_account_id FROM {$providers_table} WHERE is_active = 1 LIMIT 1"
        );

        $remoteContactId = $wpdb->get_var(
            $wpdb->prepare(
                "SELECT remote_contact_id FROM {$sync_table} WHERE contact_id = %d AND provider_account_id = %d",
                $contact->contact_id,
                $active_provider_id
            )
        );

        if ($remoteContactId) {
            $response = wp_remote_request("https://api.brevo.com/v3/contacts/" . urlencode($contact->email), [
                'method' => 'DELETE',
                'headers' => [
                    'api-key' => $apiKey,
                    'Accept' => 'application/json',
                    'Content-Type' => 'application/json',
                ],
                'timeout' => 60,
            ]);

            if (is_wp_error($response)) {
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => 'API_ERROR',
                    'error_message' => $response->get_error_message(),
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
                return;
            }

            $response_code = wp_remote_retrieve_response_code($response);

            if ($response_code === 204) {
                // Delete local sync entry after successful API deletion
                $wpdb->delete(
                    $sync_table,
                    [
                        'contact_id' => $contact->contact_id,
                        'provider_account_id' => $active_provider_id,
                    ],
                    ['%d', '%d']
                );
            } else {
                $response_body = wp_remote_retrieve_body($response);
                $error = json_decode($response_body, true);
                mailerpress_log_sync_error([
                    'provider_account_id' => $active_provider_id,
                    'entity_type' => 'contact',
                    'entity_local_id' => $contact->contact_id,
                    'error_code' => $error['code'] ?? 'UNKNOWN',
                    'error_message' => $error['message'] ?? 'Unknown error',
                    'context' => [
                        'contact_email' => $contact->email,
                    ],
                ]);
            }
        }
    }
}
